classdef pointillisticDataClass < handle
% This class is used for storing the accpeted (the raw localization data or
% that passed the filtering) and the rejected localization data and for
% performing the filtering operation on it. It can also frift correct or
% calculate additional/supplementary localization data from the basic ones.

    
    properties
        
        dataName = ''
        
        dataType = ''
        
        %operations = struct([])
        
        analysisParameters = struct([])
        
        % localization data
        accepted=table()                 % the filtered localization data (or the original if no filtering has been performed)
        rejected=table()                 % localization data that did not pass the filtering
        
        filteringBooleanVect=[]     % resultant boolean vector with which the filtering of the full data was performed
        filteringBooleanVect_noFrameIndex = [];
        
        % should it be moved to the "pointillisticDataParams"
        driftTrajectory = struct([])
        
        derivativeDatas = {}
        
        reconstructedImage = [];
        pixelCenters = [];
        
    end
    
    % Constructor
    methods
        % construct an empty object when called
        function pointillisticDataObject = pointillisticDataClass()
            
        end
    end
    
    methods
        %% Functions for importing the pointillistic data
        
        function initialize(pointillisticDataObject, analysisParameters, accepted, rejected)
            dataType = analysisParameters.dataType;
            
            pointillisticDataObject.dataType = dataType;
            
            
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected = rejected;
            
            if isempty(pointillisticDataObject.rejected)
                pointillisticDataObject.rejected = pointillisticData_fieldManagement.createEmptyData(pointillisticDataObject.accepted);
            else
                
            end
        end
        
        function [originalDataName, originalDataPath, dataType] = dataImport(pointillisticDataObject, operations, dataFileFullName, convention)

            [analysisParameters, pointillisticData] = runOperation.creation(operations, 'localizationData', 'loading', {}, dataFileFullName, convention);

            
            dataType = analysisParameters.dataType;
            
            pointillisticDataObject.dataType = dataType;
            
            
            pointillisticDataObject.accepted = pointillisticData;
            pointillisticDataObject.rejected = pointillisticData_fieldManagement.createEmptyData(pointillisticDataObject.accepted);
            
            
            pointillisticDataObject.filteringBooleanVect = true(size(pointillisticDataObject.accepted.x_coord));
            pointillisticDataObject.filteringBooleanVect_noFrameIndex = pointillisticDataObject.filteringBooleanVect;
            
            pointillisticDataObject.analysisParameters = analysisParameters;
        
            originalDataName = analysisParameters.sharedParameters.originalDataName;
            originalDataPath = analysisParameters.sharedParameters.originalDataPath;
            
        end
        
        function estimatePrecision(pointillisticDataObject)
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            
            operations.(dataType).processing.('precisionCalculation') = rainSTORM_GUIsettings_precision.settings('Mortensen, Gaussian mask estimation');
            
            accepted = pointillisticDataObject.accepted;
            rejected = pointillisticDataObject.rejected;
            
            analysisParameters = pointillisticDataObject.analysisParameters;
            
            [analysisParameters, accepted] = runOperation.processing(operations, 'precisionCalculation', analysisParameters, accepted);
            [analysisParameters, rejected] = runOperation.processing(operations, 'precisionCalculation', analysisParameters, rejected);
            
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected = rejected;
            pointillisticDataObject.analysisParameters = analysisParameters;
            
        end
        
        function filter(pointillisticDataObject, filteringOperation)
            

            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('filtering') = filteringOperation;
            
            [analysisParameters, accepted, rejected, filteringBooleanVect, filteringBooleanVect_noFrameIndex] = runOperation.processing(operations, 'filtering', pointillisticDataObject.analysisParameters, pointillisticDataObject.accepted, pointillisticDataObject.rejected, pointillisticDataObject.filteringBooleanVect);
            
            
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected = rejected;
            pointillisticDataObject.filteringBooleanVect = filteringBooleanVect;
            pointillisticDataObject.filteringBooleanVect_noFrameIndex = filteringBooleanVect_noFrameIndex;
            
        end
        
        function driftTrajectory = correctDrift(pointillisticDataObject, driftCorrectionOperation)
            
            % remove the "frame range" filtering for the drift trajectory
            % calculation
            filterBoolVect_accepted_noFrame = pointillisticDataObject.filteringBooleanVect_noFrameIndex(pointillisticDataObject.filteringBooleanVect);
            filterBoolVect_rejected_noFrame = pointillisticDataObject.filteringBooleanVect_noFrameIndex(~pointillisticDataObject.filteringBooleanVect);
            [pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new] = pointillisticData_filtering.refilter(pointillisticDataObject.accepted, pointillisticDataObject.rejected, pointillisticDataObject.filteringBooleanVect, filterBoolVect_accepted_noFrame, filterBoolVect_rejected_noFrame);
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('driftCorrection') = driftCorrectionOperation;
            
            [analysisParameters, accepted, rejected, driftTrajectory] = runOperation.processing(operations, 'driftCorrection', pointillisticDataObject.analysisParameters, pointillisticDataObject.accepted, pointillisticDataObject.rejected);
            % apply the drift correction on the localization data:
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected = rejected;
            
            pointillisticDataObject.analysisParameters = analysisParameters;
            % add the calculated drift to the already existing/not existing one, considering all fields (e.g. x_coord, y_coord and z_coord, if they exist)
            driftFieldNames = fieldnames(driftTrajectory);
            if isempty(pointillisticDataObject.driftTrajectory)
                pointillisticDataObject.driftTrajectory = driftTrajectory;
            else
                for idxField = 1:numel(driftFieldNames)
                    field = driftFieldNames{idxField};
                    pointillisticDataObject.driftTrajectory.(field) =  pointillisticDataObject.driftTrajectory.(field) + driftTrajectory.(field);
                end
            end
            
        end
        
        function undoDriftCorrection(pointillisticDataObject)
            
            
            % undo the drift correction on the localization data:
            pointillisticDataObject.accepted = driftCorrection.add(pointillisticDataObject.accepted, pointillisticDataObject.driftTrajectory);
            pointillisticDataObject.rejected = driftCorrection.add(pointillisticDataObject.rejected, pointillisticDataObject.driftTrajectory);
            
            pointillisticDataObject.driftTrajectory = struct([]);
            
            
            for idxOperation = numel(pointillisticDataObject.analysisParameters.processingOperations):-1:1
                operation = pointillisticDataObject.analysisParameters.processingOperations{idxOperation};
                % delete all the drift correction parameters data
                if strcmp(operation.operationType, 'drift correction')
                   % TODO proper type
                   pointillisticDataObject.analysisParameters.processingOperations(idxOperation) = [];
                end
            end
            
        end
        
        function calculateAstigmatic3D(pointillisticDataObject, calibrationData, astigmatic3DOperation)
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('astigmatic3D') = astigmatic3DOperation;
            
            [analysisParameters, accepted, rejected] = runOperation.processing(operations, 'astigmatic3D', pointillisticDataObject.analysisParameters, pointillisticDataObject.accepted, pointillisticDataObject.rejected, calibrationData);
            
            pointillisticDataObject.analysisParameters = analysisParameters;
            
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected = rejected;
            
        end
        
        function undoAstigmatic3D(pointillisticDataObject)
            
            % remove the "z_coord" field
            pointillisticDataObject.accepted = pointillisticData_fieldManagement.delete(pointillisticDataObject.accepted, {'z_coord'});
            pointillisticDataObject.rejected = pointillisticData_fieldManagement.delete(pointillisticDataObject.rejected, {'z_coord'});
            
        end
        
        
        function [clusterProperties, appliedSettings] = clusterize(pointillisticDataObject, clusterizationOperation)
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('clusterization') = clusterizationOperation;
            
            [analysisParameters, accepted, clusterProperties, operationIndex] = runOperation.processing(operations, 'clusterization', pointillisticDataObject.analysisParameters, pointillisticDataObject.accepted);
            
            pointillisticDataObject.analysisParameters = analysisParameters;
            
            pointillisticDataObject.accepted = accepted;
            pointillisticDataObject.rejected.cluster_idx = zeros(size(pointillisticDataObject.rejected.x_coord));
            
            appliedSettings = analysisParameters.processingOperations{operationIndex}.settings;
            
        end
        
        function undoClusterization(pointillisticDataObject)
            
            % remove the "cluster_idx" field
            pointillisticDataObject.accepted = pointillisticData_fieldManagement.delete(pointillisticDataObject.accepted, {'cluster_idx'});
            pointillisticDataObject.rejected = pointillisticData_fieldManagement.delete(pointillisticDataObject.rejected, {'cluster_idx'});
            
        end
        
        
        function driftFigure = driftVisualization(pointillisticDataObject)
            
            if ~isempty(pointillisticDataObject.driftTrajectory)
                driftFigure = driftCorrection.visualize(pointillisticDataObject.driftTrajectory);
            else
                driftFigure  = matlab.ui.Figure.empty();
               warning('Perform drift correction before drift trajectory visualization.') 
            end
            
        end
        
        function [localizationFigure, localizationImageMatrix, colormapVector] = pixelize_accepted(pointillisticDataObject, pixelizationOperationSettings, visualizationOperationSettings)
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('pixelization') = pixelizationOperationSettings;
            operations.(dataType).processing.('visualization') = visualizationOperationSettings;
            
            analysisParameters = pointillisticDataObject.analysisParameters;
            
            [analysisParameters, localizationImageMatrix, pixelCenters] = runOperation.processing(operations, 'pixelization', analysisParameters, pointillisticDataObject.accepted);
            [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters, localizationImageMatrix, pixelCenters);
            
            pointillisticDataObject.analysisParameters = analysisParameters;
            
            pointillisticDataObject.reconstructedImage = localizationImageMatrix;
            pointillisticDataObject.pixelCenters = pixelCenters;
            
        end
        
        
        function [localizationFigure] = visualize_accepted(pointillisticDataObject, pixelizationOperationSettings, visualizationOperationSettings)

            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('pixelization') = pixelizationOperationSettings;
            operations.(dataType).processing.('visualization') = visualizationOperationSettings;
            
            if isempty(pointillisticDataObject.reconstructedImage) || isempty(pointillisticDataObject.pixelCenters)
                [~, ~, ~] = pointillisticDataObject.pixelize_accepted(pixelizationOperationSettings, visualizationOperationSettings);
            end
                
            localizationImageMatrix = pointillisticDataObject.reconstructedImage;
            pixelCenters = pointillisticDataObject.pixelCenters;
            [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', pointillisticDataObject.analysisParameters, localizationImageMatrix, pixelCenters);

            pointillisticDataObject.analysisParameters = analysisParameters;

        end
        
        
        function [localizationFigure, localizationImageMatrix, colormapVector] = pixelize_rejected(pointillisticDataObject, pixelizationOperationSettings, visualizationOperationSettings)
            
            dataType = pointillisticDataObject.analysisParameters.dataType;
            operations.(dataType).processing.('pixelization') = pixelizationOperationSettings;
            operations.(dataType).processing.('visualization') = visualizationOperationSettings;
            
            analysisParameters = pointillisticDataObject.analysisParameters;
            
            [analysisParameters, localizationImageMatrix, pixelCenters] = runOperation.processing(operations, 'pixelization', analysisParameters, pointillisticDataObject.rejected);
            [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters, localizationImageMatrix, pixelCenters);
            
            pointillisticDataObject.analysisParameters = analysisParameters;
            
        end
        
        
        function histogramsFigure = createHistograms(pointillisticDataObject)
            
            
            localizationData_accepted = pointillisticDataObject.accepted;
            localizationData_rejected = pointillisticDataObject.rejected;
            title = {'channel name', 'recon name'};
            numberOfFrames = pointillisticDataObject.analysisParameters.sharedParameters.stackSize(3);
            exposureTime_ms = pointillisticDataObject.analysisParameters.sharedParameters.cameraSignalConversion.exposureTime_ms;
            pixelSize_nm = pointillisticDataObject.analysisParameters.sharedParameters.cameraSignalConversion.pixelSize_nm;
            countsPerElectron = pointillisticDataObject.analysisParameters.sharedParameters.cameraSignalConversion.countsPerElectron;
            
            histogramsFigure = rainSTORM_histograms(localizationData_accepted, localizationData_rejected, title, numberOfFrames, exposureTime_ms, countsPerElectron, pixelSize_nm);
            
        end
        
        function dataExport(pointillisticDataObject, exportPath, exportName, convention)
            
            pointillisticData = pointillisticDataObject.accepted;
            analysisParameters = pointillisticDataObject.analysisParameters;
            
            if isempty(exportName)
                dataOutputName = pointillisticDataObject.dataName;
            else
                dataOutputName = exportName;
            end
            
            dataExportFunction = conventionFunctions.dataExport(convention);
            dataExportFunction(exportPath, dataOutputName, pointillisticData, analysisParameters)
            
        end
        
        function figuresExport(pointillisticDataObject, pixelizationOperationSettings, visualizationOperationSettings, exportPath, exportName)
            
            if isempty(exportName)
                outputName = pointillisticDataObject.dataName;
            else
                outputName = exportName;
            end
            
            pixelSize_nm = pointillisticDataObject.analysisParameters.sharedParameters.cameraSignalConversion.pixelSize_nm;
            [localizationFigure_accepted, localizationImageMatrix_accepted, colormapVector_accepted] = pointillisticDataObject.pixelize_accepted(pixelizationOperationSettings, visualizationOperationSettings);
            outputAcceptedImageName = [exportPath, filesep, outputName, '.tif'];
            imageMatrixExport(localizationImageMatrix_accepted, colormapVector_accepted, pixelSize_nm, outputAcceptedImageName);
            
            [localizationFigure_rejected, localizationImageMatrix_rejected, colormapVector_rejected] = pointillisticDataObject.pixelize_rejected(pixelizationOperationSettings, visualizationOperationSettings);
            outputRejectedImageName = [exportPath, filesep, outputName, '_rejected', '.tif'];
            imageMatrixExport(localizationImageMatrix_rejected, colormapVector_rejected, pixelSize_nm, outputRejectedImageName);
            
            driftFigure = pointillisticDataObject.driftVisualization();
            if ~isempty(driftFigure)
                outputDriftFigureName = [exportPath, filesep, outputName, '_drift', '.png'];
                saveas(driftFigure, outputDriftFigureName, 'png');
            end
            
            histogramsFigure = pointillisticDataObject.createHistograms();
            outputHistogramsFigureName = [exportPath, filesep, outputName, '_histogram', '.png'];
            saveas(histogramsFigure, outputHistogramsFigureName, 'png');
            
        end
        
    end
    
    methods (Static)
        
        
    end
    
end
