% script to run the data processing from the image stack reading to
% localization concatenation and clustering, usin the "operations"

% change the variables in the first section according to the actual needs

% path of this script relative to the main folder:
codePath='..';

[currentDir,~,~] = fileparts(mfilename('fullpath'));

% change the working directory
cd(fullfile(currentDir, codePath))


% adding the directories containing the required functions
addSubfoldersToPath()

% path of the operation settings file relative to the script location:
operationSettingsPath='settings';
% name of the operation settings file to be used:
settingsFile='operationSettings_rainSTORM_GUI_defaults.json';

% path of the camera settings file relative to the script location:
cameraCalibrationPath='settings';
% name of the camera settings file to be used:
cameraCalibrationFile='cameraCalibration_AdOptIm.json';

% save the result in a folder named the same as the data file:
resultDirectory='${dataName}';
% save the result in the folder with specific name:
%resultDirectory='';

% choose whether the directory where the results are being saved to
% should be an absolute path or a relative path compared to the
% original data:
outputPathType='relative';
%outputPathType='absolute';

%% loading the settings and select the datas

% chose a single image stack:
%[dataFileFullNames, imageType]=imageStack_selection.singleStack();
% chose multiple image stacks:
[dataFileFullNames, imageType]=imageStack_selection.multipleStacks();

% alternatively, manually set the file name and its type:
%dataFullFileNames={''};
%imageType='TIFF image stack (*.tif)';

% % alternatively, load a csv data instead of an image stack (then disable the image stack processing steps...) 
% % set the data convention:
% dataConvention='rainSTORM';
% %dataConvention='rainSTORM legacy';
% %dataConvention='rainSTORM scripting version';
% %dataConvention='ThunderSTORM';
% % select a single localization data:
% dataFileFullNames=pointillisticData_selection.single();
% % select a multiple localization datas:
% dataFileFullNames=pointillisticData_selection.multiple();

% in case of a "simple" settings file:
%[operations] = settingsManagement.loadOperations(operationSettingsPath, settingsFile);
% in case of GUI settings file:
[GUI_settings] = loadOperations(operationSettingsPath, settingsFile);
operations = rainSTORM_GUI_settings_extract.full(GUI_settings);

% update the settings with the camera calibration file
[operations] = loadCameraCalibration(cameraCalibrationPath, cameraCalibrationFile, operations);

% optical magnification beside the nominal one:
additionalMagnification = 1.0;
%additionalMagnification = 1.5;
operations.localizationData.creation.loading.settings.additionalMagnification = additionalMagnification;

% The image metadata can be read from ND2 or OME-TIFF file, otherwise set
% these manually (e.g. when reading TIFF file geneerated with TestSTORM):
% cameraSignalConversionParameters = struct(...
%     'pixelSize', struct('value', 160,'unit', 'nm'),...
%     'exposureTime', struct('value', 30,'unit', 'ms'),...
%     'frameInterval', struct('value', 30.9,'unit', 'ms'),...
%     'countsPerElectron', struct('value', 100/67.8),...
%     'countsPerPhoton', struct('value', 0.92 * 100/67.8),...
%     'baseline', struct('value', 100,'unit', 'camera count')...
%     );
%operations.localizationData.creation.loading.settings.missingCameraSignalConversionParameterOverwrite = cameraSignalConversionParameters;
%operations.imageStack.creation.reading.settings.settings.missingCameraSignalConversionParameterOverwrite = cameraSignalConversionParameters;

for idxData=1:numel(dataFileFullNames)
    %% Preparation for the actual data
    
    % the file that will be processed:
    dataFileFullName=dataFileFullNames{idxData};
    
    % get the file name and path:
    [dataPath, dataName, ~]=fileparts(dataFileFullName);
    
    % the path of the results:
    resultDirectory_act=strrep(resultDirectory, '${dataName}', dataName);
    
    % create the directory where the results will be saved:
    if ~isfolder(fileparts(outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '', '')))
        mkdir(fileparts(outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '', '')));
    end
    
    % initialize the analysis metadata that stores the analysis steps
    analysisParameters=struct();
    
    %% Read the stack
    
    [analysisParameters, frameStack, frameStack_originalMetadata, frameStack_omeMetadata] = runOperation.creation(operations, 'imageStack', 'reading', {}, dataFileFullName, imageType);

    cameraSignalConversion = analysisParameters.sharedParameters.cameraSignalConversion;
    
    %% Background subtraction
    
    % get the background removed and the background stack
    [analysisParameters, frameStack, frameStack_background] = runOperation.processing(operations, 'backgroundRemoval', analysisParameters, frameStack);
    
    % save the filtered stack for inspection
    outputFullFileName_filtered=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_blinkings', '.tif');
    imageStack_saving.multipage_tif(frameStack, outputFullFileName_filtered, frameStack_originalMetadata, frameStack_omeMetadata);
    % save the background stack for inspection
    outputFullFileName_background=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_background', '.tif');
    imageStack_saving.multipage_tif(frameStack_background, outputFullFileName_background, frameStack_originalMetadata, frameStack_omeMetadata);
    
    %% Localization
    
    % localize the image stack
    [analysisParameters, localizationData, ~] = runOperation.creation(operations, 'localizationData', 'localization', {analysisParameters}, frameStack);
    
    %% Localization data loading
%     % alternatively, skip the previous step and load an already existing
%     % localization data
%     [analysisParameters, localizationData, ~] = runOperation.creation(operations, 'imageStack', 'reading', {}, dataFileFullName, dataConvention);
    
    %% Localization precision estimation
    
    % calculate the localization precision, because it needs for other analysis steps...
    [analysisParameters, localizationData, ~] = runOperation.processing(operations, 'precisionEstimation', analysisParameters, localizationData);
    
     %% Filter the localizations
    
    localizationData_rejected = [];
    filteringBooleanVect = [];
    % set and overwrite the thresholds
    thresholds = [...
        struct('field', 'std', 'bounds', [0, 25], 'unit', 'nm'),...
        struct('field', 'res', 'bounds', [0, 0.5], 'unit', ''),...
        ];
    operations.localizationData.processing.filtering.settings.thresholds = thresholds;
    [analysisParameters, localizationData, localizationData_rejected, filteringBooleanVect, ~] = runOperation.processing(operations, 'filtering', analysisParameters, localizationData, localizationData_rejected, filteringBooleanVect);
    
    %% Drift correction
    
    [analysisParameters, localizationData, localizationData_rejected, driftTrajectory] = runOperation.processing(operations, 'driftCorrection', analysisParameters, localizationData, localizationData_rejected);
    
    % save the drift trajectory figure
    driftFigure = driftCorrection.visualize(driftTrajectory);
    outputDriftFigureName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_drift', '.png');
    saveas(driftFigure, outputDriftFigureName, 'png')
    
    %% save the localization data:
    outputLocalizationDataFullFileName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_localizationData', '.csv');
    [directoryPath, dataFileName, extension] = fileparts(outputLocalizationDataFullFileName);
    pointillisticData_save.tabularData(directoryPath, [dataFileName, extension], localizationData);
    infoFileName = [dataFileName, '_info', '.json'];
    pointillisticData_save.infoStruct(directoryPath, infoFileName, analysisParameters);
    
    %% Visualization (localizations)
    
    [analysisParameters, localizationImageMatrix, pixelCenters] = runOperation.processing(operations, 'pixelization', analysisParameters, localizationData);
    [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters, localizationImageMatrix, pixelCenters);
    
    % get the localization histograms:
    title = {dataName, 'localizations'};
    numberOfFrames = analysisParameters.sharedParameters.stackSize(3);
    exposureTime_ms = cameraSignalConversion.exposureTime_ms;
    pixelSize_nm = cameraSignalConversion.pixelSize_nm;
    countsPerElectron = cameraSignalConversion.countsPerElectron;
    histogramsFigure = rainSTORM_histograms(localizationData, localizationData_rejected, title, numberOfFrames, exposureTime_ms, countsPerElectron, pixelSize_nm);
    
    %% Saving the localization super-resolved figures
    
    % save the super-resolution image:
    outputLocalizationImageName=outputNaming.custom('relative', '', outputLocalizationDataFullFileName, '', '.tif');
    imageMatrixExport(localizationImageMatrix, colormapVector, pixelSize_nm, outputLocalizationImageName);
    % save the histograms image:
    outputHistogramsImageName=outputNaming.custom('relative', '', outputLocalizationDataFullFileName, '_Histograms', '.png');
    saveas(histogramsFigure, outputHistogramsImageName, 'png');
    
    %% Save the sum image
    
    frameSum = sum(uint32(frameStack),3);
    
    sharedParameters = analysisParameters.sharedParameters;
    prevSF = 100 * sharedParameters.supResImagePixelSize_nm/sharedParameters.cameraSignalConversion.pixelSize_nm;
                
    [sumImageFigure, sumImageMatrix] = rainSTORM_sumImage(frameSum, prevSF, dataName);
    % save the histograms image:
    sumImageName=outputNaming.custom('relative', '', outputLocalizationDataFullFileName, '_sumImage', '.png');
    imwrite(sumImageMatrix, sumImageName, 'png');
    
    %% Trajectory fitting/concatenation
    
    % concatenate the localizations into temporal trajectories:
    [analysisParameters_traj, trajectoryData, trajectoryData_complex] = runOperation.creation(operations, 'concatenatedData', 'concatenation', {analysisParameters}, localizationData);
    
    % save the trajectory data:
    outputConcatenatedDataFullFileName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_concatenatedData', '.csv');
    [directoryPath, dataFileName, extension] = fileparts(outputConcatenatedDataFullFileName);
    pointillisticData_save.tabularData(directoryPath, [dataFileName, extension], trajectoryData);
    infoFileName = [dataFileName, '_info', '.json'];
    pointillisticData_save.infoStruct(directoryPath, infoFileName, analysisParameters_traj);
    
    %% Visualization (trajectories)
    
    [analysisParameters_traj, trajectoryImageMatrix, pixelCenters] = runOperation.processing(operations, 'pixelization', analysisParameters_traj, trajectoryData);
    [analysisParameters_traj, trajectoryFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters_traj, trajectoryImageMatrix, pixelCenters);
    
    % get the trajectory photophysics histograms:
    cameraSignalConversion=analysisParameters_traj.sharedParameters.cameraSignalConversion;
    % camera parameters
    calcParamsStruct.exposureTime=cameraSignalConversion.exposureTime_ms/1000;
    calcParamsStruct.frameTime=cameraSignalConversion.frameInterval_ms/1000;
    calcParamsStruct.photonPerCount=1/cameraSignalConversion.countsPerPhoton;
    % filtering bounds
    calcParamsStruct.NPhotonBounds=[-Inf, Inf];
    calcParamsStruct.frameDurationBounds=[-Inf, Inf];
    calcParamsStruct.emPowerBounds=[-Inf, Inf];
    calcParamsStruct.lifeTimeBounds=[-Inf, Inf];
    % histogram parameters
    calcParamsStruct.NBins=50;
    
    quantityType={'photon number', 'emission power', 'frame duration', 'lifetime' };
    mapBool = false;
    frameIdxBool = false;
    concatComplexTrajsBool = false;
    [trajQuantitesAccepted, trajQuantitesRejected]=rainSTORM_trajectoryFitting.filter(trajectoryData, trajectoryData_complex, concatComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool);
    
    text_figure_name = 'Photophysics Histogram';
    photoPhysicsHistogramsFigure = rainSTORM_traj_photophysicsHist( trajQuantitesAccepted, trajQuantitesRejected, text_figure_name, calcParamsStruct );
    
    %% Saving the trajectory super-resolved figures:
    
    % save the super-resolved concatenated data image
    outputLocalizationImageName=outputNaming.custom('relative', '', outputConcatenatedDataFullFileName, '', '.tif');
    imageMatrixExport(trajectoryImageMatrix, colormapVector, pixelSize_nm, outputLocalizationImageName);
    
    % save the histograms image:
    outputHistogramsImageName=outputNaming.custom('relative', '', outputConcatenatedDataFullFileName, '_photophysicsHistograms', '.png');
    saveas(photoPhysicsHistogramsFigure, outputHistogramsImageName, 'png');
    
    
    %% Cluster analysis
    
    % choose which data to be clusterized:
    pointillisticData=trajectoryData;
    analysisParameters_cluster = analysisParameters_traj;
    %pointillisticData=localizationData;
    %analysisParameters_cluster = analysisParameters;
    
    [analysisParameters, pointillisticData, clusterProperties, operationIndex] = runOperation.processing(operations, 'clusterization', analysisParameters_cluster, pointillisticData);
    clustering_appliedSettings = analysisParameters.processingOperations{operationIndex}.settings;
    
    outputClusteredDataFullFileName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, '_clusteredData', '.csv');
    [directoryPath, dataFileName] = fileparts(outputClusteredDataFullFileName);
    pointillisticData_save.tabularData(directoryPath, dataFileName, pointillisticData);
    infoFileName = [dataFileName, '_info', '.json'];
    pointillisticData_save.infoStruct(directoryPath, infoFileName, analysisParameters_cluster);
    
    
    %% Save the cluster scatter plot
    
    clusterFig = rainSTORM_clusterAnalysis.plotClusters(pointillisticData, clusterProperties.centers, clustering_appliedSettings);
    clusterPropFig = rainSTORM_clusterAnalysis.plotClusterProperties(clusterProperties, clustering_appliedSettings);

    % save the cluster scatter plot as image:
    clusterImageName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, 'clusterImage', '.png');
    saveas(clusterFig, clusterImageName, 'png');
    % save the cluster scatter plot as Matlab figure:
    clusterFigName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, 'clusterFigure', '.fig');
    savefig(clusterFig, clusterFigName);
    % save the cluster histograms as Matlab figure:
    clusterHistogramsName=outputNaming.custom(outputPathType, resultDirectory_act, dataFileFullName, 'clusterHistograms', '.fig');
    savefig(clusterPropFig, clusterHistogramsName);
    
    %% Save the info file of the evaluation parameters
    
    [~, dataName, ~] = fileparts(dataFileFullName);
    outputInfoFullFileName = [ dataName, '_info', '.json'];
    pointillisticData_save.infoStruct( directoryPath, outputInfoFullFileName,  analysisParameters_cluster);
   


    %% free up the memory
    
    % delete the unnecessary datas:
    frameStack = [];
    frameStack_originalMetadata = [];
    frameStack_omeMetadata = [];
    localizationData = [];
    localizationData_rejected = [];
    filteringBooleanVect_localizations = [];
end

% removing the directories added for the script
removeSubfoldersFromPath()
