classdef imageStack_getConversionFactors
    
    methods (Static)
        
        function cameraSignalConversion = cameraSignal(cameraPrameters, additionalMagnification, cameraCalibrationFile)
            % Determines the conversions factors that are neccessary to
            % convert the raw camera signals to physical quantites based on
            % the extracted image stack metadata and the defined camera
            % properties.
            
            failedParametersList = {};
            
            cameraCalibration = jsonManagement.import(cameraCalibrationFile);
            
            try
                cameraSignalConversion.pixelSize_nm=cameraPrameters.pixelSize_nm / additionalMagnification;
            catch
                cameraSignalConversion.pixelSize_nm=nan;
                failedParametersList = {failedParametersList{:}, 'pixelSize_nm', ' '};
            end
            
            try
                cameraSignalConversion.exposureTime_ms=cameraPrameters.exposureTime_ms;
            catch
                cameraSignalConversion.exposureTime_ms=nan;
                failedParametersList = {failedParametersList{:}, 'exposureTime_ms', ' '};
            end
            
            try
                cameraSignalConversion.frameInterval_ms=cameraPrameters.frameInterval_ms;
            catch
                cameraSignalConversion.frameInterval_ms=nan;
                failedParametersList = {failedParametersList{:}, 'frameInterval_ms', ' '};
            end

            try
                countsPerPhoton=1/imageStack_getConversionFactors.photonsPerCounts(cameraCalibration, cameraPrameters);
                cameraSignalConversion.countsPerPhoton=countsPerPhoton;
            catch
                cameraSignalConversion.countsPerPhoton=nan;
                failedParametersList = {failedParametersList{:}, 'countsPerPhoton', ' '};
            end

            try
                electronsPerCount=imageStack_getConversionFactors.electronsPerADCount(cameraCalibration, cameraPrameters);
                cameraSignalConversion.countsPerElectron=1/electronsPerCount;
                
            catch
                cameraSignalConversion.countsPerElectron=nan;
                failedParametersList = {failedParametersList{:}, 'countsPerElectron', ' '};
            end
            
            try
                baseline=imageStack_getConversionFactors.baseline(cameraCalibration, cameraPrameters);
                cameraSignalConversion.baseline_count=baseline;
            catch
                cameraSignalConversion.baseline_count=nan;
                failedParametersList = {failedParametersList{:}, 'baseline', ' '};
            end
            
            if~isempty(failedParametersList)
                warning('Could not determine the following camera signal conversion factors using the image stack metadata and the camera calibration file: %s', [failedParametersList{:}])
            end
            
        end
        
        function QE = quantumEfficiency(cameraCalibration, cameraPrameters)
            
            cameraCellIndex = imageStack_getConversionFactors.findCamera(cameraCalibration, cameraPrameters);
            
            QE_vector = cameraCalibration(cameraCellIndex).properties.quantumEfficiency.value;
            wavelength_vector = cameraCalibration(cameraCellIndex).properties.quantumEfficiency.wavelength_nm;
            
            emissionWavelength=cameraPrameters.emissionWavelength_nm;
            
            if numel(QE_vector)~=numel(wavelength_vector)
                error('The number of wavelength and quantum efficiency data values do not match. Please correct the settings.')
            end 
            
            if numel(QE_vector)==1
                QE=QE_vector;
            elseif numel(QE_vector)>1
                QE=interp1(wavelength_vector, QE_vector, emissionWavelength);
            else
               error('') 
            end
        end
        
        
        function photoelectronsPerADCount = electronsPerADCount(cameraCalibration, cameraPrameters)
            % This function calculates the photoelectrons/count conversion factor for different camera settings
            
            cameraCellIndex = imageStack_getConversionFactors.findCamera(cameraCalibration, cameraPrameters);
            
            conversionFactorCellIndex = imageStack_getConversionFactors.findConversionFactor(cameraCalibration, cameraPrameters, cameraCellIndex);
            
            photoelectronsPerADCount=cameraCalibration(cameraCellIndex).properties.electronsPerADCount(conversionFactorCellIndex).value;
            
            
            if cameraPrameters.settings.emMultiplication
                EM_multiplication=cameraPrameters.settings.amplificationGain;
                photoelectronsPerADCount=photoelectronsPerADCount/EM_multiplication;
            end
            
        end
        
        
        function photonsPerCounts = photonsPerCounts(cameraCalibration, acquisitionPrameters)
            % This function calculates the photons/count conversion factor for different camera settings
            
            photoelectronsPerADCount = imageStack_getConversionFactors.electronsPerADCount(cameraCalibration, acquisitionPrameters);
            
            QE = imageStack_getConversionFactors.quantumEfficiency(cameraCalibration, acquisitionPrameters);
            
            photonsPerCounts = photoelectronsPerADCount/QE;
            
        end
        
        
        function photonFluxPerCountPerPixel = photonFluxPerCountPerPixel(cameraCalibration, cameraPrameters)
            % This function calculates the photon flux conversion factor incident on pixels based on the acquisition parameters.

            photonsPerCounts = imageStack_getConversionFactors.photonsPerCounts(cameraCalibration, cameraPrameters);
            
            photonFluxPerCountPerPixel = photonsPerCounts/exposureTime;
            
        end
        
        function baseline = baseline(cameraCalibration, cameraPrameters)
            
            cameraCellIndex = imageStack_getConversionFactors.findCamera(cameraCalibration, cameraPrameters);
            
            baseline=cameraCalibration(cameraCellIndex).properties.baseline;
            
        end
        
        function finalParameters = overwriteParameters(extractedParameters, definedParametersStruct, missingOnlyBoolean)
            % Over those camera signal conversion parameters that couldn't
            % have been extracted from the metadata (of the image stack or
            % of the saved csv file) with predifed ones. These parameters
            % are required for some analyses. The predefined parameters
            % might not be the "correct" ones, but at least the analyses
            % will run....
            
            definedParameters = struct();

            % convert the predefined parametrers to the "right" units
            % it already required a "cameraSignalConversion" (dummyConversionParameters)
            % variable to run, bet the predefined parameters should be
            % defined in physical units, so the values in the
            % "dummyConversionParameters" variable does not matter
            dummyConversionParameters = struct();
            dummyConversionParameters.pixelSize_nm = 1;
            dummyConversionParameters.exposureTime_ms = 1;
            dummyConversionParameters.frameInterval_ms = 1;
            dummyConversionParameters.countsPerElectron = 1;
            dummyConversionParameters.countsPerPhoton = 1;
            dummyConversionParameters.baseline_count = 1;
            if isfield(definedParametersStruct, 'pixelSize')
                definedParameters.pixelSize_nm = unitConversion.convertValue(definedParametersStruct.pixelSize.value, definedParametersStruct.pixelSize.unit, 'nm', dummyConversionParameters);
            end
            if isfield(definedParametersStruct, 'exposureTime')
                definedParameters.exposureTime_ms = unitConversion.convertValue(definedParametersStruct.exposureTime.value, definedParametersStruct.exposureTime.unit, 'ms', dummyConversionParameters);
            end
            if isfield(definedParametersStruct, 'frameInterval')
                definedParameters.frameInterval_ms = unitConversion.convertValue(definedParametersStruct.frameInterval.value, definedParametersStruct.frameInterval.unit, 'ms', dummyConversionParameters);
            end
            if isfield(definedParametersStruct, 'countsPerElectron')
                definedParameters.countsPerElectron = definedParametersStruct.countsPerElectron.value;
            else
                % add this, so the baseline conversion can be performed
                % in more cases...
                definedParameters.countsPerElectron = extractedParameters.countsPerElectron;
            end
            if isfield(definedParametersStruct, 'countsPerPhoton')
                definedParameters.countsPerPhoton = definedParametersStruct.countsPerPhoton.value;
            else
                % add this, so the baseline conversion can be performed
                % in more cases...
                definedParameters.countsPerPhoton = extractedParameters.countsPerPhoton;
            end
            if isfield(definedParametersStruct, 'baseline')
                dummyConversionParameters.countsPerElectron = definedParameters.countsPerElectron;
                dummyConversionParameters.countsPerPhoton = definedParameters.countsPerPhoton;
                definedParameters.baseline_count = unitConversion.convertValue(definedParametersStruct.baseline.value, definedParametersStruct.baseline.unit, 'camera count', dummyConversionParameters);
            end
            
            parameterNames = fieldnames(extractedParameters);
            overwrittenParameterList = {};
            
            % update all the parameters that could be extracted from the image metadata
            % if they are defined by the user:
            for idxParameter = 1:numel(parameterNames)
                
                parameter = parameterNames{idxParameter};
                
                finalParameters.(parameter) = extractedParameters.(parameter);
                
                if missingOnlyBoolean
                    if isnan(finalParameters.(parameter)) && isfield(definedParameters, parameter)
                        if ~isempty(definedParameters.(parameter))
                            finalParameters.(parameter) = definedParameters.(parameter);
                            overwrittenParameterList = [overwrittenParameterList, parameter, ' '];
                        end
                    end
                else
                    if isfield(definedParameters, parameter)
                        if ~isempty(definedParameters.(parameter))
                            finalParameters.(parameter) = definedParameters.(parameter);
                            overwrittenParameterList = [overwrittenParameterList, parameter, ' '];
                        end
                    end
                end
                
            end
            
            if~isempty(overwrittenParameterList)
                warning('The following camera signal conversion parameters were overwritten with predefined ones: %s.', [overwrittenParameterList{:}])
            end

            % check the physical validity of the (partially) dependent parameters, if they are not
            % valid, assume ideal detertor and modify them:
            if isnan(finalParameters.exposureTime_ms) && ~isnan(finalParameters.frameInterval_ms)
                finalParameters.exposureTime_ms = finalParameters.frameInterval_ms;
            elseif isnan(finalParameters.frameInterval_ms) && ~isnan(finalParameters.exposureTime_ms)
                finalParameters.frameInterval_ms = finalParameters.exposureTime_ms;
            end
            if finalParameters.frameInterval_ms<finalParameters.exposureTime_ms
                finalParameters.frameInterval_ms=finalParameters.exposureTime_ms;
            end
            if isnan(finalParameters.countsPerElectron) && ~isnan(finalParameters.countsPerPhoton)
                finalParameters.countsPerElectron = finalParameters.countsPerPhoton;
            elseif isnan(finalParameters.countsPerPhoton) && ~isnan(finalParameters.countsPerElectron)
                finalParameters.countsPerPhoton = finalParameters.countsPerElectron;
            end
            if finalParameters.countsPerPhoton>finalParameters.countsPerElectron
                finalParameters.countsPerPhoton=finalParameters.countsPerElectron;
            end
            
            % check which parameters are still missing:
            missingParametersList = {};
            for idxParameter = 1:numel(parameterNames)
                
                parameter = parameterNames{idxParameter};
                
                if isnan(finalParameters.(parameter))
                    missingParametersList = [missingParametersList, parameter, ' '];
                end
                
            end
            
            if~isempty(missingParametersList)
                warning('The following camera signal conversion parameters are still missing after updating them with the user defined ones: %s \n Define them in the default or used settings file or overwrite them inside the code if they are needed for the further analysis.', [missingParametersList{:}])
            end
            
        end
        
    end
       
    
    methods (Static, Access=private)
        
        function cameraCellIndex = findCamera(cameraCalibration, cameraPrameters)
        
            cameraN=numel(cameraCalibration);
            
            cameraCellIndex=0;
            for idxCamera=1:cameraN
               
                serialNumber_act=cameraCalibration(idxCamera).serialNumber;
                
                if strcmp(serialNumber_act, cameraPrameters.serialNumber)
                    
                    cameraCellIndex=idxCamera;
                    
                    break;
                end
            end
            
            if cameraCellIndex == 0
               error('Could not identify the camera.')
            end
            
        end
        
        
        function conversionFactorCellIndex = findConversionFactor(cameraCalibration, cameraPrameters, cameraCellIndex)
            
            conversionFactorN=numel(cameraCalibration(cameraCellIndex).properties.electronsPerADCount);
            
            conversionFactorCellIndex=0;
            for conversionFactorIdx=1:conversionFactorN
                
                actSettings=cameraCalibration(cameraCellIndex).properties.electronsPerADCount(conversionFactorIdx).settings;
                
                cameraSettings=cameraPrameters.settings;
                if isfield(cameraSettings,'amplificationGain')
                   cameraSettings=rmfield(cameraSettings,'amplificationGain');
                end
                
                % compare the two structures
                if isequal(actSettings, cameraSettings)
                    
                    conversionFactorCellIndex=conversionFactorIdx;
                    
                    break;
                end
            end
            
            if conversionFactorCellIndex == 0
                error('Could not identify which camera readout settings was used.')
            end
            
        end
        
        
    end
end

