classdef imageStack_linearFiltering

    methods(Static)
    
        function [backgroundStack] = lowPassFourier(imageStack, backgroundFilteringSettings)
            % This function ferform a temporal low pass frequency filtering on the
            % given image stack in order to remove the static or slowly varying
            % background. Performs discrete Fourier transformation on the "z" dimension
            % (which normalyy corresponds to the "time"), applies the given window
            % function on the spectrum and tranforms it back.
            
            frameStackClass=class(imageStack);
            frameStackClassFunctionHandle=str2func(frameStackClass);
            
            % passing the settings:
            frequencyWidth=backgroundFilteringSettings.frequencyWidth;
            frameTime=backgroundFilteringSettings.frameTime;
            frequencyWindowType=backgroundFilteringSettings.frequencyWindowType;
            
            % image stack measurement parameters:
            samplingN=size(imageStack, 3);
            measurementTime=samplingN*frameTime;
            
            % properties of the DFT:
            df=1/measurementTime;
            fmax=df*(samplingN-1);
            frequencyPoints=0:df:fmax;
            
            % Calculating the frequency window:
            % TODO: find a general way for calculating these function (required here, required for the kernel function calculation....)
            switch frequencyWindowType
                case 'Gaussian'
                    windowVect=zeros(1,1,samplingN);
                    frequencySTD=frequencyWidth/sqrt(8*log(2));
                    windowVect(1,1,:)=exp(-(frequencyPoints).^2/(2*frequencySTD^2))+exp(-(frequencyPoints-fmax).^2/(2*frequencySTD^2));
                case 'rectangular'
                    windowVect=zeros(1,1,samplingN);
                    windowVect(1,1,frequencyPoints<frequencyFWHM/2 | frequencyPoints>fmax-frequencyFWHM/2)=1;
                otherwise
                    error('Uknown window type for the low pass frequency filtering.')
            end
            
            % window for the whole stack:
            stackSize=size(imageStack);
            repetitonSize=stackSize;
            repetitonSize(3)=1;
            windowMat=repmat(windowVect, repetitonSize);
            
            % Fourier transform of the image
            imageDFT=fft(imageStack, [], 3);
            
            % low pass Fourier filtering:
            imageDFT=imageDFT.*windowMat;
            
            % get the background after from the low frequency spectrum:
            background=real(ifft(imageDFT, [], 3));
            
            % return the background as grey scale image stack:
            backgroundStack=frameStackClassFunctionHandle(background);
            
        end

    end
end
