classdef imageStack_saving
    % Collection of methods for saving the processed image data the data
    % into an image stack.
    
    methods (Static)
        
        function save(frameStack, outputFullFileName, originalMetadata, omeMetadata)
            % This function saves the given array to an image stack to the
            % output path. The given output file's extension determines the
            % file format the stack is saved to.
            % Currently, can only save into *.tif files with 16 bit depth.
            
            % getting the extension of the stack to be saved
            [~, ~, outputFileExt]=fileparts(outputFullFileName);
            
            %% Saving the background filtered stack
            disp('Saving the image stack...')
            % Writing out the background substracted stack.
            if strcmp(outputFileExt, '.tif') || strcmp(outputFileExt, '.tiff')
                imageStack_saving.tif(frameStack, outputFullFileName, originalMetadata, omeMetadata)
            else
                % If a name with wrong extension was given as the output...
                error('Saving the image stack opnly to TIFF format is supported.');
            end
            disp('Finished saving the stack.')
            
        end
        
        
        function multipage_tif(frameStack, outputFullFileName, originalMetadata, omeMetadata)
            % This function saves the given image stack into a TIFF file
            % with the OME metadata.
            
            % using the modified "saveastiff" function
            imageStack_saving.saveastiff(frameStack, outputFullFileName, originalMetadata, omeMetadata);
            
            % do not use the bioformat's bfsave function
            % The bfsave is buggy.
            % - Saving the image stack with LZW
            % compresssion its size is larger than files saved with the
            % "saveastiff" function.
            % - Saving the image stack to "*.tif" format "*.ome.tiff" with
            % the original metadata, discards most of the metadata
            % - Saving the image stack to ".ome.tiff" format with the
            % original metadata caueses corrupted (upscaled?) image stack with the
            % right metadata (saving the image stack with the example% 
            % minimal metadata produces correct image stack)
            
            % Moreover the 6.4.0. version consumes a very high amount of memory
            %  and the output stack is corrupted
            
            %imageStack_saving.bioformats(frameStack, outputFullFileName, originalMetadata, omeMetadata);
        end
        
        function bioformats(frameStack, outputFullFileName, originalMetadata, omeMetadata)
            % see:
            % https://docs.openmicroscopy.org/bio-formats/6.6.0/developers/matlab-dev.html#saving-files
            
            if isfile(outputFullFileName)
                % delete the stack if it exist
                % the "bfsave" function appends to the existing
                % file and not overwrites it.
                delete(outputFullFileName);
            end
            
            % changing the dimensions does not help with the
            % plain count problem:
            frameStack=reshape(frameStack(:,:,:), [size(frameStack, 1), size(frameStack, 2), 1, 1, size(frameStack, 3)]);
            
            % bfsave(frameStack, outputFullFileName, 'metadata', omeMetadata);
            bfsave(frameStack, outputFullFileName,'Compression', 'LZW', 'metadata', omeMetadata);
            % bfsave(frameStack, outputFullFileName,'Compression', 'LZW', 'metadata', omeMetadata, 'BigTiff', true);
            
            
            % TODO:
            % - save the original metadata?
        end
        
        function saveastiff(frameStack, outputFullFileName, originalMetadata, omeMetadata)
            % If 'tiff' was chosen as the output format, use the 'saveastiff'
            % external function
            tic
            options.overwrite=true;
            options.compress='lzw';
            %options.compress='adobe';
            options.color=false;
            options.message=false;
            
            % using an extarnal function, should be much faster than
            % MATLAB's "imwrite" for stacks containing large number of
            % frames
            % taken from: https://www.mathworks.com/matlabcentral/fileexchange/35684-save-and-load-a-multiframe-tiff-image
            
            % saving the image stack with the OME metadata:
            %saveastiff(frameStack, outputFullFileName, options);
            
            % converting the OME metadata object to XMl string:
            imageDescriptionTag=char(omeMetadata.dumpXML());
            saveastiff_modified(frameStack, outputFullFileName, options, imageDescriptionTag);
            
            % Could pass the metadata to the TIFF ImageDescription tag, but
            % it still not follows the OME specification, there is some
            % problem with the plane idices. I guess, according to forums,
            % it has something to do with the TIFF Image File Directory
            % offsets which I could not set....
            
            
            toc
        end
        
    end
    
end

