classdef imageStack_selection
    % Collection of methods for selecting image stacks.
    
    methods (Static)
        
        function stackFormatFilterList=getStackFormatFilterList()
            % This function holds the definition of the image stack format
            % list from which can be chosen in the file selection dialog.
            
            stackFormatFilterList={...
                '*.tif', 'TIFF image stack (*.tif)';...
                '*.tiff', 'TIFF image stack (*.tiff)';...
                '*.raw', 'RAW image stack (*.raw)';...
                '*.nd2', 'Nikon image stack (*.nd2)';...
                '*.*', 'All files'};
        end
        
        
        function imageFolderTypeList=getImageFolderTypeList()
            % the list is image stack format from which can be chosen in
            % the file selection dialog.
            
            imageFolderTypeList={...
                'TIFF folder'};
        end
        
        
        function [imageStackFullNameList, imageStackFormat] = singleStack(varargin)
            % This function lets the user select a single image stack.
            
            % check the input arguments:
            if numel(varargin)==0
                % open the dialog in the actual work path:
                defaultPath='';
            elseif numel(varargin)==1
                % open the dialog in the given path:
                defaultPath=varargin{1};
            else
                error('Invalid number if input arguments')
            end
            
            stackFormatFilterList=imageStack_selection.getStackFormatFilterList();
            
            % File chooser dialog to select one image stacks:
            [dataFiles, dataPath, filterIndex] = uigetfile(...
                stackFormatFilterList,...
                'Select a single image stack',...
                defaultPath,...
                'MultiSelect', 'off');
            
            if iscell(dataFiles) || ischar(dataFiles)
                imageStackFormat=stackFormatFilterList{filterIndex, 2};

                [imageStackFullNameList, ~] =...
                    imageStack_selection.conformFileNameList(dataFiles, dataPath);
            elseif dataFiles == 0
                imageStackFormat = '';
                imageStackFullNameList = {};
            end
        end
        
        
        function [imageStackFullNameList, imageStackFormat] = multipleStacks(varargin)
            % This function lets the user select multiple image stacks.
            
             % check the input arguments:
            if numel(varargin)==0
                % open the dialog in the actual work path:
                defaultPath='';
            elseif numel(varargin)==1
                % open the dialog in the given path:
                defaultPath=varargin{1};
            else
                error('Invalid number if input arguments')
            end
            
            stackFormatFilterList=imageStack_selection.getStackFormatFilterList();
            
            % File chooser dialog to select one or several image stacks:
            [dataFiles, dataPath, filterIndex] = uigetfile(...
                stackFormatFilterList,...
                'Select multiple image stacks',...
                defaultPath,...
                'MultiSelect', 'on');
            
            if iscell(dataFiles) || ischar(dataFiles)
                imageStackFormat=stackFormatFilterList{filterIndex, 2};

                [imageStackFullNameList, ~] =...
                    imageStack_selection.conformFileNameList(dataFiles, dataPath);
            elseif dataFiles == 0
                imageStackFormat = '';
                imageStackFullNameList = {};
            else
                error('Some error during image stack selection. Check what has happened, it should not have occured!')
            end
        end
        
        
        function [imageDataPath, imageStackFormat] = TIFFfolder(varargin)
            % This function lets the user to select a folder that contains
            % the image data in several TIFF image files.
            
            imageDataPath = imageStack_selection.folder(varargin);
            
            if imageDataPath == 0
                imageStackFormat = '';
                imageDataPath = '';
            else
                imageFolderTypeList=imageStack_selection.getImageFolderTypeList();
                imageStackFormat=imageFolderTypeList{1};
            end
        end
        
    end
    
    
    methods (Static, Access=private)
        
        function [dataFileFullNameList, N_dataFiles, dataFileNameList, dataFileExtList] = conformFileNameList(dataFiles, dataPath)
            % This function conforms the file name list for further use.
            % If only one file was selected, puts its name into a one
            % element cell array to be consistent with multiple file
            % selection. It also concatenates the file paths and names to
            % get the full paths.
            
            % The number of input data files
            if iscell(dataFiles)
                N_dataFiles=numel(dataFiles);
            else
                N_dataFiles=1;
            end
            
            % full name of the data files
            dataFileFullNameList=cell(1,N_dataFiles);
            % names of the data files
            dataFileNameList=cell(1,N_dataFiles);
            % extensions of the data files
            dataFileExtList=cell(1,N_dataFiles);
            
            % adding the data files into a cell array (the "list")
            if N_dataFiles==1
                dataFileFullNameList{1}=[dataPath, dataFiles];
                [~, dataFileNameList{1}, dataFileExtList{1}]=fileparts(dataFiles);
            else
                for idxFile=1:N_dataFiles
                    dataFileFullNameList{idxFile}=[dataPath, dataFiles{idxFile}];
                    [~, dataFileNameList{idxFile}, dataFileExtList{idxFile}]=fileparts(dataFiles{idxFile});
                end
            end
        end
        
        
        function [imageDataPath] = folder(varargin)
            % This function lets the user to select a folder that contains
            % the image data in seperate files.
            
             % check the input arguments:
            if numel(varargin)==0
                % open the dialog in the actual work path:
                defaultPath='';
            elseif numel(varargin)==1
                % open the dialog in the given path:
                defaultPath=varargin{1};
            else
                error('Invalid number if input arguments')
            end
            
            
            [imageDataPath] = uigetdir(...
                defaultPath,...
                'Select an folder containing images frames');
        end
        
    end
end

