classdef exponentialSampling
% Collection of methods for modelling the discretized data set of an
% exponentially distributed quantity.
    
    
    methods (Static)
        
        function sampledExponential=getSampledValues(samplingCoordiantes, exponentialParameters, exponentialModelSettings)
            % This function returns the discretized data set of multi 
            % exponential distribution with constant background using the
            % chosen sampling method (point samplign or integration over bins).
            
            % number of single exponential in the distributions:
            exponentialNumber=numel(exponentialParameters.decay);
            if numel(exponentialParameters.weight)~=exponentialNumber
                % the number of relative weights of the exponentials must match the
                % number of exponentials
                error('The number of exponential weight and decays must match.')
            end
            
            % initialize the strucrure for holding the parameters of the
            % actual single exponential:
            exponentialModelSettings_act=struct();
            switch exponentialModelSettings.samplingType
                case 'sampling at points'
                    % if point sampling was chosen
                    
                    % function handle for the point sampling:
                    samplingFunctionHandle=@exponentialSampling.sampingAtPoints;
                    % area of the single exponentials of the multi
                    % exponential distributioin:
                    areaUnderSingleExponentials=exponentialSampling.getAreaUnderDistribution_pointSampling(exponentialParameters, exponentialModelSettings, exponentialModelSettings.interval);
                case 'bin central value approximation'
                    % if approximative integral over the bin were chosen:
                    
                    % function handle for the bin sampling:
                    samplingFunctionHandle=@exponentialSampling.binCetralValueApproximation;
                    % area of the single exponentials of the multi
                    % exponential distributioin:
                    areaUnderSingleExponentials=exponentialSampling.getAreaUnderDistribution_binSampling(exponentialParameters, exponentialModelSettings, samplingCoordiantes, @exponentialSampling.binCetralValueApproximation);
                    % other parameters for the sampling:
                    exponentialModelSettings_act.binSize=exponentialModelSettings.binSize;
                case 'integrating over bins, fixed start'
                    % if integral over the bin were chosen:
                    
                    % function handle for the bin sampling:
                    samplingFunctionHandle=@exponentialSampling.integratingOverBins_fixedStart;
                    % area of the single exponentials of the multi
                    % exponential distributioin:
                    areaUnderSingleExponentials=exponentialSampling.getAreaUnderDistribution_binSampling(exponentialParameters, exponentialModelSettings, samplingCoordiantes, @exponentialSampling.integratingOverBins_fixedStart);
                    % other parameters for the sampling:
                    exponentialModelSettings_act.binSize=exponentialModelSettings.binSize;
                case 'integrating over bins, random start'
                    % if frame duration distribution was chosen:
                    
                    % function handle for the frame duration:
                    samplingFunctionHandle=@exponentialSampling.integratingOverBins_randomStart;
                    % area of the single exponentials of the multi
                    % exponential distributioin:
                    areaUnderSingleExponentials=exponentialSampling.getAreaUnderDistribution_binSampling(exponentialParameters, exponentialModelSettings, samplingCoordiantes, @exponentialSampling.integratingOverBins_randomStart);
                    % other parameters for the sampling:
                    exponentialModelSettings_act.deadTime=exponentialModelSettings.deadTime;
                    exponentialModelSettings_act.minimalDetectionTime=exponentialModelSettings.minimalDetectionTime;
                otherwise
                    error('Unknown exponential samplingtype.')
            end
            
            % go through the single exponentials:
            sampledExponential=zeros(size(samplingCoordiantes));
            for idxExponential=1:exponentialNumber
                % the decay rate of the actual exponential:
                exponentialModelSettings_act.decay=exponentialParameters.decay(idxExponential);
                
                % add the actual single exponential to the whole
                % distribution:
                sampledExponential=sampledExponential+areaUnderSingleExponentials(idxExponential)*samplingFunctionHandle(exponentialParameters, exponentialModelSettings_act, samplingCoordiantes);
            end
            
            % add the constant background:
            sampledExponential=sampledExponential+exponentialParameters.background;
            
        end

        
        function sampledExponential=samplingAtPoints(exponentialParameters, exponentialModelSettings, samplingPoints)
            % This function return the values of the given single 
            % exponential distribution at the given points.
            
            % decay of the single exponential distribution:
            expDecay=exponentialParameters.decay;
            
            % values at the given sampling points of the normalized distribution:
            sampledExponential=expDecay*exp(-expDecay*samplingPoints);
            
            % the values at points below 0 should be 0 (the distribution should be a probability distribution)
            sampledExponential(samplingPoints<0)=0;
        end
        
        
        function sampledExponential=binCetralValueApproximation(exponentialParameters, exponentialModelSettings, binCenters)
            % This function return the integrated values of the given
            % single exponential distribution over the given bins. The
            % integration is approximated with the central value of the
            % bin.
            
            % decay of the single exponential distribution:
            expDecay=exponentialParameters.decay;
            
            % values at the given bin centers of the normalized distribution:
            sampledExponential=expDecay*exp(-expDecay*binCenters)*exponentialModelSettings.binSize;
            
            % the values at points below 0 should be 0 (the distribution should be a probability distribution)
            sampledExponential(binCenters<0)=0;
        end
        
        
        function sampledExponential=integratingOverBins_fixedStart(exponentialParameters, exponentialModelSettings, binCenters)
            % This function return the integrated values of the given
            % single exponential distribution over the given bins. The
            % integration of the exponential is calculated ananilically
            % over the bin.
            
            % decay of the single exponential distribution:
            expDecay=exponentialParameters.decay;
            
            % lower and upper bounds of the integrations:
            binLowerEdges=binCenters-exponentialModelSettings.binSize/2;
            binUpperEdges=binCenters+exponentialModelSettings.binSize/2;
            
            % if the integration would go below 0, do not integrate there
            % (the exponential distribution should be 0 below 0):
            binLowerEdges(binLowerEdges<0)=0;
            binUpperEdges(binUpperEdges<0)=0;
            
            % values of the bins of the normalized distribution:
            sampledExponential=exp(-expDecay*binLowerEdges)-exp(-expDecay*binUpperEdges);
            
        end
        
        
        function sampledExponential=integratingOverBins_randomStart(exponentialParameters, exponentialModelSettings, frameN)
            % This function calculates the modelled distribution of the
            % discretized dataset when both that the start and the end of the
            % random process lies random within the integration time
            % intervals used for discretization.
            % It was originally developed for calculating the frame
            % duration probablility of blinking events.
            
            if prod(rem(frameN, 1)==0)==1
                % the frame numbers must be integers
            else
                error('Please give integer frame numbers.');
            end
            
            % dead time ratio (compared to the frame time) of the detector:
            deadTime=exponentialModelSettings.deadTime;
            % minimum emission time ratio (compared to the frame time) that
            % provides the sufficient minimal number of photons for
            % localization:
            minimalDetectionTime=exponentialModelSettings.minimalDetectionTime;
            
            % decay of the single exponential distribution:
            expDecay=exponentialParameters.decay;
            
            switch 'time duration'
                case 'frame number'
                    % Approach where the formulas are drawn from the
                    % starting time and time duration conditions required 
                    % for the blinking event to be detected on the given
                    % number of frames.
                    
                    % let fit the lifetime in frame duration number and not
                    % in other units:
                    frameTime=1;                        
                    
                    % lifetime of the single exponential process:
                    tau=1/expDecay;
                    
                    const_F_N=tau/frameTime*(exp(frameTime/tau)-1)^2*exp(-(deadTime+2*minimalDetectionTime)/tau);
                    
                    % frame duration probability when the the frame numbers
                    % are largen than 2:
                    frameNProbability=const_F_N*exp(-(frameN*frameTime/tau));
                    
                    % correction factor for the frame numbers of 0 and 1:
                    P_0_corr=1-tau/frameTime*exp(-(2*minimalDetectionTime+deadTime-frameTime)/tau)*(exp(frameTime/tau)-2)-(frameTime-deadTime-minimalDetectionTime+tau)/frameTime*exp(-minimalDetectionTime/tau);
                    P_1_corr=(tau+frameTime-deadTime-minimalDetectionTime)/frameTime*exp(-minimalDetectionTime/tau)-tau/frameTime*exp(-(deadTime+2*minimalDetectionTime-frameTime)/tau);
                    % correct the frame numbers of 0 and 1:
                    frameNProbability=frameNProbability+(frameN==0)*P_0_corr;
                    frameNProbability=frameNProbability+(frameN==1)*P_1_corr;
                    
                    % the frame duration probability distribution:
                    sampledExponential=frameNProbability;

                case 'time duration'
                    % Approach where the formulas are drawn from the
                    % probabilities of how many frames will be the blinking
                    % event with fixed time duration but with random
                    % starting time detected on.
                    
                    %% Initializing constants
                    % minimal number of frames on the histogram
                    frameNMin=ceil(frameN(1));
                    % maximal number of frames on the histogram
                    frameNMax=floor(frameN(end));
                    % frame time
                    frameTime=1;                        % let fit the lifetime in frame duration number and not in seconds
                    
                    if frameNMin<0
                        error('The minimal frame duration should not be smaller than "0"')
                    end
                    
                    NMin_frameNMin_shift=-2;
                    NMax_frameNMax_shift=1;
                    
                    % let "N" donate to the floor(t/frameTime) and , let "t"
                    % donate to the time duration of the emission
                    NMin=(frameNMin+NMin_frameNMin_shift);
                    NMax=(frameNMax+NMax_frameNMax_shift);
                    
                    % time interval within if the end of the trajectory falls, then it is not
                    % detected on the that frame (let this time interval be placed on the
                    % beginning of every frame), blind time
                    H=deadTime+2*minimalDetectionTime;
                    
                    % vector containing the lower limits of integration, where the length
                    % of the emission is "N*frameTime<t<N*frameTime+H"
                    N_vect=NMin:1:NMax-1;
                    TLowerLimit_vect=N_vect*frameTime;
                    
                    % integration should not start from "0" but from the "minDetEmTime" as
                    % blinking shorter than this, can not be localized at all
                    TLowerLimit_vect_mod=TLowerLimit_vect;
                    TLowerLimit_vect_mod(N_vect==0)=minimalDetectionTime;
                    
                    % vector containing the upper limits of integration, where the length
                    % of the emission is
                    % "N*frameTime<t<N*frameTime+H", and the
                    % lower limit of the other integration
                    T_H_vect=TLowerLimit_vect+H;
                    % vector containing the upper limits of integration, where the length
                    % of the emission is
                    % "N*frameTime+H<t<(N+1)*frameTime"
                    TUpperLimit_vect=TLowerLimit_vect+frameTime;
                    
                    %% Integrations (ananlytical) for the frame number probability
                    % constant for the frame duration probability, if
                    % "N*frameTime<t<N+H"
                    c1_withinH_vect=(TLowerLimit_vect+H)/frameTime;
                    c2_withinH_vect=1-c1_withinH_vect;
                    
                    % constant for the frame duration probability, if
                    % "N*frameTime+H<t<(N+1)*frameTime"
                    c2_outsideH_vect=(TUpperLimit_vect+H)/frameTime;
                    c3_outsideH_vect=1-c2_outsideH_vect;
                    
                    % integrals of the exponentials evaluated at the different integral limits
                    exp_TLower_vect=-expDecay*exp(-expDecay*TLowerLimit_vect_mod)/expDecay;
                    exp_T_H_vect=-expDecay*exp(-expDecay*T_H_vect)/expDecay;
                    exp_TUpper_vect=-expDecay*exp(-expDecay*TUpperLimit_vect)/expDecay;
                    
                    % integrals, that starts in the negative regime, should not be accounted for
                    exp_TLower_vect(N_vect<0)=0;
                    exp_T_H_vect(N_vect<0)=0;
                    exp_TUpper_vect(N_vect<0)=0;
                    
                    % probability of the trajectory that it resides on "floor(t/frameTime)"
                    % number of frames, if
                    % "N*frameTime<t<N*frameTime+H"
                    p1_withinH_vect=...
                        (c1_withinH_vect-(T_H_vect+1/expDecay)/frameTime).*exp_T_H_vect...
                        -(c1_withinH_vect-(TLowerLimit_vect_mod+1/expDecay)/frameTime).*exp_TLower_vect;
                    
                    % probability of the trajectory that it resides on "floor(t/frameTime)+1"
                    % number of frames, if
                    % "N*frameTime<t<N*frameTime+H"
                    p2_withinH_vect=...
                        (c2_withinH_vect+(T_H_vect+1/expDecay)/frameTime).*exp_T_H_vect...
                        -(c2_withinH_vect+(TLowerLimit_vect_mod+1/expDecay)/frameTime).*exp_TLower_vect;
                    
                    
                    % probability of the trajectory that it resides on "floor(t/frameTime)+1"
                    % number of frames, if
                    % "N*frameTime+H<t<(N+1)*frameTime"
                    p2_outsideH_vect=...
                        (c2_outsideH_vect-(TUpperLimit_vect+1/expDecay)/frameTime).*exp_TUpper_vect...
                        -(c2_outsideH_vect-(T_H_vect+1/expDecay)/frameTime).*exp_T_H_vect;
                    % probability of the trajectory that it resides on "floor(t/frameTime)+2"
                    % number of frames, if
                    % "N*frameTime+H<t<(N+1)*frameTime"
                    p3_outsideH_vect=...
                        (c3_outsideH_vect+(TUpperLimit_vect+1/expDecay)/frameTime).*exp_TUpper_vect...
                        -(c3_outsideH_vect+(T_H_vect+1/expDecay)/frameTime).*exp_T_H_vect;
                    
                    %% Calculating the frame number probabilities
                    % probabilty of the different frame durations
                    frameNProbability=p1_withinH_vect(1-NMin_frameNMin_shift:end)...
                        +p2_withinH_vect(1-NMin_frameNMin_shift-1:end-1)...
                        +p2_outsideH_vect(1-NMin_frameNMin_shift-1:end-1)...
                        +p3_outsideH_vect(1-NMin_frameNMin_shift-2:end-2);
                    
                    % make sure that the function to fit results in the same number of
                    % localization as the measurement
                    sampledExponential=frameNProbability;
                    
                    %% For testing
                    %histEdges=linspace(frameNMin-0.5,frameNMax+0.5, frameNMax-frameNMin+2);
                    %frameNProbability_normalized=histTrajN/(exp(-expDecay*histEdges(end))-exp(-expDecay*histEdges(1)))*(exp(-expDecay*histEdges(2:end))-exp(-expDecay*histEdges(1:end-1)));
                    %histogram('BinCounts', frameNProbability_normalized, 'BinEdges', histEdges)
                    
                    % figure()
                    % scatter(frameN_vect, frameNProbability)
                    
                    % % frame duration probability, for fixed "t", if
                    % % "N*frameTime<t<N*frameTime+H"
                    % t=(NMin:0.01:NMin+2)*frameTime;
                    %
                    % p1_withinH_test=(c1_withinH_vect(floor(t/frameTime)+1)-(t)/frameTime);
                    % p2_withinH_test=(c2_withinH_vect(floor(t/frameTime)+1)+(t)/frameTime);
                    %
                    % p2_outsideH_vect_test=(c2_outsideH_vect(floor(t/frameTime)+1)-(t)/frameTime);
                    % p3_outsideH_vect_test=(c3_outsideH_vect(floor(t/frameTime)+1)+(t)/frameTime);
                    %
                    % figure()
                    % plot(t, p1_withinH_test, t, p2_withinH_test)
                    % figure()
                    % plot(t, p2_withinH_test, t, p2_outsideH_vect_test)
                    % figure()
                    % plot(t, p2_outsideH_vect_test, t, p3_outsideH_vect_test)
            end
        end
        
        
        function area=getAreaUnderDistribution_pointSampling(exponentialParameters, exponentialModelSettings, interval)
            % This function returns the value of the exponential
            % distribution at 0 based on the given given area under the
            % distribution within an interval or based on the given value a
            % at the left boundary of the interval ("initialValue").
            
            if isfield(exponentialModelSettings, 'interval')
                if ~isfield(exponentialParameters, 'sampleNumber')
                    error('Please give the number of samples .')
                end
                intervalLength=exponentialModelSettings.interval(2)-exponentialModelSettings.interval(1);
            else
                intervalLength=[];
            end
            
            exponentialParameters = exponentialSampling.removeBackground(exponentialParameters, intervalLength);
            
            % the exponential decay  of the random process:
            expDecay=exponentialParameters.decay;
            
            if isfield(exponentialParameters, 'valueAtZero')
                area=exponentialParameters.valueAtZero./expDecay;
            elseif isfield(exponentialParameters, 'initialValue')
                % If the value at the left bounds of the interval was
                % given:
                area=exponentialParameters.initialValue.*exp(interval(1)*expDecay)./expDecay;
            elseif isfield(exponentialParameters, 'sampleNumber')
                % if the area within the interval was given:
                area=exponentialParameters.sampleNumber./(exp(-expDecay*interval(1))-exp(-expDecay*interval(2)));   % the initial value and the normalization factor (the area under the exponential should match that of the histogram)
            else
                error('At least the "area" under the distribution or its "initial value" must be given.')
            end
            
        end
        
        
        function area=getAreaUnderDistribution_binSampling(exponentialParameters, exponentialModelSettings, bins, functionHandle)
            % This function returns the areas under the single exponential
            % distributions of the given multi exponential distribution. 
            
            % sampling interval "lenght" from with the background's area
            % contribution can be calculated:
            intervalLength=numel(bins);
            
            % remove the contribution of the background to the given
            % multi exponential exponential parameters:
            exponentialParameters = exponentialSampling.removeBackground(exponentialParameters, intervalLength);
            
            % the exponential decay  of the random process:
            expDecay=exponentialParameters.decay;
            
            % normalized weights of the individual single exponentials of
            % the multi exponential function:
            weights=exponentialParameters.weight/sum(exponentialParameters.weight);
            
            % initialize the vector containing the relative contributions
            % of the individual single exponentials to the given value tio be fit:
            singleExpContributions=zeros(size(expDecay));
            % initialize the parameters of the actual single exponential:
            exponentialParameters_act=exponentialParameters;
            % remove the unnecessary fields if needed:
            %exponentialParameters_act=rmfield(exponentialParameters_act,'weight');
            % go throught he single exponentials:
            for idxExp=1:numel(expDecay)
                
                % decay rate of the actual exponential:
                exponentialParameters_act.decay=expDecay(idxExp);
                
                if isfield(exponentialParameters, 'valueAtZero')
                    % if the multiexponential function's value at 0 was
                    % given:
                    
                    % get the area under the actual single exponential
                    % distribution:
                    singleExpContributions(idxExp)=expDecay*weights(idxExp);
                elseif isfield(exponentialParameters, 'initialValue')
                    % if the value at the left bounds of the interval was
                    % given:
                    
                    % remove the unnecessary fields if needed:
                    %exponentialParameters_act=rmfield(exponentialParameters_act,'initialValue');
                    
                    % get the value of the first bin of the normalized distribution:
                    sampledExponential=functionHandle(exponentialParameters_act, exponentialModelSettings, bins(1));
                    
                    % get the area under the actual single exponential
                    % distribution:
                    singleExpContributions(idxExp)=sampledExponential*weights(idxExp);
                    
                elseif isfield(exponentialParameters, 'sampleNumber')
                    % if the area within the interval was given:
                    
                    % remove the unnecessary fields if needed:
                    %exponentialParameters_act=rmfield(exponentialParameters_act,'sampleNumber');
                    
                    sampledExponential=functionHandle(exponentialParameters_act, exponentialModelSettings, bins);
                    singleExpContributions(idxExp)=sum(sampledExponential)*weights(idxExp);   % the initial value and the normalization factor (the area under the exponential should match that of the histogram)
                else
                    error('At least the "area" under the distribution or its "initial value" must be given.')
                end
            end
            
            % get the scaling (total area under the multi exponentials
            % without the background):
            if isfield(exponentialParameters, 'valueAtZero')
                % if the multiexponential function's value at 0 was
                % given:
                
                scaling=exponentialParameters.valueAtZero/sum(singleExpContributions);
            elseif isfield(exponentialParameters, 'initialValue')
                % if the value at the left bounds of the interval was

                scaling=exponentialParameters.initialValue/sum(singleExpContributions);
            elseif isfield(exponentialParameters, 'sampleNumber')
                % if the area within the interval was given:
                
                scaling=exponentialParameters.sampleNumber/sum(singleExpContributions);
            else
                error('At least the "sample number" (area  under the histogram), the exponential''s "value at zero", or its "initial value" must be given.')
            end
            
            % vector containing the areas of the single exponentials:
            area=scaling*weights;
        end
        
        
        function exponentialParameters = removeBackground(exponentialParameters, intervalLength)
            % This function removes the background's contribution for the
            % given parameters. It is used for getting the parameters of
            % the individual single exponentials from the multi-exponential
            % with background parameters.
            
            background=exponentialParameters.background;
            
            if isfield(exponentialParameters, 'valueAtZero')
                % if the multi exponential with background function's value at 0 was given:
                exponentialParameters.valueAtZero=exponentialParameters.valueAtZero-background;
            elseif isfield(exponentialParameters, 'initialValue')
                % if the multi exponential with background function's value at the first sampling bin was given:
                exponentialParameters.initialValue=exponentialParameters.initialValue-background;
            elseif isfield(exponentialParameters, 'sampleNumber')
                % if the number of sample (data) points were given (not the
                % number of bins but the number of data used to create the histogram):
                exponentialParameters.sampleNumber=exponentialParameters.sampleNumber-intervalLength*background;
            else

                error('At least the "sample number" (area  under the histogram), the exponential''s "value at zero", or its "initial value" must be given.')
            end
        end
        
    end
end

