function [fittedParameters, FVAL, EXITFLAG] = functionFitting(measuredData, functionToFitString, modelSettings, functionParameters, fittingParameterList, constraints, algorithmString, functionalToOptimizeString, minimizationFlag, options)
% This function performs a general function fitting on the given data. The
% optimization algorithm and the model function can be set with the
% function handle inputs. The initial values of the optimization iteration
% has to given in the model function parameters structure and has to be
% selected with the fitting parameters list cell array.

% INPUT VARIABLES
% - measurementData: the given data that the selected model function should be fitted on
% - functionToFitString: name of the model function that should bi fitted on the given data
% - functionParameters: structure of parameters of the model function, contains both the "static" parameters and the ones that should be varied with the iteration
% - fittingParameterList: cell array containing string, these strings select the field names from the "functionParameters" from which the initial parameters for the fitting should be constructed
% - functionalToOptimizeString: name of the functional created from the given data and model function that should be optimized with the iterative optimization algorithm, e.g. "mean squared error" or "likelihood"
% - minimizationFlag: "true" if the functional should be minimized, "false" if it should be maximized 
% - algorithmString: name of the selected optimization algorithm
% - constrains: constains of values that should be fit, only relevant if the optimization algorithm supports it, otherwise give empty value
% - options: options for the optimization algorithm


% options parameter structure for the fitting algorithm:
if isempty(options)
    options = optimset('MaxIter',1000, 'TolFun', eps, 'TolX', eps);
    %options = optimset();
else
    % do nothing
    % the user has already set them (properly, hopefully)
end

% function handle of the optmization algorithm:
%algorithmHandle=str2func(algorithmString);
% function handle of the model function:
functionToFitHandle=str2func(functionToFitString);
% function handle of the functional used for calculating the value to be
% optimized:
functionalToOptimizeHandle=str2func(functionalToOptimizeString);

% set the initial parameters vector for the fitting:
[initialParameters_vect, constaints_vect]=getInitialParametersVector(fittingParameterList, functionParameters, constraints);

% run the optimization (minimization) with the chosen algorithm:
switch algorithmString
    case 'fmincon'
        % constrains for the fitting:
        constraints_lower=constaints_vect(:, 1);
        constraints_upper=constaints_vect(:, 2);
        
        [fittedParamsVect, FVAL, EXITFLAG, ~]=fmincon(@(x) getValueToMinimize(x, measuredData, functionToFitHandle, fittingParameterList, functionParameters, modelSettings, functionalToOptimizeHandle, minimizationFlag), initialParameters_vect, [], [], [], [], constraints_lower, constraints_upper, [], options);
    case 'fminsearch'
        [fittedParamsVect, FVAL, EXITFLAG, ~]=fminsearch(@(x) getValueToMinimize(x, measuredData, functionToFitHandle, fittingParameterList, functionParameters, modelSettings, functionalToOptimizeHandle, minimizationFlag), initialParameters_vect, options);
    case 'fminunc'
        [fittedParamsVect, FVAL, EXITFLAG, ~]=fminunc(@(x) getValueToMinimize(x, measuredData, functionToFitHandle, fittingParameterList, functionParameters, modelSettings, functionalToOptimizeHandle, minimizationFlag), initialParameters_vect, options);
end

% update the model function parameters with the fitted ones:
fittedParameters=updateFunctionParameters(fittingParameterList, fittedParamsVect, functionParameters);
end


function valueToMinimize=getValueToMinimize(x, measuredData, functionToFitHandle, fittingParameterList, functionParameters, modelSettings, functionalToOptimizeHandle, minimizationFlag)
% This function calculates the value (mean squared error, -1*likelihood,
% etc..) that should be minimized with the iteration. This value is
% calculated from the given measured data and from the model function
% values calculated wih the actual function parameters .

% overwrite the given function parameters with the varied value of the
% iteration's actual step's values:
functionParameters=updateFunctionParameters(fittingParameterList, x, functionParameters);

% values of the model function:
[ sampledFunction ] = functionToFitHandle(measuredData.coordinates, functionParameters, modelSettings);

% get the value with the choosen functional that should be minimized:
valueToMinimize=double(functionalToOptimizeHandle(sampledFunction, measuredData.values, minimizationFlag));      % mean squared error as the residue
% the "fmincon requires the return value to be the type of double"

%% FOR TESTING:
% figure(55)
% dataValues=measuredData.values;
% dataCoordinates=measuredData.coordinates;
% if numel(dataCoordinates)==numel(dataValues)
%     samplingCoordinates=dataCoordinates;
%     plot(dataCoordinates, dataValues)
% elseif numel(dataCoordinates)==numel(dataValues)+1
%     samplingCoordinates=(dataCoordinates(1:end-1)+dataCoordinates(2:end))/2;
%     histogram('BinCounts', dataValues', 'BinEdges', dataCoordinates')
% else
%    error('Some problem with the data coordinates.') 
% end
% hold on;
% plot(samplingCoordinates, sampledFunction)
% hold off;
% valueToMinimize
end


function [initialParameters_vect, constaints_vect]=getInitialParametersVector(fittingParameterList, functionParameters, constraints)
% Tfis function creates the vector of the initial parameters for the
% iteration from the values of the right fields of the given function
% parameters. Also does the same for the parameters constraints.

% count the number of parameters for initialization:
numelParams=0;
for idxParamName=1:numel(fittingParameterList)
    numelParams=numelParams+numel(functionParameters.(fittingParameterList{idxParamName}));
end
initialParameters_vect=zeros(numelParams, 1);
constaints_vect=zeros(numelParams, 2);

% go trhough the parameters that should be fit and set the initial
% parameters vector for the fitting:
idxParam=0;
for idxParamName=1:numel(fittingParameterList)
    for idxActParamVect=1:numel(functionParameters.(fittingParameterList{idxParamName}))
        idxParam=idxParam+1;
        initialParameters_vect(idxParam)=functionParameters.(fittingParameterList{idxParamName})(idxActParamVect);
        
        constaints_vect(idxParam, :)=constraints.(fittingParameterList{idxParamName});
        % the "constains" structure should contain a single 1x2 size vector
        % for each parameter even if the "functionParameters" structure
        % contains several values (e.g. if the "funtionParameters.height"
        % contains values of two peaks, the "constains.height" still should
        % be a 1x2 element vector like in the single peak case (the two
        % peaks will have the same constaint) )
    end
end
end


function functionParameters=updateFunctionParameters(fittingParameterList, actualParamsVect, functionParameters)
% This function updates the model function parameters with the varied
% parameters of the iteration.

% go through the actual parameters of the iteration and update the function
% parameters with them:
idxParam=0;
for idxParamName=1:numel(fittingParameterList)
    for idxActParamVect=1:numel(functionParameters.(fittingParameterList{idxParamName}))
        idxParam=idxParam+1;
        functionParameters.(fittingParameterList{idxParamName})(idxActParamVect)=actualParamsVect(idxParam);
    end
end
end
