classdef kernelFunction_1D
    % These function calculates the weight contribution of data points to
    % given bins in 1 dimension. The weights are calculated by applying a
    % kernel function on the data points and then integrating the curve
    % ober the bin (either calculated with the integral function of the
    % kernel function or just approximating the integral with the product
    % of the smeared curve value at the pixel center and the bin size).
    % The supplied data points must fall within the support of the kernel.
    % These functions does not perform any check whether the the given data
    % point coordinates, kernel function sizes or the bin sizes/position
    % are in the same units, it is the user's task.
    % Either the data point or the bin parameters must be a scalar, the
    % other one can be a vector (There can be a single bin and several data
    % points, or a single data point and several bins but no several data
    % points and bins at once).
    
    methods (Static)
        function weights = Gaussian(dataPointCoord, kernelFunctionParameters, samplingType, binCenter, binSize)
            % This functinon calculates the weights of the data points that
            % they contribute to the given bins with a normalized Gaussian
            % kernel function. The result of convolution of the data points
            % with the Gaussian kernel is sampled at the bin centers or
            % integrated over the bins.
            
            % size (STD) of the kernel function (converted from the FWHM)
            sigma=kernelFunctionParameters.size.value/(2*sqrt(2*log(2)));
            
            switch samplingType
                case 'pixel central value'
                    % sample the curve of the smeared data points at the
                    % bin centers
                    
                    % calculate the weigths of the data points that they
                    % contribute to the bin:
                    weights=1/(sqrt(2*pi)*sigma)*...
                        exp(-(binCenter-dataPointCoord).^2/(2*sigma^2))*binSize;
                case 'integral over pixels'
                    % integrate the curve of the smeared data points over
                    % the bins
                    
                    % bounds of the integration
                    upperBound=binCenter+binSize*0.5;
                    lowerBound=binCenter-binSize*0.5;
                    
                    % calculate the weigths of the data points that they
                    % contribute to the bin:
                    weights=0.5*(erf((upperBound-dataPointCoord)/(sqrt(2)*sigma))-erf((lowerBound-dataPointCoord)/(sqrt(2)*sigma)));
                otherwise
                    error('Check the kernel settings. Not proper string was given for the sampling.')
            end
        end
        
        
        function weights = rectangular(dataPointCoord, kernelFunctionParameters, samplingType, binCenter, binSize)
            % This functinon calculates the weights of the data points that
            % they contribute to the given bins with a normalized
            % rectangular kernel function. The result of convolution of the
            % data points with the rectangular kernel is sampled at the bin
            % centers or integrated over the bins.
            
            if isempty(dataPointCoord) || isempty(binCenter)
                weights=[];
            else
                
                % initialize the weights vector with thew right dimension
                % it has to be done otherwise it will be a row vector (and
                % the "dataPointCoord" is a column vector...)
                weights=zeros(size(dataPointCoord));
			
                % size of the kernel function (full length of the
                % rectangle, also the FWHM):
                rectangleSize=kernelFunctionParameters.size.value;
                
                switch samplingType
                    case 'pixel central value'
                        % sample the curve of the smeared data points at the
                        % bin centers
                        
                        % check which data points lies within the bins:
                        withinBool=dataPointCoord<(binCenter+rectangleSize/2) & dataPointCoord>(binCenter-rectangleSize/2);
                        
                        % check which data points lies on the bin bounds:
                        onBoundBool=dataPointCoord==(binCenter+rectangleSize/2) | dataPointCoord==(binCenter-rectangleSize/2);
                        
                        
                        % calculate the weigths of the data points that they
                        % contribute to the bin:
                        weights=zeros(size(withinBool));
                        weights(withinBool)=1/rectangleSize*binSize;
                        weights(onBoundBool)=1/(2*rectangleSize)*binSize;
                    case 'integral over pixels'
                        % integrate the curve of the smeared data points over
                        % the bins
                        
                        % calculate the properties of the isosceles trapezium
                        % resultant of the concolution of the bin and the
                        % rectangular kernel function:
                        if rectangleSize<binSize
                            slopeCoord_lower=binCenter-binSize/2-rectangleSize/2;
                            plateauCoord_lower=binCenter-binSize/2+rectangleSize/2;
                            plateauCoord_upper=binCenter+binSize/2-rectangleSize/2;
                            slopeCoord_upper=binCenter+binSize/2+rectangleSize/2;
                            plateauHeight=1;
                            slope=plateauHeight/(plateauCoord_lower-slopeCoord_lower);
                        else
                            slopeCoord_lower=binCenter-rectangleSize/2-binSize/2;
                            plateauCoord_lower=binCenter-rectangleSize/2+binSize/2;
                            plateauCoord_upper=binCenter+rectangleSize/2-binSize/2;
                            slopeCoord_upper=binCenter+rectangleSize/2+binSize/2;
                            plateauHeight=binSize/rectangleSize;
                            slope=plateauHeight./(plateauCoord_lower-slopeCoord_lower);
                        end
                        
                        % data point(s) falling on the "left" slope of the trapezium:
                        increasingSloopBool=slopeCoord_lower<=dataPointCoord & dataPointCoord<plateauCoord_lower;
                        % data point(s) falling on the plateau of the trapezium:
                        plateauBool=plateauCoord_lower<=dataPointCoord & dataPointCoord<plateauCoord_upper;
                        % data point(s) falling on the "right" slope of the trapezium:
                        decrasingSloopBool=plateauCoord_upper<=dataPointCoord & dataPointCoord<=slopeCoord_upper;
                        
                        % calculate the weigths of the data points that they
                        % contribute to the bin:
                        if numel(binCenter)>1
                            % if there is one data point and several bins:
                            weights(increasingSloopBool)=(dataPointCoord-slopeCoord_lower(increasingSloopBool)).*slope(increasingSloopBool);
                            weights(plateauBool)=plateauHeight;
                            weights(decrasingSloopBool)=(slopeCoord_upper-dataPointCoord(decrasingSloopBool)).*slope(decrasingSloopBool);
                        else
                            % if there is one bin and several data points:
                            weights(increasingSloopBool)=(dataPointCoord(increasingSloopBool)-slopeCoord_lower)*slope;
                            weights(plateauBool)=plateauHeight;
                            weights(decrasingSloopBool)=(slopeCoord_upper-dataPointCoord(decrasingSloopBool))*slope;
                        end
                        
                    otherwise
                        error('Check the kernel settings. Not proper string was given for the sampling.')
                end
            end
        end
    end
end

