classdef kernelFunction_2D
    % These function calculates the weight contribution of data points to
    % given bins in 2 dimensions. The weights are calculated by applying a
    % kernel function on the data points and then integrating the curve
    % ober the bin (either calculated with the integral function of the
    % kernel function or just approximating the integral with the product
    % of the smeared curve value at the pixel center and the bin size).
    % The supplied data points must fall within the support of the kernel.
    % These functions does not perform any check whether the the given data
    % point coordinates, kernel function sizes or the bin sizes/position
    % are in the same units, it is the user's task.
    % Either the data point or the bin parameters must be a scalar, the
    % other one can be a vector (There can be a single bin and several data
    % points, or a single data point and several bins but no several data
    % points and bins at once).
    
    methods (Static)
        function weights = Gaussian(dataPointCoord, kernelFunctionParameters, samplingType, binCenter, binSize)
            % This functinon calculates the weights of the data points that
            % they contribute to the given bins with a normalized Gaussian
            % kernel function. The result of convolution of the data points
            % with the Gaussian kernel is sampled at the bin centers or
            % integrated over the bins.
            
            % calculate the "x" and "y" components on the weights:
            kernelFunctionSettigns_x.size.value=kernelFunctionParameters.size.x;
            weights_x=kernelFunction_1D.Gaussian(dataPointCoord.x, kernelFunctionSettigns_x, samplingType, binCenter.x, binSize.x);
            kernelFunctionSettigns_y.size.value=kernelFunctionParameters.size.y;
            weights_y=kernelFunction_1D.Gaussian(dataPointCoord.y, kernelFunctionSettigns_y, samplingType, binCenter.y, binSize.y);
            
            % the weights are the product of the "x" and "y" components:
            weights=weights_x.*weights_y;
            
        end
        
        function weights = rectangular(dataPointCoord, kernelFunctionParameters, samplingType, binCenter, binSize)
            % This functinon calculates the weights of the data points that
            % they contribute to the given bins with a normalized
            % rectangular kernel function. The result of convolution of the
            % data points with the rectangular kernel is sampled at the bin
            % centers or integrated over the bins.
            
            % calculate the "x" and "y" components on the weights:
            kernelFunctionSettigns_x.size.value=kernelFunctionParameters.size.x;
            weights_x=kernelFunction_1D.rectangular(dataPointCoord.x, kernelFunctionSettigns_x, samplingType, binCenter.x, binSize.x);
            kernelFunctionSettigns_y.size.value=kernelFunctionParameters.size.y;
            weights_y=kernelFunction_1D.rectangular(dataPointCoord.y, kernelFunctionSettigns_y, samplingType, binCenter.y, binSize.y);
            
            % the weights are the product of the "x" and "y" components:
            weights=weights_x.*weights_y;
        end
        
        function weights = disk(dataPointCoord, kernelFunctionParameters, samplingType, binCenter, binSize)
            % This functinon calculates the weights of the data points that
            % they contribute to the given bins with a normalized 
            % elliptical disk shaped kernel function. The result of the
            % convolution of data points with the disk shaped kernel is
            % sampled at the bin centers or integrated over the bins.
            
            if isempty(dataPointCoord) || isempty(binCenter)
            % id there is no data point or bin center in the current 
                weights=[];
            else
                
                % initialize the weights vector with thew right dimension
                % it has to be done otherwise it will be a row vector (and
                % the "dataPointCoord" is a column vector...)
                weights=zeros(size(dataPointCoord));
                
                % size of the kernel function i the two direction (the
                % major and minor axes of the ellipse):
                diskAxis_x=kernelFunctionParameters.size.x/2;
                diskAxis_y=kernelFunctionParameters.size.y/2;
                
                % area of the kernel function
                diskArea=diskAxis_x*diskAxis_y*pi;
                
                switch samplingType
                    case 'pixel central value'
                        % sample the result of the convolution at the
                        % bin centers
                        
                        % check which data points lies within the bins:
                        withinBool=(dataPointCoord.x-binCenter.x).^2/diskAxis_x^2+(dataPointCoord.y-binCenter.y).^2/diskAxis_y^2<1;
                        
                        % check which data points lies on the bin bounds:
                        onBoundBool=(dataPointCoord.x-binCenter.x).^2/diskAxis_x^2+(dataPointCoord.y-binCenter.y).^2/diskAxis_y^2==1;
                        
                        
                        % calculate the weigths of the data points that they
                        % contribute to the bin:
                        weights=zeros(size(withinBool));
                        weights(withinBool)=(binSize.x*binSize.y)/diskArea;
                        weights(onBoundBool)=0.5*(binSize.x*binSize.y)/diskArea;
                    case 'integral over pixels'
                        % Intergrate the result of the convolution over the
                        % bins, calculates the disk shaped kernel's and the
                        % bin's intersection area.
                        % It scales the y dimension so the kernel function
                        % will be a circle, it simplifies the calculation.
                        
                        % the scaling factor of the "y" dimension:
                        scaling_y=diskAxis_x/diskAxis_y;
                        
                        % radius of the circle
                        diskRadius=diskAxis_x;
                        
                        % "x" boundaries of the bin
                        binBoundX_left=binCenter.x-binSize.x/2;
                        binBoundX_right=binCenter.x+binSize.x/2;
                        
                        % "y" boundaries of the bin
                        binBoundY_lower=(binCenter.y-binSize.y/2)*scaling_y;
                        binBoundY_upper=(binCenter.y+binSize.y/2)*scaling_y;
                        
                        for idxData=1:numel(dataPointCoord.x)
                        % Go through the data points and check whether data
                        % data point lies above te bin, under the bin or
                        % within the bin's horizontal strip. It is
                        % required, beacuse it is easy to calculate the
                        % intersection if the rectangular bin lies fully in
                        % the upper halo of the disk (there is formula
                        % to integrate a half disk).
                        
                            % center of the kernel:
                            diskCenterX=dataPointCoord.x(idxData);
                            diskCenterY=dataPointCoord.y(idxData)*scaling_y;
                            
                            if diskCenterY<=binBoundY_lower
                                % if the data point lies under the bin
                                
                                intersectionArea = kernelFunction_2D.diskIntersection_upper(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower, binBoundY_upper);
                                
                            elseif diskCenterY>=binBoundY_upper
                                % if the data point lies above the bin
                                
                                intersectionArea = kernelFunction_2D.diskIntersection_lower(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower, binBoundY_upper);
                                
                            elseif diskCenterY>binBoundY_lower && diskCenterY<binBoundY_upper
                                % if the data point lies within the bin's horizontal strip
                                % Seperate the problem into two sipler one.
                                % Divide the bin into two part, so one part
                                % will lie above the horizontal line going
                                % through the disk center, the other part
                                % eill lie fully under it.
                                
                                %  the "upper" part of the divided bin:
                                binBoundY_lower1=diskCenterY;
                                intersectionArea_1 = kernelFunction_2D.diskIntersection_upper(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower1, binBoundY_upper);
                                
                                %  the "lower" part of the divided bin:
                                binBoundY_upper2=diskCenterY;
                                intersectionArea_2 = kernelFunction_2D.diskIntersection_lower(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower, binBoundY_upper2);
                                
                                intersectionArea=intersectionArea_1+intersectionArea_2;
                                
                            else
                                error('Some problem with the disk kernel function and bin pozition. This condition should never have been met. Check it!')
                            end
                            
                            % To determine the weights of the data points,
                            % divide the intersection area with the area of
                            % the disk:
                            weights(idxData)=intersectionArea/diskArea/scaling_y;
                            
                        end
                    otherwise
                        error('Check the kernel settings. Not proper string was given for the sampling.')
                end
            end
            
        end
    end
    
    
    methods(Static, Access=private)
    
        function [leftIntersection, rightIntersection] = circle_semiInfiniteStrip_intersectionVerticalBound(radius, centerX, centerY, boundX_left, boundX_right, boundY)
            % This function calculates the horizontal ("x") coordinates of
            % the vertical bound of the intersection of a semi-infinite
            % strip and a circle. The semi infine strip has two vertical
            % "x" bounds and a single horizontal "y" bound. Always two
            % points are returned even if they only intersect at a single
            % point ar do not intersect at all.
            
            if abs(boundY-centerY)>radius
                % no intersection:
                leftIntersection=NaN;
                rightIntersection=NaN;
            elseif abs(boundY-centerY)==radius
                % intersection oat one point only:
                leftIntersection=centerX;
                rightIntersection=centerX;
            elseif abs(boundY-centerY)<radius
                % "proper" intersection:
                a=sqrt(radius^2-(boundY-centerY)^2);
                leftIntersection=centerX-a;
                rightIntersection=centerX+a;
            else
                error('some problem')
            end
            
            % determine where the rectangle and the disk intersect on the right and left sides
            if ~isnan(leftIntersection)
                leftIntersection=max(leftIntersection, boundX_left);
            end
            if ~isnan(rightIntersection)
                rightIntersection=min(rightIntersection, boundX_right);
            end
            
            % % if the result is unmeaningful, i.e. the bin is too far
            % aside to have intersection
            if leftIntersection>rightIntersection
                leftIntersection=NaN;
                rightIntersection=NaN;
            end
            
        end
        
        
        function intersectionArea = circle_semiInfiniteStrip_intersectionArea(diskRadius, diskCenterX, diskCenterY, boundX_left, boundX_right, boundY)
            % This function calculates the intersection area a
            % semi-infinite strip and a disk. The semi infine strip has two
            % vertical "x" bounds and a single horizontal "y" lower bound.
            % The "y" bound must be above the center of the disk to get a
            % proper a result.
            
            % the left on right bound of the intersection of the disk and
            % the bin:
            [leftIntersection, rightIntersection] = kernelFunction_2D.circle_semiInfiniteStrip_intersectionVerticalBound(diskRadius, diskCenterX, diskCenterY, boundX_left, boundX_right, boundY);
            
            
            if isnan(leftIntersection) || isnan(rightIntersection)
                % there is no intersection:
                intersectionArea=0;
            else
                % calculate the area of the intersection:
                integralUpperBound = rightIntersection-diskCenterX;
                integralLowerBound = leftIntersection-diskCenterX;
                
                % integral of the half disk within the bounds:
                halfDiskIntegral = kernelFunction_2D.halfDiskPrimitiveIntegral(diskRadius, integralUpperBound)-kernelFunction_2D.halfDiskPrimitiveIntegral(diskRadius, integralLowerBound);
                
                % correct the intersection area as the bin's lower bound is
                % usually above the horizontal line going through the bin's
                % center (subtract the corresponding rectangle):
                intersectionArea = halfDiskIntegral-(integralUpperBound-integralLowerBound)*(boundY-diskCenterY);
            end
            
        end
        
        function intersectionArea = diskIntersection_upper(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower, binBoundY_upper)
            % This function returns the intersection area of a circle and a
            % rectangular bin when the bin is fully in the upper half of
            % the circle (the lower bound of the bin must be above the
            % center of the circle).
            
            if binBoundY_lower<diskCenterY
                error('There is problem with the semi-infinite strip and disk intersection area calculation.');
            end
            
            % area of the intersection if the bin has no upper bound (intersection of the disk with an upright semi-infinite strip with the same width and lower bound as the bin):
            intersectionArea_1 = kernelFunction_2D.circle_semiInfiniteStrip_intersectionArea(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower);
            
            % the leftover area above the bin's upper bound (intersection of the disk with an upright semi-infinite strip with the same width as the bin and its lower bound is the bin's upper bound):
            intersectionArea_2 = kernelFunction_2D.circle_semiInfiniteStrip_intersectionArea(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_upper);
            
            intersectionArea = intersectionArea_1-intersectionArea_2;
            
        end
        
        
        function intersectionArea = diskIntersection_lower(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower, binBoundY_upper)
            % This function returns the intersection area of a circle and a
            % rectangular bin when the bin is fully in the lower half of
            % the circle (the lower bound of the bin must be under the
            % center of the circle).
            
            if binBoundY_upper>diskCenterY
                error('There is problem with the ssemi-infinite strip and disk intersection area calculation.')
            end
            
            % reflect the bin on the vertical line going through the disk's
            % center, so the bin will be fully in the upper half of the
            % disk:
            binBoundY_upper_temp = diskCenterY+(diskCenterY-binBoundY_lower);
            binBoundY_lower_temp = diskCenterY+(diskCenterY-binBoundY_upper);
            
            % get the intersection area of the bin in the upper half of the
            % disk:
            intersectionArea = kernelFunction_2D.diskIntersection_upper(diskRadius, diskCenterX, diskCenterY, binBoundX_left, binBoundX_right, binBoundY_lower_temp, binBoundY_upper_temp);
            
        end
        
        
        function integral= halfDiskPrimitiveIntegral(radius, x)
            % this function calculates the indefinite integral of a half disk.
            % source: https://www.numberempire.com/integralcalculator.php?function=sqrt(r^2-x^2)&var=x
            
            % The primitive integral of a half disk:
            integral=(radius^2*asin(x/abs(radius))+x*sqrt(radius^2-x^2))/2;
        end
        
    
    end
end

