classdef kernelOperations
    % These methods calculate the given data point weights over the given
    % bins with some kernel function and performs some mathemathical
    % operations on the weighted data points.
    
    methods (Static)
        
        function kernelValue = simpleNumel(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function simply returns the numbers of data points
            % within the kernel support range without any weighting.
            
            if issctuct(dataPointCoord)
                % in case of 2 dimensional input:
                kernelValue=numel(dataPointCoord.x_coord);
            else
                % in case of 1 dimensional input:
                kernelValue=numel(dataPointCoord);
            end
        end
        
        function kernelValue = counts(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the counts (the integrated smeared
            % data point density) on the bins (the weighted number of data
            % points wothin the kernel support).
            
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, kernelSettings.functionParameters, kernelSettings.samplingType, binCenter, binSize);
            
            % integrated data point density on the bins:
            kernelValue=sum(weights);
            
        end
        
        function  kernelValue = density(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the smeared data point density on the bins.
            
            % counts within the bins
            counts=kernelOperations.counts(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
            
            % data point density on the bins:
            kernelValue=counts/(binSize.x*binSize.y);
            
        end
        
        
        function  kernelValue = sum(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the kernel weighted sum of the data point
            % values on the bins.
            
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, kernelSettings.functionParameters, kernelSettings.samplingType, binCenter, binSize);
            
            % the weighted sum:
            kernelValue=sum(weights.*dataValue);
            
        end
        
        
        function  kernelValue = mean(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the kernel weighted mean of the data point
            % values on the bins.
            
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, kernelSettings.functionParameters, kernelSettings.samplingType, binCenter, binSize);
            
            % the weighted mean:
            kernelValue=sum(weights.*dataValue)/sum(weights);
            
        end
        
        
        function  kernelValue = STD(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the kernel weighted standard deviation of the data point
            % values on the bins (from the weighted variance).
            
            % weights of the data points on the bins:
            variance=kernelOperations.weightedVariance(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
            
            % the standard deviation:
            kernelValue=sqrt(variance);
        end
        
        
        
        function  kernelValue = weightedVariance(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the kernel weighted variance of the data point
            % values on the bins
            
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
            
            % weighted mean of the data point values:
            weightedMean = sum(weights.*dataValue)/sum(weights);
            
            % the weighted variance:
            kernelValue=sum(weights.*(dataValue-weightedMean).^2)/sum(weights);
        end
        
        function kernelValue = weightedSampleVariance(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function calculates the kernel weighted variance of the data point
            % values on the bins
            
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
            
            V1=sum(weights);
            V2=sum(weights.^2);
            
            % weighted mean of the data point values:
            weightedMean = sum(weights.*dataValue)/sum(weights);
            
            % the weighted variance:
            kernelValue=sum(weights.*(dataValue-weightedMean).^2)/V1/(1-V2/V1^2);
        end
        
        function kernelValue = standardError(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
        
            % weights of the data points on the bins:
            weights=kernelSettings.functionHandle(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
        
			sampleVariance = kernelOperations.weightedSampleVariance(dataPointCoord, dataValue, kernelSettings, binCenter, binSize);
			
			normalizedWeights=weights/sum(weights);
			
			kernelValue = sampleVariance*sqrt(sum(normalizedWeights.^2));
			
        end
        
        function kernelValue = median(dataPointCoord, dataValue, kernelSettings, binCenter, binSize)
            % This function returns the meadian of data points within the
            % kernel support.
            
            % TODO: consider implementing the wieghted median
            
            kernelValue=median(dataValue);
            
        end
        
    end
    
end

