classdef kernelAffectedRegions
    % These methods return the bounds of regions that are affected by the
    % kernel for every pixel or bin used for sampling. These bounds are
    % calculated from the pixel/bin positions and sized and from the kernel
    % support size.
    

    
    methods (Static)
        function [xBounds, yBounds]=getRegionBounds_2D(ROI_bounds, imagePixelSize, supportSize, samplingType)
            % This function returns the bounds of the rectangular regions
            % belonging to the individual image pixels. The regions are
            % determined by the image pixel and kernel suppot size (only
            % within these regions the data points contribute to the values
            % of the image pixels).
            
            
            % region of interest of the evaluation:
            ROI_bounds_x=ROI_bounds.x;
            ROI_bounds_y=ROI_bounds.y;
            
            % spacing of the sampling points on the to be created image:
            imagePixelSize_x=imagePixelSize.x;
            imagePixelSize_y=imagePixelSize.y;
            
            
            % support size for the kernel function:
            supportSize_x=supportSize.x;
            supportSize_y=supportSize.y;

            
            % get the "x" and "y" bounds:
            xBounds=kernelAffectedRegions.getRegionBounds_1D(ROI_bounds_x, imagePixelSize_x, supportSize_x, samplingType);
            yBounds=kernelAffectedRegions.getRegionBounds_1D(ROI_bounds_y, imagePixelSize_y, supportSize_y, samplingType);
        end
        
        
        
        function [xBounds, yBounds]=getPixelBounds(ROI_bounds, pixelSize)
            % This function returns the bounds of the image pixels.
            
            [xBounds, yBounds]=kernelAffectedRegions.getRegionBounds_2D(ROI_bounds, pixelSize, pixelSize, 'pixel central value');
            
        end
        
        
        function bounds=getRegionBounds_1D(ROI_bounds, imagePixelSize, supportSize, samplingType)
            % This function calculates the region bounds from their centers
            % and sizes.
            
            % spacing of the centers of the image pixels:
            regionSpacing=imagePixelSize;
            
           switch samplingType
                case 'integral over pixels'
                    % regions belonging to the image pixels:
                    regionSize=imagePixelSize+supportSize;
               case 'pixel central value'
                   % regions belonging to the image pixels:
                   regionSize=supportSize;
               otherwise
                   error('Check the kernel settings. Not proper string was given for the sampling.')
            end
            
            % number of regions inside the ROI:
            N=floor((abs(ROI_bounds(2)-ROI_bounds(1))-regionSpacing)/regionSpacing+1);
            
            regionCenters=(ROI_bounds(1)+ROI_bounds(2))/2+((0:N-1)-(N-1)/2)*regionSpacing;
            
            % lower coordinates of the region:
            bounds(:, 1)=regionCenters-regionSize*0.5;
            % upper coordinates of the region:
            bounds(:, 2)=regionCenters+regionSize*0.5;
        end
    end
end

