classdef pointillisticData_kernelOnBins
    % These methods apply the selected kernel function (it determines the
    % data point weights) and operation (kernel weighted summation, mean,
    % variance...) on the data points and create a histogram out of this.
    % Instead of going through the data points and adding their
    % contribution to the histogram bins, these methods go through the
    % bins, find which data contribute to the selected bin and
    % performs the opertations only these data points.
    % The ROI is divided so that the largest possible integer number of
    % fix sized histogram bins fit in it. Because of the non-zero support
    % size, the marginal bins might contain inconsistent information
    % compared to the intermediate bins. The histogram is centered at the ROI
    % center.
    % For performace reasons, every kernel has a finite support size (e.g.
    % the Gaussian is cropped) so only not every data point contribute to
    % every histogram bin.
    % It is like performing the convolution with the kernel function on the
    % selected histogram bin and collecting the data points that fall within
    % the non-zero regions.
    
    
    methods (Static)
        function [projectedProfile, binCenters] = getProjectedProfile(pointillisticData, coordinateFieldname, binningSettings, kernelSettings, acquisitionParameters)
            % This function calculates the bin values out of the kernel
            % weighted data points with the given operation. For each bin
            % only thoose data points are selected that affect that bin's
            % value (the kernel support is finite).
            % For efficiency, the data points are sorted in ascending order
            % regarding their coordiantes and then it is fast and
            % straightforward to collect those data point that fall within
            % the given bin's affected region.
            
            % convert the ROIs and region sizes to camera pixel units:
            ROI_bounds=unitConversion.convertScalarQuantity(binningSettings.ROI_bounds, 'camera pixel length', acquisitionParameters);
            profileBinSize=unitConversion.convertScalarQuantity(binningSettings.binSize, 'camera pixel length', acquisitionParameters);
            kernelSupportSize=unitConversion.convertScalarQuantity(kernelSettings.supportSize, 'camera pixel length', acquisitionParameters);
            kernelSettings.functionParameters.size=unitConversion.convertScalarQuantity(kernelSettings.functionParameters.size, 'camera pixel length', acquisitionParameters);
            
            % cut out the required fields and omit the NaNs:
            [dataCoords, dataValues]=pointillisticData_kernelOnBins.returnRequiredData(pointillisticData, coordinateFieldname, kernelSettings.dataFieldName);
            
            % get the bounds of the affected regions of the bins:
            [bounds]=kernelAffectedRegions.getRegionBounds_1D(ROI_bounds.value, profileBinSize.value, kernelSupportSize.value, kernelSettings.samplingType);
            
            % number of bins:
            binN=size(bounds, 1);
            projectedProfile=single(zeros(binN,1));
            
            % sorting the coordiantes:
            [sortedData, sortingOrder]=sort(dataCoords);
            
            % function handle of the operation that should b performed on
            % the data points:
            operationHandle=str2func(kernelSettings.operationName);
            kernelSettings.functionHandle=str2func(kernelSettings.functionName);
            
            % lower and upper indices of the sorted coordiantes of that
            % fall between the given bounds: 
            indexRanges=sortedPointillisticData_divide.getIndexRanges_seperateRegions(sortedData, bounds);
            for idx=1:binN
                % coordinates that fall between the current bounds:
                originalDataIndices_cut=sortingOrder(indexRanges(idx,1):indexRanges(idx,2));
                
                if ~isempty(originalDataIndices_cut)
                    
                    
                    % cut tha data points affected by the kernelized bin:
                    dataCoords_cut=dataCoords(originalDataIndices_cut);
                    if numel(dataCoords)==numel(dataValues)
                        dataValues_cut=dataValues(originalDataIndices_cut);
                    else
                        dataValues_cut=dataValues;
                    end
                    
                    binCenter=(bounds(idx, 1)+bounds(idx, 2))/2;
                    % get the bin value from the data points with kernel function applied on them:
                    projectedProfile(idx)=operationHandle(dataCoords_cut, dataValues_cut, kernelSettings, binCenter, profileBinSize.value);
                    
                else
                    % do nothing
                    % then let it be the initialized value
                end

            end
            
            % return the centers of the pixels too:
            binCenters.value=(bounds(:, 1)+bounds(:, 2))/2;
            binCenters.unit='camera pixel length';
            
        end
        
        
        function [dataCoords, dataValues]=returnRequiredData(pointillisticData, coordinateFieldname, dataFieldName)
            % This function return the data (coordiantes and the selected
            % value field) on which the operations will be performed.
            
            if ~strcmp(dataFieldName, 'data number')
                % find the non NaN data values:
                nonNaNBool=~isnan(pointillisticData.(dataFieldName));
                dataCoords=pointillisticData.(coordinateFieldname)(nonNaNBool);
                dataValues=pointillisticData.(dataFieldName)(nonNaNBool);
            else
                % if no data field is specified, accept all the data:
                dataCoords=pointillisticData.(coordinateFieldname);
                dataValues=1;
            end
        end
        
    end
end

