classdef pointillisticData_kernelOnImagePixels
    % These methods apply the selected kernel function (it determines the
    % data point weights) and operation (kernel weighted summation, mean,
    % variance...) on the data points and create an image out of this.
    % Instead of going through the data points and adding their
    % contribution to the image pixels, these methods go through the image
    % pixels, find which data contribute to the selected pixel and
    % performs the opertations only these data points.
    % The ROI is divided so that the largest possible integer number of
    % fix sized image pixels fit in it. Because of the non-zero support
    % size, the marginal fixels might contain inconsistent information
    % compared to the intermediate pixels. The image is centered at the ROI
    % center.
    % For performace reasons, every kernel has a finite support size (e.g.
    % the Gaussian is cropped) so only not every data point contribute to
    % every image pixel.
    % It is like performing the convolution with the kernel function on the
    % selected image pixel and collecting the data points that fall within
    % the non-zero regions.
    
    
    methods (Static)
        
        
        function [imageMatrix, pixelCenters] = getImageMatrixViaSorting(pointillisticData, pixelizationSettings, kernelSettings, cameraSignalCOnversion)
            % This function calculates the pixel values out of the kernel
            % weighted data points with the given operation. For each pixel
            % only thoose data points are selected that affect that pixel's
            % value (the kernel support is finite).
            % For efficiency, the data points are sorted in ascending order
            % regarding their coordiantes and then it is fast and
            % straightforward to collect those data point that fall within
            % the given pixel's affected region. The sorting is first
            % performed on the "x" then on the "y" coordinates.
            
            % function handle of the operation that should be performed on
            % the data points:
            operationHandle=str2func(kernelSettings.operationName);
            kernelSettings.functionHandle=str2func(kernelSettings.functionName);
            
            % convert the ROIs and region sizes to camera pixel units:
            ROI_bounds=unitConversion.convert2DVectorQuantity(pixelizationSettings.ROI_bounds, 'camera pixel length', cameraSignalCOnversion);
            pixelSize=unitConversion.convert2DVectorQuantity(pixelizationSettings.pixelSize, 'camera pixel length', cameraSignalCOnversion);
            kernelSupportSize=unitConversion.convert2DVectorQuantity(kernelSettings.supportSize, 'camera pixel length', cameraSignalCOnversion);
            kernelSettings.functionParameters.size=unitConversion.convert2DVectorQuantity(kernelSettings.functionParameters.size, 'camera pixel length', cameraSignalCOnversion);
            samplingType = kernelSettings.samplingType;
            
            binSize.x=pixelSize.x;
            binSize.y=pixelSize.y;
            
            % cut out the required fields and omit the NaNs:
            [dataCoords, dataValues]=pointillisticData_kernelOnImagePixels.returnRequiredData(pointillisticData, kernelSettings.dataFieldName);
            
            % get the pixel bounds on sorted order
            [xBounds_xSorted, yBounds_ySorted]=kernelAffectedRegions.getRegionBounds_2D(ROI_bounds, pixelSize, kernelSupportSize, samplingType);

            % number of image pixels in the rows and columns
            N_x=size(xBounds_xSorted, 1);
            N_y=size(yBounds_ySorted, 1);
            
            % go through the pixels of the image
            imageMatrix=zeros(N_x, N_y);
            
            % sorting the "x" coordinates:
            [sortedData_x, sortingOrder_x]=sort(pointillisticData.x_coord);

            % sorting the "y" coordinates after sorting them in "x"
            % ascending order:
            y_coord_xSorted=pointillisticData.y_coord(sortingOrder_x);
            [sortedData_y, sortingOrder_y]=sort(y_coord_xSorted);

            lowerIndex_starting = 1;
            indexRanges_xSorted = sortedPointillisticData_divide.slideBoundaryIndex(sortedData_x, xBounds_xSorted, lowerIndex_starting);
            indexRanges_ySorted = sortedPointillisticData_divide.slideBoundaryIndex(sortedData_y, yBounds_ySorted, lowerIndex_starting);
            
            for idx_x=1:N_x
                for idx_y=1:N_y
                    
                    % lower and upper indices of the "x" sorted coordiantes of that
                    % fall between the given "x" bounds:
                    indexRanges_xDivision=indexRanges_xSorted(idx_x,:);

                    % lower and upper indices of the "y" sorted coordiantes of that
                    % fall between the given "y" bounds:
                    indexRanges_yDivision=indexRanges_ySorted(idx_y,:);

                    xSortedIndices_yCut = sortingOrder_y(indexRanges_yDivision(1):indexRanges_yDivision(2));

                    xSortedIndices_xyCut = xSortedIndices_yCut(xSortedIndices_yCut>=indexRanges_xDivision(1) & xSortedIndices_yCut<=indexRanges_xDivision(2));
                   
                    % the indices of the data points falling in the
                    % affected zone of the currect pixel:
                    originalDataIndices_xyCut=sortingOrder_x(xSortedIndices_xyCut);
                    
                    if ~isempty(originalDataIndices_xyCut)

                        
                        % cut tha data points affected by the kernelized pixel:
                        %[dataCoords_x, dataCoords_y, dataValues]=pointillisticData_kernelOnImagePixels.cutPointillisticData(pointillisticData, originalDataIndices_xyCut, kernelSettings.dataFieldName);
                        dataCoords_cut.x=dataCoords.x(originalDataIndices_xyCut);
                        dataCoords_cut.y=dataCoords.y(originalDataIndices_xyCut);
                        if numel(dataCoords.x)==numel(dataValues)
                            dataValues_cut=dataValues(originalDataIndices_xyCut);
                        else
                            dataValues_cut=dataValues;
                        end
                        
                        binCenter.x=(xBounds_xSorted(idx_x, 1)+xBounds_xSorted(idx_x, 2))/2;
                        binCenter.y=(yBounds_ySorted(idx_y, 1)+yBounds_ySorted(idx_y, 2))/2;
                        % get the pixel value from the data points with kernel function applied on them:
                        imageMatrix(idx_x, idx_y)=operationHandle(dataCoords_cut, dataValues_cut, kernelSettings, binCenter, binSize);
                        
                    else
                        % do nothing
                        % then let it be the initialized value
                    end
                    
                end
                
                % return the centers of the pixels too:
                pixelCenters.x=(xBounds_xSorted(:, 1)+xBounds_xSorted(:, 2))/2;
                pixelCenters.y=(yBounds_ySorted(:, 1)+yBounds_ySorted(:, 2))/2;
                pixelCenters.unit='camera pixel length';
            end
        end
        
        function [dataCoords, dataValues]=cutPointillisticData(pointillisticData, originalDataIndices_xyCut, dataFieldName)
            % This function cuts tha datas (coordinates and a given field)
            % that fall within the range (support) of the kernel applied on the
            % pixel.
            
            
            if ~strcmp(dataFieldName, 'data number')
                % find the NaN data values:
                isnanBool=isnan(pointillisticData.(dataFieldName)(originalDataIndices_xyCut));
            else
                % if no data field is specified, accept all the data:
                isnanBool=false(size(originalDataIndices_xyCut));
            end
            
            % drop the datas that are NaN:
            originalDataIndices_xyCut(isnanBool)=[];
            
            % cut the datas that are affected by the kernel around the
            % pixel:
            dataCoords.x=pointillisticData.x_coord(originalDataIndices_xyCut);
            dataCoords.y=pointillisticData.y_coord(originalDataIndices_xyCut);
            if strcmp(dataFieldName, 'data number')
                % return empty value if no data field is specified:
                 dataValues=[];
            else
                dataValues=pointillisticData.(dataFieldName)(originalDataIndices_xyCut);
            end
            
        end
        
        
        function [dataCoords, dataValues]=returnRequiredData(pointillisticData, dataFieldName)
            % This function return the data (coordiantes and the selected
            % value field) on which the operations will be performed.
            
            if ~strcmp(dataFieldName, 'data number')
                % find the non NaN data values:
                nonNaNBool=~isnan(pointillisticData.(dataFieldName));
                dataCoords.x=pointillisticData.x_coord(nonNaNBool);
                dataCoords.y=pointillisticData.y_coord(nonNaNBool);
                dataValues=pointillisticData.(dataFieldName)(nonNaNBool);
            else
                % if no data field is specified, accept all the data:
                dataCoords.x=pointillisticData.x_coord;
                dataCoords.y=pointillisticData.y_coord;
                dataValues=1;
            end
        end
        
        
    end
end

