classdef sortedPointillisticData_divide
    % This class contains methods that divides the given data into regions according to
    % the given bounds. The data must be sorted in ascending order, and if
    % several bounds are given, they also must be sorted in ascending
    % order. The methods try to employ a computation efficient region
    % halving method to divide the data according to the given bounds.
    % If a data point coincides with of the region's bound, it will only
    % belong to the region if it equals the lower bound and not the upper one.
    
    % Methods:
    % - getOriginalDataIndices_from_bounds:
    % returns the undices of the original data within the regions defined
    % by the bounds
    % - getOriginalDataIndices_from_indexRanges:
    % returns the undices of the original data within the regions defined
    % by the sorted data index ranges
    % - getIndexRanges_adjacentRegions:
    % returns the undex ranges of the sorted data defining each region (which also contant each other)
    % getIndexRanges_seperateRegions:
    % returns the undex ranges of the sorted data defining each region (they are considered non-contacting)
    % - getBoundaryIndices_higherOrEqual:
    % returns the lowest sorted data indices that are equal to or higher
    % than the given bound
    % - getBoundaryIndex_higherOrEqual:
    % returns the lowest sorted data index that is equal to or higher
    % than the single given bound
    
    methods (Static)
        
        function originalDataIndices_regions=getOriginalDataIndices_from_bounds(sortedData, sortingOrder, inputBounds, boundType)
            % This function returns the origial data indices in a cell array
            % belonging to each region. The bounds of the regions can be
            % given in a vector format ('adjacent bounds'), where every
            % neighbouring region have common bounds, or in a 
            % (region number)x2 matrix format ('seperate bounds'), where each region's lower
            % and upper bounds have to be defined and they do not need to
            % have common bounds.
            
            % check which bound type was given
            switch boundType
                case 'seperate bounds'
                    % if the bounds were given in (region number)x2 matrix
                    % format:
                    regionBounds=inputBounds;
                    % get the index ranges of the sorted data between the
                    % regions' lower and upper bounds:
                    indexRanges=sortedPointillisticData_divide.getIndexRanges_seperateRegions(sortedData, regionBounds);
                case 'adjacent bounds'
                    % if the bounds were given in a vector format:
                    bounds=inputBounds;
                    % get the index ranges of the sorted data between the
                    % between bounds:
                    indexRanges=sortedPointillisticData_divide.getIndexRanges_adjacentRegions(sortedData, bounds);
                otherwise
                    error('Unknown bound type was given.');
            end
            % get the original data indices from the sorted data index
            % ranges belonging to each region:
            originalDataIndices_regions=sortedPointillisticData_divide.getOriginalDataIndices_from_indexRanges(sortingOrder, indexRanges);
        end
        
        
        function originalDataIndices_regions=getOriginalDataIndices_from_indexRanges(sortingOrder, indexRanges)
            % This function returns all the original data indices in a cell array
            % that are between the lowest and highest indices of the sorted data defined by the index ranges matrix ((number of regions)x2 size).
            
            % number of regions given by the index ranges matrix:
            divisionN=size(indexRanges,1);
            
            % create a cell array for storing the original data indices:
            originalDataIndices_regions=cell(divisionN,1);
            
            % fill the cell array with the original data indices:
            for idx=1:divisionN
                originalDataIndices_regions{idx}=sortingOrder(indexRanges(idx,1):indexRanges(idx,2));
            end
        end
        
        
        function indexRanges=getIndexRanges_adjacentRegions(sortedData, bounds)
            % This function returns the lowest and highest indices of the
            % sorted data belonging to each region. The regions are defined
            % with a "bounds" vector. The elements of the "bounds" vector
            % must be sorted in ascending order. The intermediate elements
            % correspond to the common, contacting bounds of the neighbouring regions.
            
            % at least to bounds have to be given to be at least one
            % region:
            if numel(bounds)<2
               error('Too few bounds were given.'); 
            end
            
            % get the lowest indices of the sorted data that are equal or just
            % higher than the given bounds:
            sortedDataBoundaryIndices=sortedPointillisticData_divide.getBoundaryIndices_higherOrEqual(sortedData, bounds);
            
            % lowest data indices of the regions:
            lowerIndices=sortedDataBoundaryIndices(1:end-1);
            % highest data indices of the regions:
            upperIndices=sortedDataBoundaryIndices(2:end)-1;
            
            % fill the index ranges (lowest and higest index values) of
            % each regions in a (number of regions)x2 matrix format:
            indexRanges(:,2)=upperIndices;
            indexRanges(:,1)=lowerIndices;
        end
        
        
        function indexRanges=getIndexRanges_seperateRegions(sortedData, regionBounds)
            % This function returns the lowest and highest indices of the
            % sorted data belonging to each region. The regions are defined
            % with a "regionBounds" matrix, which should be
            % (number of regions)x2 sized array, where
            % the first dimension corresponds to the different
            % regions and the second dimension corresponds to the
            % region's upper and lower bounds, otherwise do not need to be
            % ordered.
            
            % check the bounds matrix:
            if size(regionBounds, 2)~=2
               error('The bounds matrix of the seperateregions was not given in a correct size format.'); 
            end
            
            % rearrange the bounds of the matrix into vector with ascending
            % order:
            [sortedBounds, boundSortingOrder]=sort(regionBounds(:));
            % get the lowest indices of the sorted data that are equal or just
            % higher than each of the bounds:
            sortedDataBoundaryIndices=sortedPointillisticData_divide.getBoundaryIndices_higherOrEqual(sortedData, sortedBounds);
            
            % rearrange back the vector into matrix form:
            [~, inverseSortingOrder]=sort(boundSortingOrder);
            indexRanges=reshape(sortedDataBoundaryIndices(inverseSortingOrder), size(regionBounds));
            
            % fill the index ranges (lowest and higest index values) of
            % each regions in a (number of regions)x2 matrix format:
            indexRanges(:,2)=indexRanges(:,2)-1;
            indexRanges(:,1)=indexRanges(:,1);
        end
        
        
        function sortedDataBoundaryIndices=getBoundaryIndices_higherOrEqual(sortedData, bounds)
            % This function find the indices of the sorted that are greater
            % than or equal to the given bounds (several bounds can be
            % given in a vector, in ascending order, just like the sorted
            % data).
            % It tries to save some computation time by dynamically
            % adjusting the minimal and maximal indices between the search
            % is performed. It starts with the first and with the last
            % bound than just to the middle bound. After finding the index
            % belonging to the middle bound, the minimal and maximal
            % searching indices of the yet unconsidered (sorted) bounds can be
            % adjusted. It continus halving the bounds and updates the
            % searching indicdes.
            
            % check the bounds vector:
            if numel(bounds)~=max(size(bounds))
               error('The bounds vector of the adjacent regions was not given in a vector format.'); 
            end
            
            % vector for storing the sorted data indices that are equal or higher
            % than the given bounds
            sortedDataBoundaryIndices=zeros(size(bounds));
            
            if numel(sortedData)==0
               return; 
            end
            
            % the indices of the sorted data's first and last element
            maxIndex=numel(sortedData);
            minIndex=min(1, maxIndex);      % consider the case when the data is empty, too
            
            % at first, let the minimal and maximal searching indices be
            % the indices of the first and of the last elements of the
            % sorted data
            minDataIndices=ones(size(bounds))*minIndex;
            maxDataIndices=ones(size(bounds))*maxIndex;
            
            % finding the data index belonging to the lower most bound:
            initIdx=ceil((minDataIndices(1)+maxDataIndices(1))/2);
            sortedDataBoundaryIndices(1)=sortedPointillisticData_divide.getBoundaryIndex_higherOrEqual(sortedData, bounds(1), minDataIndices(1), maxDataIndices(1), initIdx);
            % finding the data index belonging to the upper most bound:
            initIdx=ceil((minDataIndices(end)+maxDataIndices(end))/2);
            sortedDataBoundaryIndices(end)=sortedPointillisticData_divide.getBoundaryIndex_higherOrEqual(sortedData, bounds(end), minDataIndices(end), maxDataIndices(end), initIdx);
            
            % vector for storing the bound indices that has been
            % considered, the data index belonging to them has been found,
            % now it has been done to the first and last bounds:
            consideredBoundIndices=[1, numel(bounds)];
            
            while true
                % halve the bounds until all has been considered
                
                % these are required for the region halving, for the minimal and
                % maximal search index updating
                % the indices of the considred bound except the last one:
                consideredBoundIndices_lower=consideredBoundIndices(1:end-1);
                % the indices of the considred bound except the first one:
                consideredBoundIndices_upper=consideredBoundIndices(2:end);
                
                % the actual bound indices after the new halving, these
                % bound are right between halfway of the already considered
                % bounds:
                actBoundIndices=floor((consideredBoundIndices_lower+consideredBoundIndices_upper)/2);
                
                % variable for indexing the actual bound indices:
                tempIndices=1:numel(actBoundIndices);
                
                % boolean for deletion of the actual bound indices that has
                % already been considered in a previous step (for which the
                % halving did not produced a new bound as these were adjacent bounds)
                deleteBool=consideredBoundIndices_upper-consideredBoundIndices_lower==1;
                
                % deleting these unnecessary indices:
                actBoundIndices(deleteBool)=[];
                tempIndices(deleteBool)=[];
                
                % if there is no any remaining bound, exit the loop
                if isempty(actBoundIndices)
                    break
                end
                
                % go through the actual bounds
                for idx=1:numel(actBoundIndices)
                    
                    % let the initial data index be in the middle of the
                    % searching reagion:
                    initIdx=ceil((minDataIndices(actBoundIndices(idx))+maxDataIndices(actBoundIndices(idx)))/2);
                    % find the data index belonging to the actual bound:
                    sortedDataBoundaryIndices(actBoundIndices(idx))=sortedPointillisticData_divide.getBoundaryIndex_higherOrEqual(sortedData, bounds(actBoundIndices(idx)), minDataIndices(actBoundIndices(idx)), maxDataIndices(actBoundIndices(idx)), initIdx);
                    
                    
                    % these are required for updating the minimal and maximal
                    % searching indices, the code runs without these, it is
                    % safe to comment them out, the speed should drop only
                    % slightly:
                    if actBoundIndices(idx)-consideredBoundIndices_lower(tempIndices(idx))>1
                        maxDataIndices(consideredBoundIndices_lower(tempIndices(idx))+1:actBoundIndices(idx)-1)=min(sortedDataBoundaryIndices(actBoundIndices(idx)), maxIndex);
                        % the "min()" function is required because the
                        % returned index can be higher than the maximal
                        % index of the data by one
                    end
                    
                    if consideredBoundIndices_upper(tempIndices(idx))-actBoundIndices(idx)>1
                        minDataIndices(actBoundIndices(idx)+1:consideredBoundIndices_upper(tempIndices(idx))-1)=min(sortedDataBoundaryIndices(actBoundIndices(idx)), maxIndex);
                        % the "min()" function is required because the
                        % returned index can be higher than the maximal
                        % index of the data by one
                    end
                    
                end
                % update the list of the already considered bound indices:
                consideredBoundIndices=sort([consideredBoundIndices, actBoundIndices]);
            end
        end
        
        
        function boundaryIdx = getBoundaryIndex_higherOrEqual(sortedData, bound, dataIdxMin, dataIdxMax, initIdx)
            % This function returns the index belonging to the sorted data
            % that is equal or higher than a given bound
            % (sortedData(index) >= bound). It searches for the index
            % between the given minimal and maximal index values.
            % For the searching it halves the possible index range and
            % checks in which half the bound is. It continues until it can
            % not halve the region any more.
            
            % the index of the data which is greater or equal to the bound:
            boundaryIdx=[];
            % boolean which account for the case when could not find the
            % required data index (the bound is out of the given maximal
            % index or out of the whole data):
            
            if dataIdxMin>dataIdxMax
                % check whether correct minimal and maximal index ranges
                % were given
                % it is not checked whether the given maximal and minimal
                % indices are out of the data, it should be the user's
                % responsibility to give correct minimal and maximal
                % indices
                error('The minimal data index can not be higher than the maximal');
            elseif bound<=sortedData(dataIdxMin)
                % if the bound is lower than the data of the given minimal
                % index, return that index
                boundaryIdx=dataIdxMin;
            elseif bound>=sortedData(dataIdxMax)
                % if the bound is lower than the data of the given maximal
                % index, return that index plus 1
                % The plus 1 addition makes it convenient when collecting
                % the indices of the original data between two bounds
                % (it makes the aforementioned boolean unnecessary). It
                % threates this index as it belongs to a theoretical data
                % point that is higher than the bound. But be aware, it can
                % give an error when one want to refrence that data!!!
                boundaryIdx=dataIdxMax+1;
            else
                % if the bound is between the given indices, halve the data
                % ranges until it can not halve it anymore and (found the
                % right index)
                currentIdx=initIdx;
                while true
                    
                    % check in which half of the data the bound is
                    if sortedData(currentIdx)>=bound
                        % if the bound is in the upper half of the half
                        % sorted data:
                        dataIdxMax=currentIdx;
                    else
                        % if the bound is in the lower half of the half
                        % sorted data:
                        dataIdxMin=currentIdx;
                    end
                    
                    % halve the region, "ceil()" is used because the data
                    % of the index must be grater than or equal to the bound
                    currentIdx=ceil((dataIdxMin+dataIdxMax)/2);
                    
                    % check whether the exit condition satisfied (the
                    % region can not be halved anymore)
                    if dataIdxMax-dataIdxMin==1 || dataIdxMax==dataIdxMin
                        boundaryIdx=currentIdx;
                        break;
                    end
                end
            end
        end
        
        function indexRanges = slideBoundaryIndex(sortedData, regionBounds, lowerIndex_starting)
            % This function slides (updates) the given lower and upper
            % bounds (1X2) on a sorted data. Te indices of the previous
            % must be given.
            
            dataNumber = numel(sortedData);
            
            regionNumber = size(regionBounds, 1);

            indexRanges = zeros(regionNumber, 2);

            lowerIndex_previous = lowerIndex_starting;

            for regionIdx = 1:regionNumber

                % update the lower bound
                lowerIndex = lowerIndex_previous;
                while lowerIndex < dataNumber
                    if sortedData(lowerIndex) > regionBounds(regionIdx,1)
                    break
                    end
                    lowerIndex = lowerIndex+1;
                end

                % update the upper bound
                upperIndex = lowerIndex;
                while upperIndex < dataNumber
                    if sortedData(upperIndex) > regionBounds(regionIdx,2)
                    upperIndex = upperIndex-1;
                    break
                    end
                    upperIndex = upperIndex+1;
                end


                lowerIndex_previous = lowerIndex;

                indexRanges(regionIdx, 1) = lowerIndex;
                indexRanges(regionIdx, 2) = upperIndex;

            end
            
        end
        
    end
end
