function driftCoords = driftCalculation_fiduciary(pointillisticData,settings)

    % lets use fixed image settings for now:
    imageSettings.linMag = 8;
    imageSettings.stackSize = ceil([max(pointillisticData.x_coord), max(pointillisticData.y_coord)]);
    imageSettings.cameraPixelSize_nm = 1; % set it a dummy variable
    imagematrix = rainSTORM_recon_counts(pointillisticData, imageSettings);
    figureHandle = figure(...
        'Name','Select a fiducial marker for drift correction',...
        'NumberTitle','off');
    imshow(imagematrix)
    colormap('Hot');
    % setting the colormap range:
    axesHandle=imgca(figureHandle);
    set(axesHandle, 'CLim', [0, 20]);

    reconstructionScaleFactor = imageSettings.linMag;

    boxtrack = boxTracker(pointillisticData, figureHandle, reconstructionScaleFactor);
    
    close(figureHandle)


    % 0. Load required information
    markerFrames = boxtrack.markerFrames; % List of frame #s with mark
    markerPosits   = boxtrack.markerPosits;

    
    markerAnchor = mean(markerPosits,1);
    markerPosits = markerPosits - repmat(markerAnchor,size(markerPosits,1),1);

    nFrames = settings.frameRange(2);
    
    drift = estimateDrift( nFrames, markerFrames, markerPosits );
    
    driftCoords.x_coord = drift(:, 1);
    driftCoords.y_coord = drift(:, 2);

end

%% Collects the drift information for each frame, and 
% removes the effect of the inaccuracy of the localization of the bead.
function [drift] = estimateDrift( nFrames, markerFrames, markerPosits )

    %% Get drift for each frame
    drift = zeros( nFrames, 2 );
    firstIx = find(markerFrames == min(markerFrames), 1, 'first' );
    lastPos = markerPosits( firstIx, : );
    for i = 1:nFrames
        ix = find( i==markerFrames, 1, 'first' );
        if numel( ix ) > 0
            lastPos = markerPosits( ix, : );
        end
        drift(i, :) = lastPos;
    end
    
    % Denoising drift vectors (by coordinates) to remove the inaccuracy
    % of the localization of the bead.
    drift(:,1) = denoise( drift(:,1) );
    drift(:,2) = denoise( drift(:,2) );
end

%% Denoising a vector to remove white noise.
% Convolution with different gaussian windows is performed.
% The residual corresponding to the minimal NCP will resemble the most
% to a white noise.
function y = denoise( x )
    N = length(x);
    
    % Considering Gaussian windows of different sizes
    % The smoothing vector with the minimal NCP value provides the result.
    sigmaList = 2.^[2:1:11];
    for i = 1:length(sigmaList)
        L = sigmaList(i);

        % Building Gaussian window
        ls = linspace(-(L-1)/L, (L-1)/L, L)';
        alpha = 2.5;
        G = exp( -(alpha*ls).^2/2 );
        G = G / sum(G);
    
        padded = padarray( x, [L/2 0], 'replicate' );
        
        y{i} = conv( padded, G, 'same' );
        y{i} = y{i}(L/2+1:end-L/2);
        
        r(i) = NCP(y{i}-x);                     % Computing the NCP of the residual and save it into 'r'
    end
    y = y{ find(r==min(r),1,'first') };         % The minimal NCP indicates the best smoothing.
end

%% NCP - Normalized Cumulative Periodogram
% Provides a measure of how a signal resemble to white-noise (the minimal
% NCP value indicates the more 'similarity' to a white-noise)
function R = NCP( f )
    n = length(f);
    p = ( abs(fft( f ) ) ).^2;
  
    OFFSET = 0;
    N = floor(n/2-1)-OFFSET;
    c = zeros(N,1);
    
    for k = 1:N
        c(k) = sum( abs( p( 2+OFFSET:k+OFFSET+1) ) ) / sum( abs( p(2+OFFSET:N+OFFSET+1) ) );
    end
    
    line = (1:N)' / (N) * max(c);
    
    R = sum(abs(c-line).^2);
end


function boxtrack = boxTracker(pointillisticData, figNewReconHandle, reconstructionScaleFactor)
%
% FUNCTION
%   Selects quality-controlled localisations within a user-defined box
%   Plots the data as a scatterplot, colour coded by frame number
%   May be used to inspect data for motion blur
%   The selected "boxedPosits" can be used for Fiduciary Drift Correction
%
% USER NOTES
% 0. This tool identifies localisations within a user-defined rectangle,
%    for use as a fiducial marker for drift correction.
% 1. Apply this tool to the most recent super-res reconstruction Figure
%    It searches for reviewed Posits - i.e. accepted Localisations -
%    which are only retained for one Figure at a time
%    To avoid confusion, I recommend closing all figure windows, and then
%    plotting a Reviewed reconstruction to work on with this tool - EJR.
% 2. The tool allow you to select one rectangular region within the
%    Super-resolution reconstruction.
% 3. All localisations within this region are identified, and stored as
%    boxedPosits and associated boxedParams
%   ====> boxed Posits might be useful on their own, for identifying the
%    frames that give rise to particular reconstructed features.
% 4. Within each CCD frame for which there is at least one boxed position,
%    the brightest boxed position is identified as a fiducial marker - the
%    data for these points are stored as markerPosits and markerParams
% 4b: BEWARE: If there is one bead in the box, then it should be the
% brightest feature in the box, in each frame - I am assuming this is the
% case. If there are 2 or more beads, or if there are other bright
% features, then this method, in which I identify the brightest boxed
% feature as the fiducial marker, may go wrong and identify some incorrect
% positions for drift correction.
% 5. NOTE that when a Fiducial Mark-corrected reconstruction is plotted,
%    the markerPosits should be removed from reviewedPosits before plotting
%    THIS IS NOW DONE BY THIS SCRIPT - but not immune from overwriting
%    Since the markerPosits have been used to adjust position data, and
%    cannot (well, should not) also be used to display positions.
% 6. NOTE that plotting the scatterplot of boxed Posits can take several
%    seconds - this is (probably) not a MATLAB crash, so be patient.
%    NOTE: Co-ordinates are in the [ROW, COLUMN] image-like notation

% Part 0: Flow Control and Input Arguments
reviewedPosits = [pointillisticData.x_coord, pointillisticData.y_coord];

flagShowBoxedPosits = 0; % Set to zero to supress drawing a scatterplot
% -- plotting it can be time consuming

% Part 1: Identify user-boxed localisations
rect = getrect(figNewReconHandle);
% Above: user must use a UI to specify a region of the reconstuction.
boxCols = [rect(1), rect(1) + rect(3) ] ./ reconstructionScaleFactor;
boxRows = [rect(2), rect(2) + rect(4) ] ./ reconstructionScaleFactor;

selectedRows = ( (reviewedPosits(:,1) > boxRows(1)) & ...
    (reviewedPosits(:,1) < boxRows(2)) & ...
    (reviewedPosits(:,2) > boxCols(1)) & ...
    (reviewedPosits(:,2) < boxCols(2)) );

boxedParams = pointillisticData_filtering.filter(pointillisticData, selectedRows);
boxedPosits = reviewedPosits( selectedRows, :);

meanBoxedPosits = mean(boxedPosits,1); % Used to plot reconstuction


% Part 2: Optionally display the boxed localisations, colour-coded in
% time-order - currently using the jet colormap (first=blue -> last=red)
if(flagShowBoxedPosits)
    % Note that Y-positions are plotted as negative row positions,
    % to match image matrix orientation.
    figure
    myCmap = colormap( jet(pointillisticData.frame_idx(end)) ); % 1 entry per frame
    boxedColour = ones(size(boxedPosits,1), 3);
    for lpBox = 1:size(boxedPosits,1)
        boxedColour(lpBox,:) = myCmap(boxedParams.frame_idx(lpBox), : );
    end

    scatter(boxedPosits(:,2), -boxedPosits(:,1), 12, boxedColour, 'x')
    set(gca,'FontName','Times New Roman','Fontsize',12)
    title('Boxed Positions (Colour = Frame Number)', ...
        'fontSize', 12, 'fontWeight', 'bold');
    xlabel('CCD Col Number', 'fontSize', 12, 'fontWeight', 'bold');
    ylabel('CCD Row Number (Negative for image alignment)', ...
        'fontSize', 12, 'fontWeight', 'bold');
    colorbar
    caxis([0 size(myCmap,1) ])
    set(gca, 'fontSize', 12, 'fontWeight', 'bold');
    axis equal % Set axis increments to be of equal size; prevent stretching
end


% Part 3: Identify Fiducial Marker positions
% Find markerPosits (using ROW, COL coordinates of CCD)
% These are the brightest boxed Position in each frame
% Need the corresponding Frame numbers which have a reliable marker

markerPosits = -ones(boxedParams.frame_idx(end),2); % Initialise as -1 (error)
%markerParams = repmat(boxedParams(1),boxedParams.frame_idx(end),1); % Allocate memory
markerFrames = (1: boxedParams.frame_idx(end) )';   % Corresponding frame numbers
tic
frame_idx=[boxedParams.frame_idx];
for lpFrms = 1:boxedParams.frame_idx(end)
    myParams_BooleanIndices = frame_idx == lpFrms;
    myParams = pointillisticData_filtering.filter(boxedParams, myParams_BooleanIndices); % Finds boxed data in this frame
    switch sum(myParams_BooleanIndices)
        case 0
            % If there is no marker for this frame, then:
            markerFrames(lpFrms) = -1; % Replace "frame number" with 'error' (-1)

        case 1
            %Trivial case
            markerPosits(lpFrms,:) = [myParams.x_coord myParams.y_coord]; % Identify brightest accepted position as the fiducial marker
            %markerParams(lpFrms) = myParams; % ... and note its Parameters
        otherwise
            [~,idx]=sort([myParams.sum_signal]);
            markerPosits(lpFrms,:) = [myParams.x_coord(idx(end)) myParams.y_coord(idx(end))]; % Identify brightest accepted position as the fiducial marker
            %markerParams(lpFrms) = myParams(end); % ... and note its Parameters
    end
end
toc
%markerParams(markerFrames == -1) = [];
markerPosits(markerFrames == -1, :) = [];
markerFrames(markerFrames == -1) = [];

% Part 4: (Drift Correction moved to undrift script)

% Part 5: OUTPUTS
boxtrack.boxCols = boxCols; % For box deletion by Subtract Drift
boxtrack.boxRows = boxRows;
%boxtrack.boxedColour =  boxedColour;
boxtrack.boxedParams =  boxedParams;
boxtrack.markerFrames = markerFrames;
%boxtrack.markerParams = markerParams;
boxtrack.markerPosits = markerPosits;
boxtrack.selectedLocs = selectedRows; % boxed rows of reviewed


end
