classdef localizationPrecisionEstimation
% Collection of formulas for localization presision estimation.

    % possible future additions:
    % Mortensen's MLEwG formula
    % Mortensen's WLS formula
    % astigmatic formulas for MLE and unweighted least squares
    % other PFS models than the Gaussian
    % camera background noise modelling
    % z precision, DeSAntis, Shaevitz (2009) doi:10.1155/2009/896208
    
    methods(Static)

        function std_x = centroid(sig, sum_signal, countsPerPhotoelectron)
            % "Optimized localization analysis for single-molecule tracking and super-resolution microscopy"
            % doi:10.1038/nmeth.1447
            % centroid estimation encertainty, valid when there is no background noise

            N = sum_signal / countsPerPhotoelectron;
            
            var_x = sig.^2./N;

            std_x = sqrt(var_x);
        end

        function std_x = Thompson(sig, sum_signal, background_std, countsPerPhotoelectron)
            % Precise Nanometer Localization Analysis for Individual Fluorescent Probes
            % 
            % a weighted least square approch, without electron multiplication noise.

            s = sig;
            a = 1; % we work in camera pixel units
            N = sum_signal / countsPerPhotoelectron;
            b = background_std / countsPerPhotoelectron;
            
            var_x = (s.^2 + a^2/12)./N + 8*pi*s.^4.*b.^2./(a^2*N.^2);

            std_x = sqrt(var_x);
        end

        function std_x = Mortensen_GME(sig, sum_signal, background_std, countsPerPhotoelectron)
            % "Optimized localization analysis for single-molecule tracking and super-resolution microscopy"
            % doi:10.1038/nmeth.1447
            % an unweighted least square approach, without electron multiplication noise

            a = 1; % we work in camera pixel units
            N = sum_signal / countsPerPhotoelectron;
            b = background_std / countsPerPhotoelectron;
            sigma_a_2 = sig.^2 + 1/12 * a^2;
            
            var_x = sigma_a_2.^2./N .* (16/9 + 8*pi*sigma_a_2.^2.*b.^2./(N*a^2));

            std_x = sqrt(var_x);
        end

        function [std_x, std_y] = DeSantis(sig_x, sig_y, sum_signal, background_std, countsPerPhotoelectron)
            % "Precision analysis for standard deviation measurements of single fluorescent molecule images"
            % It is a modification of Thompson's formula, adopted to astigmatic PSF and EMCCD
            
            a = 1; % we work in camera pixel units
            N = sum_signal / countsPerPhotoelectron;
            b = background_std / countsPerPhotoelectron;
            s_x_2 = sig_x.^2 + a^2/12;
            s_x = sqrt(s_x_2);
            s_y_2 = sig_y.^2 + a^2/12;
            s_y = sqrt(s_y_2);
            
            var_x = 2*s_x_2./N + 8*pi*s_x.^3.*s_y.*b.^2./(a^2*N.^2);
            var_y = 2*s_y_2./N + 8*pi*s_y.^3.*s_x.*b.^2./(a^2*N.^2);

            std_x = sqrt(var_x);
            std_y = sqrt(var_y);

        end

    end
end

