classdef pointillisticData_coordinateTransformation
    % This class contains methods to perform various kind of coordinate
    % transformations on the given pointillistic data
    
    methods (Static)
        function pointillisticData = transform(pointillisticData, transformationSettings)
            % Perform all the set linear transformations
            
            acquisitionParams=transformationSettings.acquisitionParams;
            ROI=transformationSettings.evaluationSettings.ROI;
            
            % perform the rotation first
            if ~isempty(transformationSettings.rotation_XY)
                rotationAngle_XY = transformationSettings.rotation_XY;
                pointillisticData = pointillisticData_coordinateTransformation.rotate_XY(pointillisticData, rotationAngle_XY, ROI, acquisitionParams);
            end
            % let the translation should be the last transformation
            if ~isempty(transformationSettings.translation_XY)
                translation_XY = transformationSettings.translation_XY;
                pointillisticData = pointillisticData_coordinateTransformation.translate_XY(pointillisticData, translation_XY, acquisitionParams);
            end
            
        end
        
        
        function pointillisticData = inverseTransform(pointillisticData, transformationSettings)
            % Invert all the already performed linear transformations
            
            acquisitionParams=transformationSettings.acquisitionParams;
            ROI=transformationSettings.evaluationSettings.ROI;
            
            % perform the rotation first
            if ~isempty(transformationSettings.rotation_XY)
                rotationAngle_XY = transformationSettings.rotation_XY;
                rotationAngle_XY.value = -rotationAngle_XY.value;
                pointillisticData = pointillisticData_coordinateTransformation.rotate_XY(pointillisticData, rotationAngle_XY, ROI, acquisitionParams);
            end
            % let the translation should be the last transformation
            if ~isempty(transformationSettings.translation_XY)
                translation_XY = transformationSettings.translation_XY;
                translation_XY.x = -translation_XY.x;
                translation_XY.y = -translation_XY.y;
                pointillisticData = pointillisticData_coordinateTransformation.translate_XY(pointillisticData, translation_XY, acquisitionParams);
            end
            
        end
        
        
        function pointillisticData = rotate_XY(pointillisticData, angle, ROI, acquisitionParameters)
            % Rotata the points around the ROI geometrical center in the
            % "x-y" plane (around the "z" axis)
            
            
            angle = unitConversion.convertScalarQuantity(angle, 'radian', acquisitionParameters);

            
            % check the unit the ROI is given in and convert it to
            % camera pixels:
            ROI_center=unitConversion.convert2DVectorQuantity(ROI.center, 'camera pixel length', acquisitionParameters);
            
            % perform the coordinate rotation around the ROI geometrical
            % center
            x_rotated=cos(angle.value)*(pointillisticData.x_coord-ROI_center.x)+sin(angle.value)*(pointillisticData.y_coord-ROI_center.y);
            y_rotated=-sin(angle.value)*(pointillisticData.x_coord-ROI_center.x)+cos(angle.value)*(pointillisticData.y_coord-ROI_center.y);
            
            % get the rotated coordinates
            pointillisticData.x_coord=x_rotated+ROI_center.x;
            pointillisticData.y_coord=y_rotated+ROI_center.y;
        end
        
        
        function pointillisticData = translate_XY(pointillisticData, translation, acquisitionParameters)
            % Translate the points in the "x-y" plane
            
            % check the unit the translation is given in and convert it to
            % camera pixels:
            translation=unitConversion.convert2DVectorQuantity(translation, 'camera pixel length', acquisitionParameters);
            
            % perform the translation
            pointillisticData.x_coord=pointillisticData.x_coord+translation.x;
            pointillisticData.y_coord=pointillisticData.y_coord+translation.y;
        end
        
        
        function pointillisticData = reflect_X()
            % TODO
        end
        
        
        function pointillisticData = reflect_Y()
            % TODO
        end
        
        function [A,t] = getAffineTransform(x1,x2,x3,y1,y2,y3)
            % x1,x2,x3: vectors of the space that should be transformed
            % y1,y2,y3 :vectors of the same 3 point to where the transform should be
            % performed
            % y = A*x+t

            % agmented matrices created form the vectors: Y = T * X
            X = [vertcat(x1, 1), vertcat(x2, 1), vertcat(x3, 1)];
            Y = [vertcat(y1, 1), vertcat(y2, 1), vertcat(y3, 1)];

            % the transform matrix: T = Y * inv(A)
            T = Y/X;

            % extracting the transformation matrix and the translation vector:
            A = T(1:2,1:2);
            t = T(1:2,3);

        end

        function [localizationData] = performAffineTransform(localizationData, A, t)

            x = [localizationData.x_coord; localizationData.y_coord];

            y = A * x + t;

            localizationData.x_coord = y(1,:);
            localizationData.y_coord = y(2,:);

        end
        
    end
end

