classdef pointillisticData_fieldManagement
    % This class cintaines methods for managing the fields of a
    % pointillistic data. Acts on the structure/table of the data, not on a
    % cell array of field names.
    
    % Methods:
    % - add: add a new field to the axisting data
    % - rename: renames the selected fields of a pointillistic data
    % - rearrange: rearranges the order of the fields of a pointillistic data
    % - remove: removes the selected fields of a pointillistic data (all selected fields, error is any does not exist)
    % - delete: deletes the selected fields of a pointillistic data (only existing fields)
    % - select: keep the selected fields, delete the others (with error handling)
    % - pick: pick the selected fields of the data, delete the others (all selected fields, error is any does not exist)
    % - keep: keep the selected fields of the data, delete the others (only existing fields)
    
    % Dependencies:
    % - pointillisticData_fieldNameManagement
    % - pointillisticData_fieldNameDefinitions
    
    methods (Static)
        
        function pointillisticData = add(pointillisticData, newFieldName, newFieldData)
           % Add a new data fields or replaces the content of an existing
           % one.
            
            if istable(pointillisticData)
                % if it a table
                
%                 if pointillisticData_fieldNameManagement.isMemberField(pointillisticData, newFieldName)
%                     % replace the field of the table
%                     pointillisticData.(newFieldName) = newFieldData;
%                 else
%                     % add a new field to the table
%                     pointillisticData = addvars(pointillisticData, newFieldData, 'NewVariableNames', newFieldName);
%                     % probably it works with simple "." indexing too,
%                     % without the "addvars" function, like using 
%                 end
            elseif isstruct(pointillisticData)
                % if it is a structure
                
                %pointillisticData.(newFieldName) = newFieldData;
            else
                error('Invalid pointillistic data type')
            end
            
            pointillisticData.(newFieldName) = newFieldData;
            
        end
        
        function pointillisticData = rename(pointillisticData, fieldsToBeRenamed, fieldNamesForRenaming)
            % This function renames the given fields of a pointillistic
            % data. It operates on the pointillistic data, not on a list
            % of field names.
            
            % get the field names of the pointillistic data
            allFieldNames = pointillisticData_fieldNameManagement.getFieldNames(pointillisticData);
            
            % checking the fields of the pointillistic data
            [~, ~, fullMatchingBool] = pointillisticData_fieldNameManagement.checkMatching(fieldsToBeRenamed, allFieldNames);
            if ~fullMatchingBool
                warning('The pointillistic data does not contain every field you want to rename. Please check it.')
            end
            
            % replace the fields that should be renamed
            allFieldNames_renamed = pointillisticData_fieldNameManagement.replace(allFieldNames, fieldsToBeRenamed, fieldNamesForRenaming);
            
            % renaming the fields/variables of the pointillistic data
            if isstruct(pointillisticData)
                % if it is a structure containing arrays
                pointillisticData=cell2struct(struct2cell(pointillisticData), allFieldNames_renamed);
            elseif istable(pointillisticData)
                % if it a table
                pointillisticData.Properties.VariableNames=allFieldNames_renamed;
            end
            
        end
        
        
        function pointillisticData = rearrange(pointillisticData, orderedFieldNames)
            % This function changes the order of the pointillistic data's
            % fields accoding to the order of the "orderedFieldNames" cell
            % array. The "orderedFieldNames" cell array can contain less
            % fields that the pointillistic sta.
            
            if istable(pointillisticData)
                % if it a table
                
                % the "movevars" funtion reqiures a list that can be
                % shorter than the number of the data's field names
                pointillisticData=movevars(pointillisticData, orderedFieldNames, 'Before', 1);
            elseif isstruct(pointillisticData)
                % if it is a structure
                
                % get all the field names of the pointillistic data
                allFieldNames=fieldnames(pointillisticData);
                
                [allFieldNames, ~]=pointillisticData_fieldNameManagement.order(allFieldNames, orderedFieldNames);
                
                % the "movevars" funtion reqiures a list that is exactly as
                % as long as the number of the data's field names
                pointillisticData=orderfields(pointillisticData, allFieldNames);
            else
                error('Invalid pointillistic data type')
            end
        end
        
        
        function pointillisticData = remove(pointillisticData, fieldsToRemove)
            % This function removes the fields/variable of a pointillistic
            % data let it be a structure of arrays or a table. It gives an
            % error when the user want to delete a field that is not part
            % of the data. It throws an error when one wants to remove a
            % field that is not part of of data (both in cas of "table" or
            % "struct" data format).
            
            if isstruct(pointillisticData)
                % if it is a structure containing arrays
                pointillisticData=rmfield(pointillisticData, fieldsToRemove);
            elseif istable(pointillisticData)
                % if it is a table
                pointillisticData=removevars(pointillisticData, fieldsToRemove);
            else
                error('Invalid pointillistic data type')
            end
        end
        
        
        function pointillisticData = delete(pointillisticData, fieldsToRemove)
            % This function removes the fields/variable of a pointillistic
            % data let it be a structure of arrays or a table. It does not 
            % gives an error when the user want to delete a field that is
            % not part of the data.
            
            % get the field names of the pointillistic data
            fieldNames = pointillisticData_fieldNameManagement.getFieldNames(pointillisticData);
            [matchingBoolVect, ~, ~ ] = pointillisticData_fieldNameManagement.checkMatching(fieldsToRemove, fieldNames);
            pointillisticData = pointillisticData_fieldManagement.remove(pointillisticData, fieldsToRemove(matchingBoolVect));
            
        end
        
        
        function outputData = select(inputData, chosenFields, varargin)
            % This function selects the fields of the given data. It has
            % some error handling capability in case the user has given
            % any different fields than that of the data.
            
            try
                % if the localization data contained all the given fields
                outputData = pointillisticData_fieldManagement.pick(inputData, chosenFields, varargin{:});
            catch
                % if other fields were also given than what the
                % localization data contained
                outputData = pointillisticData_fieldManagement.keep(inputData, chosenFields, varargin{:});
                warning('Could not select every chosen fields. Please reconsider choosing different fields or supplementing the data with the other fields.')
            end
            
        end
        
        
        function outputData = pick(inputData, chosenFields, varargin)
            % Get the given fields from the old data and fill the new data
            % with them. It gives an error if the old data does not contain
            % any of the given fields.
            
            % check the input arguments
            if numel(varargin)==0
                % let them be empty, it will give error in any if non-specific
                % field name is defined
                pointillisticDataParams=struct();
                evaluationType='';
            elseif numel(varargin)==2
                % parameters for getting the specific field names, if there
                % is any
                pointillisticDataParams=varargin{1};
                evaluationType=varargin{2};
            else
               error('invalid number of input arguments') 
            end
            
            % get the specific fieldnames of the chosen fields in a cell array
            chosenFieldNames=pointillisticData_fieldNameDefinitions.getSpecificFieldNames(chosenFields, pointillisticDataParams, evaluationType);
            
             % number of the given fields
            fieldsN=numel(chosenFieldNames);
            
            if isstruct(inputData)
                % if the old data is a strucure of arrayys
                outputData=struct;
                for idxField=1:fieldsN
                    outputData.(chosenFieldNames{idxField})=inputData.(chosenFieldNames{idxField});
                end
            elseif istable(inputData)
                % if the old data is a table
                outputData=table;
                for idxField=1:fieldsN
                    outputData.(chosenFieldNames{idxField})=inputData.(chosenFieldNames{idxField});
                end
            else
                error('The input data can only be table or structure of arrays.')
            end
            
        end
        
        
        function pointillisticData = keep(pointillisticData, fieldsToKeep, varargin)
            % This function removes any fields of the old data that are not
            % listed in the 'fieldsToKeep'. It does not gives an error if
            % the old data does not contain some of the given fields.
            
            % check the input arguments
            if numel(varargin)==0
                % let them be empty, it will give error in any if non-specific
                % field name is defined
                pointillisticDataParams=struct();
                evaluationType='';
            elseif numel(varargin)==2
                % parameters for getting the specific field names, if there
                % is any
                pointillisticDataParams=varargin{1};
                evaluationType=varargin{2};
            else
               error('invalid number of input arguments') 
            end
            
            % get the specific fieldnames of the chosen fields in a cell array
            fieldNamesToKeep=pointillisticData_fieldNameDefinitions.getSpecificFieldNames(fieldsToKeep, pointillisticDataParams, evaluationType);
            
            % get the field names of the data
            fieldNames=pointillisticData_fieldNameManagement.getFieldNames(pointillisticData);
            
            % check if the fields of the data match with any of the given fields
            [matchingBoolVect,~, ~] = pointillisticData_fieldNameManagement.checkMatching(fieldNames, fieldNamesToKeep);
            
            % remove the unnecessary fields
            fieldsToRemove=fieldNames(~matchingBoolVect);
            pointillisticData = pointillisticData_fieldManagement.remove(pointillisticData, fieldsToRemove);
            
        end
        
        
        function emptyData = createEmptyData(templateData)
            
            % TODO create empty data without a template data
            
            fieldNames=pointillisticData_fieldNameManagement.getFieldNames(templateData);
            dataClass=class(templateData);
            
            % create basis for the rejected localization data
            if strcmp(dataClass, 'struct')
                % create empty structure for the rejected localizations
                emptyData=cell2struct(cell(size(fieldNames)), fieldNames, 1);
            elseif strcmp(dataClass, 'table')
                % create empty table for the rejected localizations
                emptyData=table();
            else
                error('The localization data can only be table or structure of arrays.')
            end
            
            % the "cell2table" and "array2table" does not work if
            % the the empty data has to be of a specific type, if the data
            % is a table
            % go through the fields and create the empty datas with
            % the right type
            for idxField=1:numel(fieldNames)
                actualFieldName=fieldNames{idxField};
                actualFieldClass=class(templateData.(actualFieldName));
                classFunc=str2func(actualFieldClass);
                % create the empty data
                emptyData.(actualFieldName)=classFunc([]);
            end
        end
        
    end
end

