classdef pointillisticData_fieldNameManagement
    % A collection of merhods acting on field names. The methods do not
    % modify the pointillistic data
    
    % Methods:
    % - defaultOrder: orders the fieldnames in the defined, default order,
    %       operates only on cell arrays of the fieldnames, not on a pointillistic
    %       data
    % - order: orders the fieldnames according the order of the reference
    %       field names, operates only on cell arrays of the fieldnames, not on a pointillistic
    %       data
    % - replace: replaces on or more field names with several fieldnames,
    %       operates only on cell arrays of the fieldnames, not on a pointillistic
    %       data
    % - getFieldNames: returns the field names of the given pointillistic
    %       data
    % - isMemberField: inspects whether the given fields are member of the
    %       pointillistic data
    % - checkMatching: checks whether the given fields exist among the 
    %       reference field names
    
    % Dependencies:
    % - pointillisticData_fieldDefinitions
    
    methods (Static)
        
        function fieldNames = getFieldNames(pointillisticData)
            % This function return the field names (variable names in case
            % of table) of the given pointillistic data
            
            if isstruct(pointillisticData)
                fieldNames=fieldnames(pointillisticData);
            elseif istable(pointillisticData)
                fieldNames=pointillisticData.Properties.VariableNames;
            else
                error('Invalid localization data.')
            end
        end
        
        function [orderedFieldNames, orderVect] = order(fieldNames, arrangedFieldNames)
            % This functiom arranges the field names according to the field name order in 
            % the "arrangedFieldNames" variable. Less field names can be 
            % given than that is defined in the "arrangedFieldNames". 
            % Fields undefined in the  "arrangedFieldames" go to the very
            % end retaining their orginal order.
            
            % indices the fieldnames in the original order
            originalOrderVect=1:numel(fieldNames);
            % vector for not yet ordered fields
            remainingOrderVect=originalOrderVect;
            
            % vector of indices in the new order
            orderVect=zeros(size(originalOrderVect));
            
            %  use two indices as it has to handle the case where several
            %  fields have the same names, it should keep them all
            idxOrdered_prev=0;
            idxOrdered_new=0;
            for idxArrangedFields=1:numel(arrangedFieldNames)
                % find the indices of the actual field name in the 
                orderBoolVect=strcmp(arrangedFieldNames{idxArrangedFields}, fieldNames);
                % if there are matching fields, add them in to the right
                % indices
                if nnz(orderBoolVect)>0
                    idxOrdered_new=idxOrdered_new+nnz(orderBoolVect);
                    orderVect(idxOrdered_prev+1:idxOrdered_new)=originalOrderVect(orderBoolVect);
                    idxOrdered_prev=idxOrdered_new;
                    % assign zeros to the ordered fields
                    remainingOrderVect(orderBoolVect)=0;
                end
            end
            % remove the already ordered fields
            remainingOrderVect(remainingOrderVect==0)=[];
            
            % concatenating the remaining fields to the end in the original order
            if ~isempty(remainingOrderVect)
                orderVect(idxOrdered_prev+1:end)=remainingOrderVect;
            end
            % rearrange the order of the field names
            orderedFieldNames={fieldNames{orderVect}};
            
        end
        
        
        function fieldNamesReplaced=replace(fieldNamesOriginal, fieldsToReplace, fieldsToPaste)
            % This function replaces the selected field names with the specified
            % field names. This function only works on a
            % cell array of field names.
            % The selected field names ("fieldsToReplace") and the specified
            % field names ("fieldsToPaste") are stored in a cell arrays. The 
            % cell array itself also contains cell arrays with different
            % number of elements (field names). The indices of the first cell arrays give
            % which fields names (second cell array) should be replaced
            % with which fields names (second cell array). If the inner, second
            % cell arrays contain several field names, the field names to
            % be pasted will be placed
            % beside each other, at the place of the first one of the field
            % names to be replaced.
            
            
            % checking the input arguments
            if ischar(fieldsToReplace)
                % if only one field were given to replace as a string
                fieldsToReplace={fieldsToReplace};
            elseif iscell(fieldsToReplace)
                % if only several fields were given to replace as a cell
                % array
                % do nothing
            else
                error('Invalid field names were given to replace.')
            end
            if ischar(fieldsToPaste)
                % if only one field were given to replace as a string
                fieldsToPaste={fieldsToPaste};
            elseif iscell(fieldsToReplace)
                % if only several fields were given to replace as a cell
                % array
                % do nothing
            else
                error('Invalid field names were given to paste.')
            end
            
            
            fieldNames_temp=fieldNamesOriginal;  % required if several fields are to be replaced
            % go through the fields to replace:
            for idxReplacement=1:size(fieldsToReplace, 1)
                % required if several fields are to be replaced:
                fieldNamesOriginal=fieldNames_temp;
                % number of fields to paste in the place of the selected
                % field name:
                pasteN=numel(fieldsToPaste(idxReplacement, :));
                % create a temporarily file containing the field names with
                % the size considering the replacement, and shift the field
                % names so there are enough free space in the beginning of
                % the cell array:
                fieldNames_temp=[cell(1, numel(fieldsToPaste(idxReplacement,:))-1), fieldNamesOriginal];
                if isempty(fieldsToReplace{idxReplacement})
                    
                else
                    % index of the field that should be replaced
                    indices=zeros(size(fieldsToReplace,2),1);
                    for idx=1:size(fieldsToReplace,2)
                        foundFieldIndex=find(strcmp(fieldNames_temp, fieldsToReplace{idxReplacement, idx}));
                        if ~isempty(foundFieldIndex)
                            indices(idx)=foundFieldIndex;
                        end
                    end
                    
                    % index of the field that should be replaced (the first)
                    firstIdx=indices(1);
                    % indices of the other fields
                    otherIndices=indices(2:end);
                    if firstIdx==0
                    %if isempty(firstIdx)
                        % do not try to replace anything
                        %fieldNamesReplaced=fieldNamesOriginal;
                    else
                        % field names with index smaller than that of the one
                        % to be replaced, should be shifted back
                        if firstIdx>pasteN
                            % it only has to shifted if this condition is
                            % satisfied, otherwise it can be shifted but
                            % not needed
                            fieldNames_temp(1:firstIdx-pasteN)=fieldNames_temp(pasteN:firstIdx-1);
                        end
                        % replace the field with the specified ones by pasting
                        % them
                        for idxPaste=1:pasteN
                            fieldNames_temp{firstIdx-pasteN+idxPaste}=fieldsToPaste{idxReplacement, idxPaste};
                        end
                        % the other fields of the cell array should be deleted
                        fieldNames_temp(otherIndices)=[];
                        % field names after the replacements
                        %fieldNamesReplaced=fieldNames_temp;
                        fieldNamesOriginal=fieldNames_temp;
                    end
                end
            end
            fieldNamesReplaced=fieldNamesOriginal;
        end
        
        
        function [allFieldsBoolean, fieldBooleanVector] = isMemberField(data, fieldNames)
            % This function investigates whether the field is member of the
            % pointillistic data let it be a structure of arrays or a table
            
            % check the given field names
            if ischar(fieldNames)
                fieldNames={fieldNames};
            elseif iscell(fieldNames)
                % do nothing
            else
                error('Invalid field names were given.')
            end
            
            fieldBooleanVector=false(size(fieldNames));
            

            if isstruct(data)
                % go thought the given field names
                for idxFieldName=1:numel(fieldNames)
                    % If the localization data is a struct
                    fieldBooleanVector(idxFieldName)=isfield(data,fieldNames{idxFieldName});
                end
            elseif istable(data)
                % If the localization data is a table
                fieldNames_locData=data.Properties.VariableNames;   % in case of a table
                
                % go thought the given field names
                for idxFieldName=1:numel(fieldNames)
                    if sum(strcmp(fieldNames_locData, fieldNames{idxFieldName}))==1
                        fieldBooleanVector(idxFieldName)=true;
                    else
                        fieldBooleanVector(idxFieldName)=false;
                    end
                end
            else
                error('The given data is neither a structure nor a table.')
            end
            
            allFieldsBoolean = prod(fieldBooleanVector);
            
        end
        
        
        function [matchingBoolVect, matchingReferenceFieldN, fullMatchingBool] = checkMatching(fieldNames_given, fieldNames_reference)
            % This function investigates how many of the given fields exists among the
            % reference fields. The fields of both inputs should be stored
            % in a cell array.
            
            % boolean vector for the matching fields
            matchingBoolVect=false(size(fieldNames_given));
            
            % go through the given fields
            for idxField=1:numel(fieldNames_given)
                % check if the given field occurs in the refrence fields
                tempBool=strcmp(fieldNames_given(idxField), fieldNames_reference);
                matchingBoolVect(idxField)=nnz(tempBool)>0;      % it should not be larger than 1, but it is safer this way
            end
            
            % number of matching fields
            matchingReferenceFieldN=sum(matchingBoolVect);
            % if all the fields match:
            fullMatchingBool=prod(matchingBoolVect);
        end
        
    end
end

