classdef pointillisticData_filtering
% These function perform the filtering on the localization data with the
% given data fieldnames (variable names if the localization data is a
% table) and filtering thresholds. If the data was already filtered, for
% the refiltering both the filtered and leftout data has to be added.
% The "pontillistic data" needs to beiother a struct, a table, or a cell
% contaning the fieldnames in cell and the data i matric formats.

% Methods
% - filter: 
% - refilter: 
% - threshold: simply filters the given pontillistic data based on given
% thresholds
% - rethreshold: performs the filtering again on an already filtered dataset,
%       requires the accepted and the rejected data
% - getBooleanVect: checks which values pass the filtering, but do not
%       perform the fileting
% - select: 
    
    methods (Static)
        
        function [filterBoolVect] = getBooleanVect(pointillisticData, filteringStruct)
            % This function performs filtering on the pointillistic data (let it be
            % table or structure containing arrays) based on the thresholds given to
            % its fields. The filter thresholds must be contained in a struct.
            % The boundaries are included (if a data equals on the thresholds, it passes the filtering.)
            
            % pointillisticData: struct or table or a cell containing the
            % field names and the data matrix
            
            % boolean vector for the filtering
            fieldNames_data = pointillisticData_filtering.getFieldNames(pointillisticData);
            if ~ isempty(fieldNames_data)
                filterBoolVect=true(size(pointillisticData_filtering.getColumn(pointillisticData, fieldNames_data{1})));
            else
                filterBoolVect = true();
                return
            end
            
            % getting the fieldnames of the filtering
            fieldNames_filter={filteringStruct.field};
            
            % going through the filtering thresholds of the different fields
            % to check whether the data has the fields (the leftout ("rejected") data is empty by default)
            for idxField=1:numel(fieldNames_filter)
                
                fieldName = fieldNames_filter{idxField};
                
                if sum(strcmp(fieldNames_data, fieldName)) == 0
                    warning('The pointillistic data does not contain "%s" field, cannot filter on it.', fieldName)
                   continue; 
                end
                
                dataToFilter = pointillisticData_filtering.getColumn(pointillisticData, fieldName);
                
                % the filtering thresholds
                thresholds_bounds = filteringStruct(idxField).bounds;
                if numel(thresholds_bounds) == 2
                    lowerBound=thresholds_bounds(1);
                    upperBound=thresholds_bounds(2);
                else
                   error('The filterings thresholds of the "%s" field needs to be a 2-elements vector. Instead its size is "%d" with type of "%s".', fieldName, numel(thresholds_bounds), class(thresholds_bounds));
                end
                
                if lowerBound==-Inf && upperBound==Inf
                    % do nothing
                elseif isempty(dataToFilter)
                    % do nothing
                else
                    % check the filtering condition:
                    filterBoolVect_actQuantity=lowerBound<=dataToFilter & upperBound>=dataToFilter;
                    % do not filer on NAN datas (Matlab's relations
                    % considers them as false):
                    filterBoolVect_actQuantity(isnan(dataToFilter)) = true;
                    % update the boolean vector:
                    filterBoolVect=filterBoolVect & filterBoolVect_actQuantity;
                end
                
            end 
        end
        

        function [pointillisticData_accepted, pointillisticData_rejected] = filter(pointillisticData, filterBoolVect)
            
            % getting the fieldnames of the pointillistic data
            fieldNames_data = pointillisticData_filtering.getFieldNames(pointillisticData);
            
            % perform the filtering on the table or on the arrays of the struct
            if istable(pointillisticData)
                
                pointillisticData_accepted = pointillisticData_fieldManagement.createEmptyData(pointillisticData);
                pointillisticData_rejected = pointillisticData_fieldManagement.createEmptyData(pointillisticData);

                % data that passed the filtering
                if any(filterBoolVect)
                    pointillisticData_accepted=pointillisticData(filterBoolVect,:);
                end
                % data that did not pass the filtering
                if any(~filterBoolVect)
                    pointillisticData_rejected=pointillisticData(~filterBoolVect,:);
                end
            elseif isstruct(pointillisticData)

                pointillisticData_accepted = pointillisticData_fieldManagement.createEmptyData(pointillisticData);
                pointillisticData_rejected = pointillisticData_fieldManagement.createEmptyData(pointillisticData);

                for idxField=1:numel(fieldNames_data)
                    % data that passed the filtering
                    if any(filterBoolVect)
                        pointillisticData_accepted.(fieldNames_data{idxField})=pointillisticData.(fieldNames_data{idxField})(filterBoolVect);
                    end
                    % data that did not pass the filtering
                    if any(~filterBoolVect)
                        pointillisticData_rejected.(fieldNames_data{idxField})=pointillisticData.(fieldNames_data{idxField})(~filterBoolVect);
                    end
                end
            elseif iscell(pointillisticData)
                fiednames = pointillisticData{1};
                data = pointillisticData{2};
                
                % data that passed the filtering
                data_accepted=data(filterBoolVect,:);
                pointillisticData_accepted = {fiednames, data_accepted};
                % data that did not pass the filtering
                data_rejected=data(~filterBoolVect,:);
                pointillisticData_rejected = {fiednames, data_rejected};
            else
                error('Wrong data type to filter')
            end
            
        end
        
        
        function [pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new] = refilter(pointillisticData_accepted_old, pointillisticData_rejected_old, filterBoolVect_old, filterBoolVect_accepted_new, filterBoolVect_rejected_new)
            
            % creating the new filter boolean vector
            filterBoolVect_new=false(size(filterBoolVect_old));
            filterBoolVect_new(filterBoolVect_old)=filterBoolVect_accepted_new;
            filterBoolVect_new(~filterBoolVect_old)=filterBoolVect_rejected_new;
            % size of the new filtered data
            newFilteredDataLength=nnz(filterBoolVect_new);
            newLeftoutDataLength=nnz(~filterBoolVect_new);
            newFilteredDataSize=[newFilteredDataLength, size(pointillisticData_accepted_old, 2)];
            newLeftoutDataSize=[newLeftoutDataLength, size(pointillisticData_accepted_old, 2)];
            
            % filtering the old boolean filter vector
            filterBoolVect_old_filtered=filterBoolVect_old(filterBoolVect_new);
            filterBoolVect_old_leftout=filterBoolVect_old(~filterBoolVect_new);
            
            % getting the fieldnames of the pointillistic data
            fieldNames_data = pointillisticData_filtering.getFieldNames(pointillisticData_accepted_old);
            
            % perform the filtering on the table or on the arrays of the struct
            if isstruct(pointillisticData_accepted_old)
                
                for idxField=1:numel(fieldNames_data)
                    % create a new empty struct with the new size and with the
                    % old variable names
                    pointillisticData_accepted_new.(fieldNames_data{idxField})=zeros(newFilteredDataLength,1);
                    pointillisticData_rejected_new.(fieldNames_data{idxField})=zeros(newLeftoutDataLength,1);
                    
                    % data that passed the filtering
                    pointillisticData_accepted_new.(fieldNames_data{idxField})(filterBoolVect_old_filtered)=...
                        pointillisticData_accepted_old.(fieldNames_data{idxField})(filterBoolVect_accepted_new);
                    pointillisticData_accepted_new.(fieldNames_data{idxField})(~filterBoolVect_old_filtered)=...
                        pointillisticData_rejected_old.(fieldNames_data{idxField})(filterBoolVect_rejected_new);
                    % data that did not pass the filtering
                    pointillisticData_rejected_new.(fieldNames_data{idxField})(filterBoolVect_old_leftout)=...
                        pointillisticData_accepted_old.(fieldNames_data{idxField})(~filterBoolVect_accepted_new);
                    pointillisticData_rejected_new.(fieldNames_data{idxField})(~filterBoolVect_old_leftout)=...
                        pointillisticData_rejected_old.(fieldNames_data{idxField})(~filterBoolVect_rejected_new);
                end
            elseif istable(pointillisticData_accepted_old)
                % create a new empty table with the new size and with the
                % old variable names
                pointillisticData_accepted_new=array2table(zeros(newFilteredDataSize), 'VariableNames', fieldNames_data);
                pointillisticData_rejected_new=array2table(zeros(newLeftoutDataSize), 'VariableNames', fieldNames_data);
                
                % data that passed the filtering
                pointillisticData_accepted_new(filterBoolVect_old_filtered,:)=pointillisticData_accepted_old(filterBoolVect_accepted_new,:);
                pointillisticData_accepted_new(~filterBoolVect_old_filtered,:)=pointillisticData_rejected_old(filterBoolVect_rejected_new,:);
                % data that did not pass the filtering
                pointillisticData_rejected_new(filterBoolVect_old_leftout,:)=pointillisticData_accepted_old(~filterBoolVect_accepted_new,:);
                pointillisticData_rejected_new(~filterBoolVect_old_leftout,:)=pointillisticData_rejected_old(~filterBoolVect_rejected_new,:);
            elseif iscell(pointillisticData_accepted_old)
                
                data_accepted_old = pointillisticData_accepted_old{2};
                data_rejected_old = pointillisticData_rejected_old{2};
                
                % data that passed the filtering
                data_accepted_new(filterBoolVect_old_filtered,:)=data_accepted_old(filterBoolVect_accepted_new,:);
                data_accepted_new(~filterBoolVect_old_filtered,:)=data_rejected_old(filterBoolVect_rejected_new,:);
                % data that did not pass the filtering
                data_rejected_new(filterBoolVect_old_leftout,:)=data_accepted_old(~filterBoolVect_accepted_new,:);
                data_rejected_new(~filterBoolVect_old_leftout,:)=data_rejected_old(~filterBoolVect_rejected_new,:);
                
                pointillisticData_accepted_new = {fieldNames_data, data_accepted_new};
                pointillisticData_rejected_new = {fieldNames_data, data_rejected_new};
                
            else
                error('The pointillistic data to be filtered neeods to be a struct, a table or a cell (fieldname cell - data matrix pair). Insterad it is "%s".', class(pointillisticData))
            end
            
        end
        
        
        function [data_accepted, data_rejected, filterBoolVect] = threshold(pointillisticData, filteringThresholds)
            % Perform thresholding on the raw pointillistic data
           
            
            % get the boolean vector of the thresholding
            filterBoolVect=pointillisticData_filtering.getBooleanVect(pointillisticData, filteringThresholds);
            
            % filter the localizations with the boolean vector
            [data_accepted, data_rejected] = pointillisticData_filtering.filter(pointillisticData, filterBoolVect);
            
        end
        
        
        function [data_accepted_new, data_rejected_new, filterBoolVect_new] = rethreshold(data_accepted_old, data_rejected_old, filterBoolVect_old, filteringThresholds)
            % Perform another different filtering using the already filtered
            % and leftout data
            
            
            % the new filter boolean vector created out of the previously filtered data
            filterBoolVect_accepted_new=pointillisticData_filtering.getBooleanVect(data_accepted_old, filteringThresholds);
            % the new filter boolean vector created out of the previously leftout data
            filterBoolVect_rejected_new=pointillisticData_filtering.getBooleanVect(data_rejected_old, filteringThresholds);
            
            [data_accepted_new, data_rejected_new, filterBoolVect_new] = pointillisticData_filtering.refilter(data_accepted_old, data_rejected_old, filterBoolVect_old, filterBoolVect_accepted_new, filterBoolVect_rejected_new);
            
        end
        

        
        function [pointillisticData_accepted, pointillisticData_rejected] = select(pointillisticData, selectedIndices)
            % This functins selects the data from each field based on the
            % given data integer indices. It differs from the "filter"
            % method in that it does not accept boolean indices but
            % integer ones.
            
            % getting the fieldnames of the pointillistic data
            fieldNames_data = pointillisticData_filtering.getFieldNames(pointillisticData);
            
            % get the data indices beside the selected ones
            otherIndices=1:numel(pointillisticData.(fieldNames_data{1}));
            otherIndices(selectedIndices)=[];
            
            if isstruct(pointillisticData)
                % go through the fields of the structure
                for idxField=1:numel(fieldNames_data)
                    % the data belonging to the selected indices
                    pointillisticData_accepted.(fieldNames_data{idxField})=pointillisticData.(fieldNames_data{idxField})(selectedIndices);
                    % the other data beside the selected ones
                    pointillisticData_rejected.(fieldNames_data{idxField})=pointillisticData.(fieldNames_data{idxField})(otherIndices);
                end
            elseif istable(pointillisticData)
                % the data belonging to the selected indices
                pointillisticData_accepted=pointillisticData(selectedIndices, :);
                % the other data beside the selected ones
                pointillisticData_rejected=pointillisticData(otherIndices, :);
            elseif iscell(pointillisticData)
                fiednames = pointillisticData{1};
                data = pointillisticData{2};
                
                % the data belonging to the selected indices
                data_accepted=data(selectedIndices, :);
                pointillisticData_accepted = {fiednames, data_accepted};
                % the other data beside the selected ones
                data_rejected=data(otherIndices, :);
                pointillisticData_rejected = {fiednames, data_rejected};
            else
                error('The pointillistic data to be filtered neeods to be a struct, a table or a cell (fieldname cell - data matrix pair). Insterad it is "%s".', class(pointillisticData))
            end
        end
        

        function strippedFilteringThresholds = stripFilteringThresholds(filteringThresholds, dataFieldNames)
            % Throws out/discards every filtering threshold that is irrevelant to
            % the data
        
            filteringFieldNames = {filteringThresholds.field};
            
            % keep only those filtering fields that the data contains too
            strippedFilteringThresholds = filteringThresholds(ismember(filteringFieldNames, dataFieldNames));
        
        end

    end

    methods(Static,Access = private)

        function fieldNames = getFieldNames(pointillisticData)
            % Returns the field names of the data to be filtered, let it be
            % a table a struct or a fieldnames cell - matrix data pair.

            fieldNames = {};
            if iscell(pointillisticData)
                if iscell(pointillisticData{1})
                   fieldNames =  pointillisticData{1};
                else
                    error('Invalid type of fieldnames (it is type of "%s" and it should be type of "cell") of the data to be filtered in a fieldname cell - data matrix pair format.', class(pointillisticData{1}))
                end
            elseif isstruct(pointillisticData)
                fieldNames=fieldnames(pointillisticData);
            elseif istable(pointillisticData)
                fieldNames=pointillisticData.Properties.VariableNames;
            else
                error('The pointillistic data to be filtered needs to be a struct, a table or a cell (fieldname cell - data matrix pair). Instead it is "%s".', class(pointillisticData))
            end

        end

        function column = getColumn(pointillisticData, fieldName)
            % Returns the chosen column of the data to be filtered, let it be
            % a table a struct or a fieldnames cell - matrix data pair.

            if iscell(pointillisticData)
                fieldnames = pointillisticData{1};
                data = pointillisticData{2};
                column = data(:, strcmp(fieldnames, fieldName));
            elseif istable(pointillisticData) || isstruct(pointillisticData)
                column = pointillisticData.(fieldName);
            else
                error('The pointillistic data to be filtered needs to be a struct, a table or a cell (fieldname cell - data matrix pair). Insterad it is "%s".', class(pointillisticData))
            end

        end

    end
end

