classdef pointillisticData_load
    % This class contains different methods for loding the localization,
    % blinking or other pointillistic data and the accompanying metadata
    % from external files.
    
    methods (Static)
        
        function [dataHeaders, dataMatrix] = headerAndMatrixData(dataFileFullName)
            % Reads the first row of a coma separated data file as headers
            % and loads the numerical data in a matrix. The advantage of
            % this approach compared to the "readtable" method is that in
            % can handle header texts with much wider character ranges and
            % formats.

            if ~isfile(dataFileFullName)
               error(['The *.csv file to be loaded as a matrix does not exist: ', dataFileFullName]) 
            end
            
            % read the headers (first row)
            fid = fopen(dataFileFullName);
            headerLine = textscan(fid, '%[^\r\n]', 1);
            fclose(fid);
            dataHeaders = {};
            if numel(headerLine) == 1
                dataHeaders = split(headerLine{1}, ',');
            elseif numel(headerLine) == 0
                error('Could not read the header row from the *.csv file, it is probably empty.')
            else
               error('Somehow the textscan read multiple lines when reading the headers...') 
            end

            % read the numeric data, below the header row
            row = 1;
            column = 0;
            dataMatrix = csvread(dataFileFullName,row,column);

        end
        
        function infoText = infoText(infoFileFullName)
            % Loads the content of the info file into a string line-by line.
            % https://stackoverflow.com/questions/3537520/read-a-whole-text-file-into-a-matlab-variable-at-once
            % https://www.mathworks.com/matlabcentral/answers/18322-append-string-to-each-element-in-string-cell-array
            
            if ~isfile(infoFileFullName)
               error(['The info file to be loaded as simple text does not exist: ', infoFileFullName]) 
            end
            
            fid = fopen(infoFileFullName);
            s = textscan(fid,'%s','Delimiter','\n');
            fclose(fid);
            s = s{1};
            s = cellfun(@(c)[c newline],s,'UniformOutput',false);

            % the additional one line to turn s to a string
            infoText = cell2mat(reshape(s, 1, []));
            %or:
            %infoText = [s{:}]

        end
        
        function pointillisticData = tabularData(dataFileFullName)
            % Loads contents of a text data file into a Matlab table
            % variable with variable names given in the header row. It is
            % only (correctly) handle header texts that are compatible with
            % Matlab structure's field and Matlab table variable names.
            
            if ~isfile(dataFileFullName)
               error(['The *.csv file to be loaded as a table does not exist: ', dataFileFullName]) 
            end
            
            pointillisticData = readtable(dataFileFullName);
        end
        
        function analysisParameters = infoStruct(infoFileFullName)
            % Load the accompanying information of the data saed onto an
            % *.json file into a structure variable.
            
            if ~isfile(infoFileFullName)
               error(['The *. json info file to be loaded as a structure does not exist: ', infoFileFullName]) 
            end
            
            infoStruct = jsonManagement.import(infoFileFullName);

            % extract the ecvaluation parameters
            analysisParameters = struct();
            if isfield(infoStruct, 'analysisParameters')
                analysisParameters=infoStruct.analysisParameters;
            elseif numel(fieldnames(infoStruct))>0
                warning('The "analysisParameters" does not exist in the loaded info file. Check whether the correct info data was loaded.')
                fieldNames=fieldnames(infoStruct);
                analysisParameters=infoStruct.(fieldNames{1});
            else
                error(['The loaded *.json info file is empty: ',infoFileFullName])
            end
            
            
        end
        
    end
end

