classdef pointillisticData_save
    % This class contains different methods for saving the localization,
    % blinking or other pointillistic data and the accompanying metadata
    % t external files.
    
    methods (Static)
        
        function headerAndMatrixData(directoryPath, dataFileName, dataHeaders, dataMatrix)
            % Saves the data given in matrix formats into a *.csv file  and
            % the names of the columns (headers) to the first line of the
            % .*csv file.
            
            if ~isfolder(directoryPath)
                mkdir(directoryPath)
            end
            
            headerNumber = numel(dataHeaders);
            columnNumber = size(dataMatrix, 2);
            if headerNumber ~= columnNumber
               error(['The number of headers (', num2str(headerNumber),') does not match that of the number of columns of the data matrix (', num2str(headerNumber) ,') when saving the data.']);
            end
            
            % create the header row:
            headerRow_cell  = cellfun(@(c)[c ','],dataHeaders,'UniformOutput',false);
            headerRow_text = [headerRow_cell{:}];
            % delelete the last ",":
            headerRow_text(end) = [];

            dataFileFullName = fullfile(directoryPath,dataFileName);
            
            % write the header row:
            fileID = fopen(dataFileFullName, 'w');
            fprintf(fileID, "%s\n", headerRow_text);
            fclose(fileID);
            
            % write the numerical data:
            dlmwrite(dataFileFullName, dataMatrix, 'delimiter', ',', '-append')
            % in newer matlab versions:
            %writematrix(dataMatrix, dataFileFullName, 'WriteMode', 'append')
            
        end
        
        function infoText(directoryPath, infoFileName, infoText)
            % Saves the accompanying information (already converted to
            % simple text) of the data into a text file.
            
            if ~isfolder(directoryPath)
                mkdir(directoryPath)
            end
            
            infoFileFullName = fullfile(directoryPath, infoFileName);
            
            fileID = fopen(infoFileFullName, 'w');
            fprintf(fileID, "%s\n", infoText);
            fclose(fileID);
            
        end
        
        
        function tabularData(directoryPath, dataFileName, pointillisticData)
            % Saves the pointillistic data, that is already in a table
            % variable directly into a *.csv file. It can only create
            % header texts that are compatible with Matlab structure's
            % field and Matlab table variable names.
            
            if ~isfolder(directoryPath)
                mkdir(directoryPath)
            end
            
            dataFileFullName = fullfile(directoryPath, dataFileName);
            
            if istable(pointillisticData)
                
                fieldNames = pointillisticData.Properties.VariableNames;
                dataLength = numel(pointillisticData.(fieldNames{1}));
                % if the data is stored in row vectors inside the table, transpose it:
                if height(pointillisticData) == 1 && dataLength > 1
                    pointillisticData = array2table(reshape(table2array(pointillisticData), dataLength, []), 'VariableNames', fieldNames);
                end
                
                writetable(pointillisticData, dataFileFullName);
            elseif isstruct(pointillisticData)
                
                % transpose the data in the struct if neccessary to be
                % column vectors and not row vectors:
                fieldNames = fieldnames(pointillisticData);
                for idxField = 1:numel(fieldNames)
                   actualFieldName =  fieldNames{idxField};
                   actualField = pointillisticData.(actualFieldName);
                   if size(actualField, 2) > size(actualField, 1)
                       pointillisticData.(actualFieldName)  = transpose(pointillisticData.(actualFieldName));
                   end
                end
                
                pointillisticData_table = struct2table(pointillisticData);
                writetable(pointillisticData_table, dataFileFullName);
            else
                error(['The given pointillistic data for saving  is neither table nor structure, but has type of: ', class(pointillisticData)]);
            end
            
        end
        
        
        function infoStruct(directoryPath, infoFileName, analysisParameters)
            % Saves the accompanying information structure of the data into
            % a into an .json info file.
            
            if ~isfolder(directoryPath)
                mkdir(directoryPath)
            end
            
            infoFileFullName = fullfile(directoryPath, infoFileName);
            
            
            info.analysisParameters=analysisParameters;
            
            % this xml stuff is somewhat buggy:
            % disdards struct field with empty cells
            % if a struct elements has a 'ranSTORM' text, it will run
            % rainSTORM.m....
            % create empty file first:
            %fclose(fopen(infoFileFullName, 'w'));
            %xmlManagement.struct2xml( info, infoFileFullName );
            
            jsonManagement.export(info, infoFileFullName);
            
        end
        
        
    end
end

