classdef pointillisticData_selectROI
    % This class contains methods with which ROI can be draw on the figure
    % and the localizations within the ROI can be chosen.

    % TODO image and scatter plot coordinates (are they switched???)!!!!
    % TODO unit of the selected ROI (pixel number of the image or the user set unit!!!!!4)

    methods (Static)

        function actualROI = getActualROI(ROI, index)
           actualROI.type = ROI.type;
           actualROI.parameters = ROI.parameters(index);
        end
        
        function ROI=full(stackSize)

            ROI.type='rectangle';
            ROI.parameters.size.x=stackSize(1);
            ROI.parameters.size.y=stackSize(2);
            ROI.parameters.size.unit='camera pixel length';
            ROI.parameters.center.x=stackSize(1)/2;
            ROI.parameters.center.y=stackSize(2)/2;
            ROI.parameters.center.unit='camera pixel length';
            ROI.parameters.angle.value=0;
            ROI.parameters.angle.unit='degree';


        end

        function ROI=drawUprightRectangle(figureObject, varargin)
            % Draw an unrotated, axis-alignes rectangle on the figure and
            % create a ROI structure containing the information.

            rect=getrect(figureObject);

            [figureForROI, imageUnit]=pointillisticData_selectROI.getFigureForROI(figureObject);

            ROI.type='rectangle';
            ROI.parameters.size.x=rect(4);
            ROI.parameters.size.y=rect(3);
            ROI.parameters.size.unit=imageUnit;
            ROI.parameters.center.x=rect(2)+rect(4)/2;
            ROI.parameters.center.y=rect(1)+rect(3)/2;
            ROI.parameters.center.unit=imageUnit;
            ROI.parameters.angle.value=0;
            ROI.parameters.angle.unit='degree';

            % convert the ROI (if the coordinates of the image were not in any unit)
            ROI=pointillisticData_selectROI.setImageROIUnits(ROI, figureForROI, varargin{:});
        end


        function ROI=drawUprightRectangle_multiple(figureObject, varargin)
            % Draw an unrotated, axis-alignes rectangle on the figure and
            % create a ROI structure containing the information.

            ROI.type='rectangle';
            ROI.parameters = struct([]);

            [figureForROI, imageUnit]=pointillisticData_selectROI.getFigureForROI(figureObject);

            dlgBool=true;                               % open later a question dialog
            idxROI=0;
            while dlgBool

                choice = pointillisticData_selectROI.choicedialog_adjustableRectangle({}, 'Select rectangular ROIs', 'Select another', 'Delete last one', 'Finish');

                switch choice
                    case 'Select another'

                        rect=getrect(figureObject);

                        idxROI=idxROI+1;

                        ROI.parameters(idxROI).size.x=rect(4);
                        ROI.parameters(idxROI).size.y=rect(3);
                        ROI.parameters(idxROI).size.unit=imageUnit;
                        ROI.parameters(idxROI).center.x=rect(2)+rect(4)/2;
                        ROI.parameters(idxROI).center.y=rect(1)+rect(3)/2;
                        ROI.parameters(idxROI).center.unit=imageUnit;
                        ROI.parameters(idxROI).angle.value=0;
                        ROI.parameters(idxROI).angle.unit='degree';

                    case 'Delete last one'
                        ROI.parameters(idxROI)=[];
                    case 'Finish'
                        dlgBool=false;
                    otherwise
                        error('Invalid choice somehow.')
                end

            end

            % convert the ROI (if the coordinates of the image were not in any unit)
            ROI=pointillisticData_selectROI.setImageROIUnits(ROI, figureForROI, varargin{:});
        end

        function ROI=drawAdjustableRectangle_multiple(figureObject, cameraSignalConversion)

            % the first ROI:
            %ROI.type='rectangle';
            %ROI.parameters = struct([]);
            ROI=pointillisticData_selectROI.drawAdjustableRectangle(figureObject, cameraSignalConversion);

            % the subsequent ROIs:
            dlgBool=true;                               % open later a question dialog
            idxROI=1;
            while dlgBool

                choice = pointillisticData_selectROI.choicedialog_adjustableRectangle({}, 'Select rectangular ROIs', 'Select another', 'Delete last one', 'Finish');

                switch choice
                    case 'Select another'

                        ROI_actual=pointillisticData_selectROI.drawAdjustableRectangle(figureObject, cameraSignalConversion);

                        if strcmp(ROI_actual.type,  ROI.type)
                            ROI.parameters = [ROI.parameters, ROI_actual.parameters];
                        else
                            error(["Invalid ROI type was retrieved (", ROI_actual.type, ") when selecting multiple rectangular ROIs."])
                        end

                        idxROI=idxROI+1;


                    case 'Delete last one'
                        ROI.parameters(idxROI)=[];
                    case 'Finish'
                        dlgBool=false;
                    otherwise
                        error('Invalid choice somehow.')
                end

            end

        end


        function ROI=drawAdjustableRectangle(figureObject, cameraSignalConversion)
            %TODO!!!!
            %[ROIFigure, imageUnit]=pointillisticData_selectROI.getFigureForROI(figureObject);
            imageUnit = 'camera pixel length';

            %figure(ROIFigure);

            ax=gca;

            % getting the axis limits after the scatter plot:
            xlimit=get(ax, 'XLim');
            ylimit=get(ax, 'YLim');

            % setting the axis limits back to span only the scatter plot:
            set(gca, 'XLim', xlimit, 'YLim', ylimit);


            imagePixelSize_temp.x=1;
            imagePixelSize_temp.y=1;
            imagePixelSize_temp.unit=imageUnit;
            ROI_initial=pointillisticData_selectROI.drawUprightRectangle(figureObject, imagePixelSize_temp);
%             ROI_initial.parameters.center.unit='image pixel length';
%             ROI_initial.parameters.size.unit='image pixel length';

            % the remnant parameters, the center position of the rectangular region
            xCenter_initial=ROI_initial.parameters.center.x;
            yCenter_initial=ROI_initial.parameters.center.y;

            unit=ROI_initial.parameters.center.unit;

            size_X=ROI_initial.parameters.size.x;
            size_Y=ROI_initial.parameters.size.y;

            angle=unitConversion.convertScalarQuantity(ROI_initial.parameters.angle, 'radian', cameraSignalConversion);
            alpha=-angle.value;

            displacement=[0,0];

            dlgBool=true;                               % open later a question dialog
            while dlgBool

                ROI.type='rectangle';
                ROI.parameters.size.x=size_X;
                ROI.parameters.size.y=size_Y;
                ROI.parameters.size.unit=ROI_initial.parameters.size.unit;
                % the displacement vector starting and ending coordinates:
                ROI.parameters.center.x=xCenter_initial+displacement(2); % starting and ending "x" coordinates of the arrow
                ROI.parameters.center.y=yCenter_initial+displacement(1); % starting and ending "y" coordinates of the arrow
                ROI.parameters.center.unit=ROI_initial.parameters.center.unit;
                ROI.parameters.angle.unit=ROI_initial.parameters.angle.unit;
                ROI.parameters.angle.value=unitConversion.convertValue(alpha, 'radian', ROI.parameters.angle.unit, cameraSignalConversion);

                [x,y] = pointillisticData_selectROI.rectangleSamplingPoints(ROI.parameters.center, ROI.parameters.size, ROI.parameters.angle, imageUnit, cameraSignalConversion);

                figure(figureObject)
                hold on;
                h=plot(x, y, 'color', 'c');   % plotting the rectangle bounding the displacement region

                % older version with fewer buttons: questdlg
                [choice] = pointillisticData_selectROI.choicedialog_adjustableRectangle({...
                    ['Current angle of the rectangle: ', num2str(alpha), '°'];...
                    ['Current [x, y] displacement the rectangle: ', num2str(displacement), ' ', unit];...
                    ['Current length of the rotated rectangle: ', num2str(size_Y), ' ', unit];...
                    ['Current height of the rotated rectangle: ', num2str(size_X), ' ', unit]}, ...
                    'Parameters of the rotated rectangle', 'Quit', 'Accept parameters', 'Set parameters');

                switch choice
                    case 'Accept parameters'
                        dlgBool=false;
                        continue
                    case 'Set parameters'
                        prompt = {'Angle of the rectangle: ', '[x, y] displacement the rectangle:', 'Length of the rotated rectangle:', 'Height of the rotated rectangle:'}; % name of the input fields
                        dlgTitle = 'Set a rectangular regionof interest';   % title of the dialog
                        nLines = 1;     % one line for each input fields
                        % the default values on the dialog
                        alpha_degrees=unitConversion.convertValue(alpha, 'radian', 'degree', cameraSignalConversion);
                        defaultAns = {num2str(alpha_degrees), num2str(displacement), num2str(size_Y), num2str(size_X)};
                        % getting the new parameters
                        answer = inputdlg(prompt,dlgTitle,nLines,defaultAns);  % the set "z" boundary values in a cell of strings
                        alpha_degrees=str2double(char(answer{1}));
                        alpha=unitConversion.convertValue(alpha_degrees, 'degree', 'radian', cameraSignalConversion);
                        displacement=str2num(answer{2});
                        size_Y=str2double(answer{3});
                        size_X=str2double(answer{4});

                    case 'Quit'
                        delete(h)
                        error('Quitting was requested.');
                end

                % delete the plotted rectangle:
                delete(h)

            end

            %ROI.parameters=unitConversion.convertImagePixelROI(ROI.parameters, imagePixelSize);
        end


        function ROI=selectPoints_ellipticROIs(figureObject, size_x, size_y, size_unit, angle_value, angle_unit, cameraSignalConversion, varargin)
            % This function lets the user select multiple points on a
            % figure and creates elliptic ROIs around these points with the given size and angle.

            [y, x]=getpts(figureObject);

            numberOfPoints=numel(x);

            [figureForROI, imageUnit]=pointillisticData_selectROI.getFigureForROI(figureObject);
            figure(figureObject)

            ROI.type='ellipse';
            ROI.parameters = struct([]);
            for idxPoint=1:numberOfPoints
                ROI.parameters(idxPoint).size.x=size_x;
                ROI.parameters(idxPoint).size.y=size_y;
                ROI.parameters(idxPoint).size.unit=size_unit;
                % TODO:rotated ellipse:
                ROI.parameters(idxPoint).angle.value=angle_value;
                ROI.parameters(idxPoint).angle.unit=angle_unit;

                ROI.parameters(idxPoint).center.x=x(idxPoint);
                ROI.parameters(idxPoint).center.y=y(idxPoint);
                ROI.parameters(idxPoint).center.unit=imageUnit;

                figure(figureObject)
                hold on;
                [ellipse_x, ellipse_y] = pointillisticData_selectROI.ellipseSamplingPoints(ROI.parameters(idxPoint).center, ROI.parameters(idxPoint).size, ROI.parameters(idxPoint).angle, imageUnit, cameraSignalConversion);
                plot(ellipse_x, ellipse_y, 'color', 'c')

            end

            %figure(figureObject)
            %hold on;
            %scatter(y, x, 'x', 'g');

            % set the ROI units
            ROI=pointillisticData_selectROI.setImageROIUnits(ROI, figureForROI, varargin{:});

            figure(figureForROI)
        end


        function ROI=selectPoints_doubleEllipticROIs(figureObject, size_x, size_y, size_unit, angle_value, angle_unit, varargin)
            % This function lets the user select multiple points on a
            % figure and creates elliptic ROIs around these points with the given size and angle.

            ROI.type='double ellipse';
            ROI.parameters = struct([]);
            while(true)

                [y, x]=getpts(figureObject);

                figure(figureObject)
                hold on;
                scatter(y, x, 'x', 'g');

                numberOfPoints=numel(x);

                if(numberOfPoints == 0)
                    % breaking out with Enter or Return without
                    break;
                elseif(numberOfPoints ~= 2)
                    % in case of incorrect mouse clicks, try again
                    warning('Invalid number of points were selected. Please select 2 point sequantial left and right clicks.')
                    continue;
                end

                [figureForROI, imageUnit]=pointillisticData_selectROI.getFigureForROI(figureObject);

                ROI_index = numel(ROI.parameters) + 1;

                ROI.parameters(ROI_index).size.x=size_x;
                ROI.parameters(ROI_index).size.y=size_y;
                ROI.parameters(ROI_index).size.unit=size_unit;
                % TODO:rotated ellipse:
                ROI.parameters(ROI_index).angle.value=angle_value;
                ROI.parameters(ROI_index).angle.unit=angle_unit;

                ROI.parameters(ROI_index).center.x(1)=x(1);
                ROI.parameters(ROI_index).center.x(2)=x(2);
                ROI.parameters(ROI_index).center.y(1)=y(1);
                ROI.parameters(ROI_index).center.y(2)=y(2);
                ROI.parameters(ROI_index).center.unit=imageUnit;
            end

            % set the ROI units
            ROI=pointillisticData_selectROI.setImageROIUnits(ROI, figureForROI, varargin{:});
        end


        function area=calculateArea(ROI_structure)
            % This function caculates the area of the given ROI.

            % calculate the area:
            switch ROI_structure.type
                case 'rectangle'
                    area.value=ROI_structure.parameters.size.x*ROI_structure.parameters.size.y;
                case 'ellipse'
                    area.value = ROI_structure.parameters.size.x/2*ROI_structure.parameters.size.y/2*pi;
                case 'double ellipse'
                    area.value = 2 * ROI_structure.parameters.size.x/2*ROI_structure.parameters.size.y/2*pi;
                case 'polygon'
                    area.value = polyarea(ROI_structure.parameters.vertices.x, ROI_structure.parameters.vertices.y);
                otherwise
                    error('Unknown ROI type.')
            end

            % determine the unit the area should be given in:
            switch ROI_structure.parameters.size.unit
                case 'camera pixel length'
                    area.unit='camera pixel area';
                case 'image pixel length'
                    error('Convert the first from the image pixel values.')
                otherwise
                    area.unit=[ROI_structure.parameters.size.unit, '^2'];
            end

        end


        function filterBoolVect=cutROI(pointillisticData, ROI_structure, cameraSignalConversion)
            % Select the localizations within the given ROI.
            % Returns a boolean vector which contains which localizations
            % are inside the ROI.

            ROI_structure.parameters=unitConversion.convertROI(ROI_structure.parameters, 'camera pixel length', cameraSignalConversion);

            % TODO: get the data number properly, not with "x_coord"
            filterBoolVect = false(numel(pointillisticData.x_coord), 1);

            ROI_number=numel(ROI_structure.parameters);

            switch ROI_structure.type
                case 'rectangle'

                    for idxROI=1:ROI_number

                        filteringThresholds(1).field = 'x_coord';
                        filteringThresholds(1).bounds=[...
                            ROI_structure.parameters(idxROI).center.x-ROI_structure.parameters(idxROI).size.x/2,...
                            ROI_structure.parameters(idxROI).center.x+ROI_structure.parameters(idxROI).size.x/2];
                        filteringThresholds(2).field = 'y_coord';
                        filteringThresholds(2).bounds=[...
                            ROI_structure.parameters(idxROI).center.y-ROI_structure.parameters(idxROI).size.y/2,...
                            ROI_structure.parameters(idxROI).center.y+ROI_structure.parameters(idxROI).size.y/2];

                        % rotate the localizations, so the ROI will be an
                        % upright rectangle:
                        dataRotation=ROI_structure.parameters(idxROI).angle;
                        dataRotation.value=-dataRotation.value;
                        pointillisticData = pointillisticData_coordinateTransformation.rotate_XY(pointillisticData, dataRotation, ROI_structure.parameters(idxROI), cameraSignalConversion);

                        filterBoolVect_act=pointillisticData_filtering.getBooleanVect(pointillisticData, filteringThresholds);

                        filterBoolVect=filterBoolVect | filterBoolVect_act;

                    end
                case 'ellipse'
                    % TODO: it is only unrotated ellipse
                    ROI_structure.parameters=unitConversion.convertROI(ROI_structure.parameters, 'camera pixel length', cameraSignalConversion);

                    for idxROI=1:ROI_number

                        filterBoolVect_act=...
                            (pointillisticData.x_coord-ROI_structure.parameters(idxROI).center.x).^2/(ROI_structure.parameters(idxROI).size.x/2)^2+...
                            (pointillisticData.y_coord-ROI_structure.parameters(idxROI).center.y).^2/(ROI_structure.parameters(idxROI).size.y/2)^2<=...
                            1;

                        filterBoolVect=filterBoolVect | filterBoolVect_act;

                    end
                case 'double ellipse'
                    % TODO: it is only unrotated ellipse
                    ROI_structure.parameters=unitConversion.convertROI(ROI_structure.parameters, 'camera pixel length', cameraSignalConversion);

                    for idxROI=1:ROI_number

                        filterBoolVect_act_1=...
                            (pointillisticData.x_coord-ROI_structure.parameters(idxROI).center.x(1)).^2/(ROI_structure.parameters(idxROI).size.x/2)^2+...
                            (pointillisticData.y_coord-ROI_structure.parameters(idxROI).center.y(1)).^2/(ROI_structure.parameters(idxROI).size.y/2)^2<=...
                            1;
                        filterBoolVect_act_2=...
                            (pointillisticData.x_coord-ROI_structure.parameters(idxROI).center.x(2)).^2/(ROI_structure.parameters(idxROI).size.x/2)^2+...
                            (pointillisticData.y_coord-ROI_structure.parameters(idxROI).center.y(2)).^2/(ROI_structure.parameters(idxROI).size.y/2)^2<=...
                            1;

                        filterBoolVect=filterBoolVect | filterBoolVect_act_1 | filterBoolVect_act_2;

                    end
                case 'polygon'
                    for idxROI=1:ROI_number
                        %pointillisticData.x_coord
                        %pointillisticData.y_coord
                        %ROI_structure(idxROI).parameters.vertices.x, ROI_structure(idxROI).parameters.vertices.y
                        [in,on] = inpolygon(...
                            pointillisticData.x_coord, ...
                            pointillisticData.y_coord, ...
                            ROI_structure(idxROI).parameters.vertices.x, ...
                            ROI_structure(idxROI).parameters.vertices.y);
                        
                            filterBoolVect=filterBoolVect | in | on;
                    end
                otherwise
                    error('Unknown ROI type.')
            end
        end


        function pointillisticData_filtered = filterROI(pointillisticData, ROI_structure, cameraSignalConversion)
            % Filters the pointillistic data according to the selected ROIs.

            filterBoolVect = pointillisticData_selectROI.cutROI(pointillisticData, ROI_structure, cameraSignalConversion);

            pointillisticData_filtered = pointillisticData_filtering.filter(pointillisticData, filterBoolVect);

        end



        function MBR_bounds=getMinimumBoundingRectangle(ROI_structure)
            % This function return the upper and loxer "x" and "y" bounds
            % of the minimum bounding rectangle of the given ROI.

            % TODO: this function is not final, the rotation within it and
            % the unit conversion is not correct

            switch ROI_structure.type
                case 'rectangle'
                    % let the center be zero:
                    ROI.center = struct();
                    ROI.center.x=0;
                    ROI.center.y=0;
                    ROI.center.unit=ROI_structure.parameters.size.unit;
                    angle=ROI_structure.parameters.angle;
                    % the coordinates of the vertices of the unrotated ROI:
                    vertexCoordinates.x_coord=[ROI_structure.parameters.size.x/2, ROI_structure.parameters.size.x/2, -ROI_structure.parameters.size.x/2, -ROI_structure.parameters.size.x/2];
                    vertexCoordinates.y_coord=[ROI_structure.parameters.size.y/2, -ROI_structure.parameters.size.y/2. -ROI_structure.parameters.size.y/2, ROI_structure.parameters.size.y/2];
                    % rotate the verties to get the centered ROI:
                    % TODO: creating dumy variable is not proper solution:
                    cameraSignalConversion.pixelSize_nm=1;
                    cameraSignalConversion.exposureTime_ms=1;
                    cameraSignalConversion.frameInterval_ms=1;
                    cameraSignalConversion.countsPerPhoton = 1;
                    cameraSignalConversion.countsPerElectron = 1;
                    vertexCoordinates = pointillisticData_coordinateTransformation.rotate_XY(vertexCoordinates, angle, ROI, cameraSignalConversion);

                    % get the minimum bounding rectangle (centered):
                    centered_MBR_bounds.x=[max(vertexCoordinates.x_coord), min(vertexCoordinates.x_coord)];
                    centered_MBR_bounds.y=[max(vertexCoordinates.y_coord), min(vertexCoordinates.y_coord)];
                    centered_MBR_bounds.unit=ROI_structure.parameters.size.unit;
                case 'ellipse'
                    % TODO: it is only onrotated ellipse
                    % get the minimum bounding rectangle
                    centered_MBR_bounds.x=ROI_structure.parameters.center.x+0.5*[-ROI_structure.parameters.size.x, ROI_structure.parameters.size.x];
                    centered_MBR_bounds.y=ROI_structure.parameters.center.y+0.5*[-ROI_structure.parameters.size.y, ROI_structure.parameters.size.y];
                    centered_MBR_bounds.unit=ROI_structure.parameters.size.unit;
                case 'polygon'
                    pshape = polyshape(...
                            ROI_structure(idxROI).parameters.vertices.x, ...
                            ROI_structure(idxROI).parameters.vertices.y);
                    [xlim,ylim] = boundingbox(pshape);
                    centered_MBR_bounds.x=xlim;
                    centered_MBR_bounds.y=ylim;
                    centered_MBR_bounds.unit=ROI_structure.parameters.size.unit;
                otherwise
                    error('Unknowk ROI type.')
            end

            % convert the ROI centr to the proper units:
            MBR_center=unitConversion.convert2DVectorQuantity(ROI_structure.parameters.center, centered_MBR_bounds.unit, cameraSignalConversion);

            % get the minimum bounding rectangle:
            MBR_bounds.x=centered_MBR_bounds.x+MBR_center.x;
            MBR_bounds.y=centered_MBR_bounds.y+MBR_center.y;
            MBR_bounds.unit=centered_MBR_bounds.unit;
        end

    end

    methods(Static, Access=private)

        function [choice] = choicedialog_adjustableRectangle(stringCell, title, btn1_string, btn2_string, btn3_string)
            % Dialog that lets th user set the initial line and set the region to be
            % cut out.

            %  source:
            % https://www.mathworks.com/help/matlab/ref/dialog.html

            % some of the window geometry's definition:
            windowWidth=500;
            windowHeight=200;
            buttonMargin=30;
            buttonWidth=150;
            buttonHeight=25;


            d = dialog('Position',[300 300 windowWidth windowHeight],'Name',title);

            txt = uicontrol('Parent',d,...
                'Style','text',...
                'Position',[20 20+60 windowWidth-20 windowHeight-2*20-60],...
                'HorizontalAlignment', 'Left',...
                'String',stringCell);



            btn1 = uicontrol('Parent',d,...
                'Style', 'pushbutton',...
                'Position',[buttonMargin 20 buttonWidth buttonHeight],...
                'String',btn1_string,...
                'Callback',@btn1_callback);

            btn2 = uicontrol('Parent',d,...
                'Style', 'pushbutton',...
                'Position',[windowWidth-buttonWidth-buttonMargin 20 buttonWidth buttonHeight],...
                'String',btn2_string,...
                'Callback',@btn2_callback);

            btn3 = uicontrol('Parent',d,...
                'Style', 'pushbutton',...
                'Position',[windowWidth/2-buttonWidth/2 60 buttonWidth buttonHeight],...
                'String',btn3_string,...
                'Callback',@btn3_callback);


            % Wait for d to close before running to completion
            uiwait(d);

            function btn1_callback(source,event)
                %choice = char(popup_items(idx,:));
                choice = get(btn1, 'String');
                delete(gcf)
            end

            function btn2_callback(source,event)
                choice = get(btn2, 'String');
                delete(gcf)
            end

            function btn3_callback(source,event)
                choice = get(btn3, 'String');
                delete(gcf)
            end

        end

        function [ROIFigure, imageUnit]=getFigureForROI(figureObject)

            imageHandlesArray = imhandles(figureObject);
            % if there are multiple images on the figure, choose the first
            % one
            idxImage=1;

            userData=get(imageHandlesArray(idxImage), 'UserData');

            ROIFigure=figure('Visible', 'Off');
            copyobj(imgca(figureObject), ROIFigure)

            imageUnit='';
            if isstruct(userData)
               if isfield(userData, 'unit')
                   imageUnit=userData.unit;


               else
                   % let the "imageUnit" be empty
               end
            else
                % let the "imageUnit" be empty
            end

            if isempty(imageUnit)
                set(imhandles(ROIFigure), 'XData', [])
                set(imhandles(ROIFigure), 'YData', [])

                imageUnit='image pixel length';
            end

        end


        function ROI=setImageROIUnits(ROI, figureObject, varargin)

            imageHandlesArray = imhandles(figureObject);
            % if there are multiple images on the figure, choose the first
            % one
            idxImage=1;

            % get the image coordinate units, if its is set:
            userData=get(imageHandlesArray(idxImage), 'UserData');
            imageUnit='';
            if isstruct(userData)
                if isfield(userData, 'unit')
                    imageUnit=userData.unit;

                else
                    % let the "imageUnit" be empty
                end
            else
                % let the "imageUnit" be empty
            end

            % if the image coordinate units were not set, convert the ROI to
            % raw image pixel lengths:
            if isempty(imageUnit)
                XData=get(imageHandlesArray(idxImage), 'XData');
                YData=get(imageHandlesArray(idxImage), 'YData');
                imageSize=size(get(imageHandlesArray(idxImage), 'CData'));
                rowNumber=imageSize(1);
                columnNumber=imageSize(2);

                % TODO: set the angle too

                imageUnit='image pixel length';

                ROI_number = numel(ROI.parameters);

                for idxROI = 1:ROI_number
                    ROI.parameters(idxROI).size.x=(ROI.parameters(idxROI).size.x-XData(1))/(XData(2)-XData(1))*(columnNumber-1);
                    ROI.parameters(idxROI).size.y=(ROI.parameters.size(idxROI).y-YData(1))/(YData(2)-YData(1))*(rowNumber-1);
                    ROI.parameters(idxROI).center.x=(ROI.parameters.center(idxROI).x-XData(1))/(XData(2)-XData(1))*(columnNumber-1);
                    ROI.parameters(idxROI).center.y=(ROI.parameters.center(idxROI).y-YData(1))/(YData(2)-YData(1))*(rowNumber-1);

                    ROI.parameters(idxROI).size.unit=imageUnit;
                    ROI.parameters(idxROI).center.unit=imageUnit;
                end

            end


            % check if the units of ROI has to be converted:
            if numel(varargin)==0
                % do nothing
            elseif numel(varargin)==1
                % conert the ROI from the raw image coordinates

                % get the user given structure that defines the conversion
                imagePixelSize=varargin{1};

                if strcmp(imageUnit, 'image pixel length')
                    % convert the ROI units:
                    ROI.parameters=unitConversion.convertImagePixelROI(ROI.parameters, imagePixelSize);
                else
                    % no conversion needed, the structure for the
                    % conversion was given in futile
                    warning('The image coordinates were already given in some unit, not converting them from the raw image pixel lengths');
                end
            else
               error('Invalid number of input arguments');
            end
        end

        function [x,y] = rectangleSamplingPoints(center, size, rotationAngle, imageUnit, cameraSignalConversion)

            center_x = unitConversion.convertValue(center.x, center.unit, imageUnit, cameraSignalConversion);
            center_y = unitConversion.convertValue(center.y, center.unit, imageUnit, cameraSignalConversion);
            size_x = unitConversion.convertValue(size.x, size.unit, imageUnit, cameraSignalConversion);
            size_y = unitConversion.convertValue(size.y, size.unit, imageUnit, cameraSignalConversion);
            angle = unitConversion.convertValue(rotationAngle.value, rotationAngle.unit, 'radian', cameraSignalConversion);

            % calculate the coordiantes of the rectangle to be plotted:
            y=zeros(5,1);               % "x" coordinates of the vertices of the recttangle bounding the displacement region
            y(1)=center_x+(cos(angle)*(-size_x/2)+sin(angle)*(+size_y/2));
            y(2)=center_x+(cos(angle)*(+size_x/2)+sin(angle)*(+size_y/2));
            y(3)=center_x+(cos(angle)*(+size_x/2)+sin(angle)*(-size_y/2));
            y(4)=center_x+(cos(angle)*(-size_x/2)+sin(angle)*(-size_y/2));
            y(5)=center_x+(cos(angle)*(-size_x/2)+sin(angle)*(+size_y/2));
            x=zeros(5,1);               % "y" coordinates of the vertices of the rectangle bounding the displacement region
            x(1)=center_y+(-sin(angle)*(-size_x/2)+cos(angle)*(+size_y/2));
            x(2)=center_y+(-sin(angle)*(+size_x/2)+cos(angle)*(+size_y/2));
            x(3)=center_y+(-sin(angle)*(+size_x/2)+cos(angle)*(-size_y/2));
            x(4)=center_y+(-sin(angle)*(-size_x/2)+cos(angle)*(-size_y/2));
            x(5)=center_y+(-sin(angle)*(-size_x/2)+cos(angle)*(+size_y/2));

        end

        function [x,y] = ellipseSamplingPoints(center, size, rotationAngle, imageUnit, cameraSignalConversion)
            % Calculates smapling point on a rotated ellipse to be drawn on
            % a figure.

            center_x = unitConversion.convertValue(center.x, center.unit, imageUnit, cameraSignalConversion);
            center_y = unitConversion.convertValue(center.y, center.unit, imageUnit, cameraSignalConversion);
            longAxis = unitConversion.convertValue(size.x, size.unit, imageUnit, cameraSignalConversion);
            shortAxis = unitConversion.convertValue(size.y, size.unit, imageUnit, cameraSignalConversion);
            angle = unitConversion.convertValue(rotationAngle.value, rotationAngle.unit, 'radian', cameraSignalConversion);

            samplingAngles = (0:20:360)*pi/180;

            y = center_x + 0.5 * longAxis * cos(samplingAngles) * cos(angle) - 0.5 * shortAxis * sin(samplingAngles) * sin(angle);
            x = center_y + 0.5 * longAxis * cos(samplingAngles) * sin(angle) + 0.5 * shortAxis * sin(samplingAngles) * cos(angle);

        end
    end
end
