classdef unitConversion
    % This is a collection of functions for converting various quantities,
    % which can be a simple number or a struct with predefined fields
    % correcponding to the data fields and the field holding the
    % information of its unit ((vector or scalar) value - unit pairs).
    
    
    methods (Static)
        
        function [outputValue, conversionFactor] = convertValue(inputValue, unitString_input, unitString_output, cameraSignalConversion)
            % This function converts an input value with th given
            % (combined) unit to an output value with diffrent units
            % (dimensions must be he same, of course).
            % The order the simple units within the combined input and
            % output units with respect to the dimensions must be the same.
            
            % divide the combined input and outputs to simple individual
            % units:
            [multiplicativeUnits_input, divisiveUnits_input]=unitConversion.getSeperatedUnits(unitString_input);
            [multiplicativeUnits_output, divisiveUnits_output]=unitConversion.getSeperatedUnits(unitString_output);
            
            % check the number of the simple input and output units:
            if (...
                    numel(multiplicativeUnits_input)~=numel(multiplicativeUnits_output) || ...
                    numel(divisiveUnits_input)~=numel(divisiveUnits_output))
                error('The input and output dimensions does not match')
            end
            
            % initialize the conversion factor:
            conversionFactor=1;
            
            % go through the multiplicative units:
            for idxMultiplicative=1:numel(multiplicativeUnits_input)
                % get the conversion factor for the actual unit:
                conversionFactor_actualUnit=unitConversion.getConversionFactor(multiplicativeUnits_input{idxMultiplicative}, multiplicativeUnits_output{idxMultiplicative}, cameraSignalConversion);
                % modify the conversion factor with that of the curect
                % unit:
                conversionFactor=conversionFactor*conversionFactor_actualUnit;
            end
            
            % go through the divisive units:
            for idxDivisive=1:numel(divisiveUnits_input)
                % get the conversion factor for the actual unit:
                conversionFactor_actualUnit=unitConversion.getConversionFactor(divisiveUnits_input{idxDivisive}, divisiveUnits_output{idxDivisive}, cameraSignalConversion);
                % modify the conversion factor with that of the curect
                % unit:
                conversionFactor=conversionFactor/conversionFactor_actualUnit;
            end
            
            % calculate the output value with the conversion factor:
            outputValue=inputValue*conversionFactor;
            
        end
        
        
        function [outputQuantity, conversionFactor] = convertScalarQuantity(inputQuantity, unitString_output, cameraSignalConversion)
            % This value converts a quantity ("value" and "unit" pair in the same data structure) to a
            % given unit.
            
            % converting the value:
            [outputValue, conversionFactor] = unitConversion.convertValue(inputQuantity.value, inputQuantity.unit, unitString_output, cameraSignalConversion);
            
            % passing the new quantity (value and unit):
            outputQuantity.value=outputValue;
            outputQuantity.unit=unitString_output;
        end
        
        
        function [outputVector, conversionFactor]=convert2DVectorQuantity(inputVector, unitString_output, cameraSignalConversion)
            % This function converts the ROI sizes the the given units. The
            % ROI must be in a predefined unit.
            
            
            % pass the ROI sizes:
            inputValue_x=inputVector.x;
            inputValue_y=inputVector.y;
            % the ROI should be in a predefined unit (it is not checked):
            unitString_input=inputVector.unit;
            
            % get the "x" and "y" ROI sizes in the desired unit:
            [outputValue_x, conversionFactor] = unitConversion.convertValue(inputValue_x, unitString_input, unitString_output, cameraSignalConversion);
            [outputValue_y, ~] = unitConversion.convertValue(inputValue_y, unitString_input, unitString_output, cameraSignalConversion);
            
            % return the ROI in the new units:
            outputVector.x=outputValue_x;
            outputVector.y=outputValue_y;
            outputVector.unit=unitString_output;
            
        end
        
        
        function [newInputROIParameters]=convertImagePixelROI(inputROIParameters, pixelSize)
            % This function converts the ROI quantities (size, center)
            % given in image pixels to the given units.
            
            ROI_number=numel(inputROIParameters);
            
            for idxROI=ROI_number:-1:1
            
                newInputROIParameters(idxROI)=inputROIParameters(idxROI);

                % if the ROI size is in image pixel units:
                if strcmp('image pixel length', inputROIParameters.size.unit)
                    newInputROIParameters.size(idxROI).x=inputROIParameters(idxROI).size.x*pixelSize.x;
                    newInputROIParameters.size(idxROI).y=inputROIParameters(idxROI).size.y*pixelSize.y;
                    newInputROIParameters.size.unit=pixelSize.unit;

                else
                   error('To use this method, the given ROI must be in image pixel numbers.') 
                end

                % if the ROI size is in image pixel units:
                if strcmp('image pixel length', inputROIParameters.center.unit)
                    newInputROIParameters.center(idxROI).x=inputROIParameters(idxROI).center.x*pixelSize.x;
                    newInputROIParameters.center(idxROI).y=inputROIParameters(idxROI).center.y*pixelSize.y;
                    newInputROIParameters.center.unit=pixelSize.unit;

                else
                   error('To use this method, the given ROI must be in image pixel numbers.') 
                end

            end
            
        end
        
        
        function [outputROIParameters, conversionFactor]=convertROI(inputROIParameters, unitString_output, cameraSignalConversion)
            % This function converts the ROI quantities (size, center, vertices)
            % to the given units.
            % TODO: make it more general by considering any field with
            % length units
            
            ROI_number=numel(inputROIParameters);
            
            
            for idxROI=ROI_number:-1:1
            
                outputROIParameters(idxROI)=inputROIParameters(idxROI);
                
                paramtersFieldNames = fieldnames(outputROIParameters(idxROI));
                
                conversionFactorStruct = struct();
                for idxField = 1:numel(paramtersFieldNames)
                    fieldName = paramtersFieldNames{idxField};
                    if sum(strcmp(fieldName, {'size', 'center', 'vertices'})>0)
                        [outputROIParameters(idxROI).(fieldName), conversionFactor]=unitConversion.convert2DVectorQuantity(inputROIParameters(idxROI).(fieldName), unitString_output, cameraSignalConversion);
                        conversionFactorStruct.(fieldName) = conversionFactor;
                    end

                end
                
            end
            
        end
        
        
        function conversionFactor=getConversionFactor(inputUnit, outputUnit, cameraSignalConversion)
            % This function calculates the conversion factor with which the
            % given input unit can be converted to the given output unit.
            
            % get the cell containing all the units and their conversion
            % factors to the base units for every dimension:
            dimensionsCell=unitConversion.getConversionFactorsToBase(cameraSignalConversion);
            
            % names of the defined dimensions:
            dimensionNames=dimensionsCell(:,1);
            
            % initialize the conversion factor:
            conversionFactor=[];
            
            % go though the dimensions and check whether the given unit
            % belongs there:
            for idxDimension=1:numel(dimensionNames)
                unitsCell=dimensionsCell{idxDimension,2};
                % conversion factor for the input unit to the base unit:
                [conversionFactor_input, booleanVect_input]=unitConversion.findUnitInCell(unitsCell, inputUnit, cameraSignalConversion);
                % conversion factor for the output unit to the base unit:
                [conversionFactor_output, booleanVect_output]=unitConversion.findUnitInCell(unitsCell, outputUnit,cameraSignalConversion);
                
                if sum(booleanVect_input)==0 && sum(booleanVect_output)==0
                    %  if this dimension does not contain the given units,
                    %  proceed to the next one:
                    continue;
                elseif sum(booleanVect_input)==1 && sum(booleanVect_output)==1
                    % conversion factor from the input unit to the output
                    % unit:
                    conversionFactor=conversionFactor_input/conversionFactor_output;
                    break;
                else
                    error('Probably wrong input and output unit pairs were given for the input ("%s") and for the output ("%s").', inputUnit, outputUnit)
                end
                
            end
            
            if isempty(conversionFactor)
                % the unit conversion failed, trying to provide information
                % why
                
                if isempty(cameraSignalConversion)
                    % check whether the user wanted to convert to or
                    % convert from a unit that needs the "camera
                    % properties" structure  
                    
                    % create a dummy camera properties structure
                    cameraSignalConversion.pixelSize_nm=1;
                    cameraSignalConversion.frameInterval_ms=1;
                    cameraSignalConversion.exposureTime_ms=1;
                    cameraSignalConversion.countsPerPhoton=1;
                    cameraSignalConversion.countsPerElectron=1;
                    
                    % get the conversion factor with the dummy structure
                    conversionFactor=unitConversion.getConversionFactor(inputUnit, outputUnit, cameraSignalConversion);
                    
                    if ~isempty(conversionFactor)
                        error('Empty "acquisition properties" struture was given and you wanted to convert to/from a camera related unit. Please give a valid "camera properties" structure.');
                    else
                        error('This condition should never be met. Check it!')
                    end
                    
                else
                    error('Could not perform the unit conversion as the requested units are not defined. Please give valid units or define the new units.')
                end
            else
               % do nothing 
            end
            
            
        end
        
    end
    
    
    methods (Static, Access=private)
        
        function [conversionFactor, booleanVect]=findUnitInCell(dimensionCell, unit, cameraSignalConversion)
            % This function searches an unit in the cell contaning the
            % units of a dimension and returns the conversion factor.
            
            % find the given unit among the units of the given dimension:
            booleanVect=strcmp(dimensionCell(:,1), unit);
            if sum(booleanVect)==1
                % get the conversion factor belonging to the unit:
                functionHandle=dimensionCell{booleanVect,2};
                conversionFactor=functionHandle(cameraSignalConversion);
            elseif sum(booleanVect)>1
                error('Strange error with the units')
            else
                % if the unit does not belong to the dimension, return an empty cnversion factor: 
                conversionFactor=[]; 
            end
            
        end
        
        
        function dimensionsCell=getConversionFactorsToBase(cameraSignalConversion)
            % This function contains all the conversion factors for all the units to
            % the base units. The units are arranged according to the
            % dimensions they belong to. 
            
            % the dimensions the algorithm might work with:
            dimensionsCell={...
                'length', cell(0);...
                'area', cell(0);...
                'time', cell(0);...
                'camera signal', cell(0);...
                'angle', cell(0)};
            
            % this if structure i reuired if one wants to use the unit
            % conversion without defining the "acquisition properties",
            % i.e. one wnts to convert to/from units that do not involve
            % any camera related quantities
            if ~isempty(cameraSignalConversion)
                % uits of length:
                dimensionsCell{1,2}={...
                    'camera pixel length', @(cameraSignalConversion) 1;...
                    'nm', @(cameraSignalConversion) 1/cameraSignalConversion.pixelSize_nm;...
                    'um', @(cameraSignalConversion) 1000/cameraSignalConversion.pixelSize_nm};
                % units of area:
                dimensionsCell{2,2}={...
                    'camera pixel area', @(cameraSignalConversion) 1;...
                    'nm^2', @(cameraSignalConversion) (1/cameraSignalConversion.pixelSize_nm)^2;...
                    'um^2', @(cameraSignalConversion) (1000/cameraSignalConversion.pixelSize_nm)^2};
                % units of time:
                dimensionsCell{3,2}={...
                    'camera exposure time', @(cameraSignalConversion) 1;...
                    'camera frame interval', @(cameraSignalConversion) cameraSignalConversion.frameInterval_ms/cameraSignalConversion.exposureTime_ms;...
                    'ms', @(cameraSignalConversion) 1/cameraSignalConversion.exposureTime_ms;...
                    's', @(cameraSignalConversion) 1000/cameraSignalConversion.exposureTime_ms};
                % units of photon number:
                dimensionsCell{4,2}={...
                    'camera count', @(cameraSignalConversion) 1;...
                    'photon', @(cameraSignalConversion) cameraSignalConversion.countsPerPhoton;...
                    'photoelectron', @(cameraSignalConversion) cameraSignalConversion.countsPerElectron};
                % units of angle:
                dimensionsCell{5,2}={...
                    'degree', @(cameraSignalConversion) 1;...
                    'radian', @(cameraSignalConversion) 180/pi};
            else
                % uits of length:
                dimensionsCell{1,2}={...
                    'nm', @(cameraSignalConversion) 1;...
                    'um', @(cameraSignalConversion) 1000};
                % units of area:
                dimensionsCell{2,2}={...
                    'nm^2', @(cameraSignalConversion) (1)^2;...
                    'um^2', @(cameraSignalConversion) (1000)^2};
                % units of time:
                dimensionsCell{3,2}={...
                    'ms', @(cameraSignalConversion) 1;...
                    's', @(cameraSignalConversion) 1000};
                % units of photon number:
                dimensionsCell{4,2}={...
                    };
                % units of angle:
                dimensionsCell{5,2}={...
                    'degree', @(cameraSignalConversion) 1;...
                    'radian', @(cameraSignalConversion) 180/pi};
            end
            
        end
        
        
        function [multiplicativeUnits, divisiveUnits]=getSeperatedUnits(unitString)
            % This function returns the individual units from a given
            % string representing a combined unit. The individual units
            % must be seperated with "*" or "/" signs. No bracket
            % supported.
            
            % divide the unit string by the multiplication sign:
            tempCell=strsplit( unitString , '*' );
            
            % number of multiplicative units:
            multiplicativeN=numel(tempCell);
            
            % initialise the cells for storing the multiplicative and
            % divisive units:
            multiplicativeUnits=cell(1, multiplicativeN);
            divisiveUnits=cell(0);
            
            % go through the already seperated units:
            for idxMultiplicative=1:multiplicativeN
                % divide the unit string by the division sign:
                tempCell2=strsplit( tempCell{idxMultiplicative} , '/' );
                
                % the first element belongs to multiplication:
                multiplicativeUnits{idxMultiplicative}=tempCell2{1};
                % the other elements belong to multiplication:
                divisiveN=numel(tempCell2)-1;
                if divisiveN>0
                    prevDivisionalN=numel(divisiveUnits);
                    for idxDivisional=divisiveN:-1:1
                        divisiveUnits{prevDivisionalN+idxDivisional}=tempCell2{idxDivisional+1};
                    end
                else
                    % do nothing
                end
                
            end
            
            % deleting the "1" from the units (e.g. if the original string
            % was '1/ms'):
            multiplicativeUnits(strcmp(multiplicativeUnits, '1'))=[];
            divisiveUnits(strcmp(divisiveUnits, '1'))=[];
        end
        
    end
end

