% Beware: it does not save empty cell arrays (within the struct input), but
% it saved other "empty" data types.

classdef xmlManagement
    methods(Static)
        
        function [ s ] = xml2struct( file )
            
            
            % source:
            % https://people.sc.fsu.edu/~jburkardt/m_src/xml2struct/xml2struct.m
            % https://people.sc.fsu.edu/~jburkardt/m_src/xml2struct_test/xml2struct_test.html
            
            
            % Copyright (c) 2010, Wouter Falkena
            % All rights reserved.
            %
            % Redistribution and use in source and binary forms, with or without
            % modification, are permitted provided that the following conditions are
            % met:
            %
            %     * Redistributions of source code must retain the above copyright
            %       notice, this list of conditions and the following disclaimer.
            %     * Redistributions in binary form must reproduce the above copyright
            %       notice, this list of conditions and the following disclaimer in
            %       the documentation and/or other materials provided with the distribution
            %
            % THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
            % AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
            % IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
            % ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
            % LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
            % CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
            % SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
            % INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
            % CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
            % ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
            % POSSIBILITY OF SUCH DAMAGE.
            
            %%
            % convert the file path if it contains the short notation of the home directory
            file = xmlManagement.homeStringReplace(file);
            
            %%
            %Convert xml file into a MATLAB structure
            % [ s ] = xml2struct( file )
            %
            % A file containing:
            % <XMLname attrib1="Some value">
            %   <Element>Some text</Element>
            %   <DifferentElement attrib2="2">Some more text</Element>
            %   <DifferentElement attrib3="2" attrib4="1">Even more text</DifferentElement>
            % </XMLname>
            %
            % Will produce:
            % s.XMLname.Attributes.attrib1 = "Some value";
            % s.XMLname.Element.Text = "Some text";
            % s.XMLname.DifferentElement{1}.Attributes.attrib2 = "2";
            % s.XMLname.DifferentElement{1}.Text = "Some more text";
            % s.XMLname.DifferentElement{2}.Attributes.attrib3 = "2";
            % s.XMLname.DifferentElement{2}.Attributes.attrib4 = "1";
            % s.XMLname.DifferentElement{2}.Text = "Even more text";
            %
            % Please note that the following characters are substituted
            % '-' by '_dash_', ':' by '_colon_' and '.' by '_dot_'
            %
            % Written by W. Falkena, ASTI, TUDelft, 21-08-2010
            % Attribute parsing speed increased by 40% by A. Wanner, 14-6-2011
            % Added CDATA support by I. Smirnov, 20-3-2012
            %
            % Modified by X. Mo, University of Wisconsin, 12-5-2012
            
            % Modifications by Tibor Novák:
            % - When reading the values from the xml file, try to convert them back to
            % their original type instead of a structure contaning a string field. It
            % can only handle numeric arrays, logical values and strings (no cell...).
            
            if (nargin < 1)
                clc;
                help xml2struct
                return
            end
            
            if isa(file, 'org.apache.xerces.dom.DeferredDocumentImpl') || isa(file, 'org.apache.xerces.dom.DeferredElementImpl')
                % input is a java xml object
                xDoc = file;
            else
                %check for existance
                if (exist(file,'file') == 0)
                    %Perhaps the xml extension was omitted from the file name. Add the
                    %extension and try again.
                    if (isempty(strfind(file,'.xml')))
                        file = [file '.xml'];
                    end
                    
                    if (exist(file,'file') == 0)
                        error(['The file ' file ' could not be found']);
                    end
                end
                %read the xml file
                xDoc = xmlread(file);
            end
            
            %parse xDoc into a MATLAB structure
            s = xmlManagement.parseChildNodes(xDoc);
            
        end
        
        % ----- Subfunction parseChildNodes -----
        function [children,ptext,textflag] = parseChildNodes(theNode)
            % Recurse over node children.
            children = struct;
            ptext = struct; textflag = 'Text';
            if hasChildNodes(theNode)
                childNodes = getChildNodes(theNode);
                numChildNodes = getLength(childNodes);
                
                for count = 1:numChildNodes
                    theChild = item(childNodes,count-1);
                    [text,name,attr,childs,textflag] = xmlManagement.getNodeData(theChild);
                    
                    if (~strcmp(name,'#text') && ~strcmp(name,'#comment') && ~strcmp(name,'#cdata_dash_section'))
                        %XML allows the same elements to be defined multiple times,
                        %put each in a different cell
                        if (isfield(children,name))
                            if (~iscell(children.(name)))
                                %put existsing element into cell format
                                children.(name) = {children.(name)};
                            end
                            index = length(children.(name))+1;
                            %add new element
                            children.(name){index} = childs;
                            if(~isempty(fieldnames(text)))
                                % modified by Tibor Novák so the structure read
                                % from the xml match to the original output
                                % structure
                                readXMLText=text.Text;
                                readValue=str2num(readXMLText);
                                if isempty(readValue) && ~isempty(readXMLText)
                                    readValue=text.Text;
                                end
                                children.(name){index} = readValue;
                            end
                            if(~isempty(attr))
                                children.(name){index}.('Attributes') = attr;
                            end
                        else
                            %add previously unknown (new) element to the structure
                            children.(name) = childs;
                            if(~isempty(text) && ~isempty(fieldnames(text)))
                                % modified by Tibor Novák so the structure read
                                % from the xml match to the original output
                                % structure
                                readXMLText=text.Text;
                                readValue=str2num(readXMLText);
                                if isempty(readValue) && ~isempty(readXMLText)
                                    readValue=text.Text;
                                end
                                children.(name) = readValue;
                            end
                            if(~isempty(attr))
                                children.(name).('Attributes') = attr;
                            end
                        end
                    else
                        ptextflag = 'Text';
                        if (strcmp(name, '#cdata_dash_section'))
                            ptextflag = 'CDATA';
                        elseif (strcmp(name, '#comment'))
                            ptextflag = 'Comment';
                        end
                        
                        %this is the text in an element (i.e., the parentNode)
                        if (~isempty(regexprep(text.(textflag),'[\s]*','')))
                            if (~isfield(ptext,ptextflag) || isempty(ptext.(ptextflag)))
                                ptext.(ptextflag) = text.(textflag);
                            else
                                %what to do when element data is as follows:
                                %<element>Text <!--Comment--> More text</element>
                                
                                %put the text in different cells:
                                % if (~iscell(ptext)) ptext = {ptext}; end
                                % ptext{length(ptext)+1} = text;
                                
                                %just append the text
                                ptext.(ptextflag) = [ptext.(ptextflag) text.(textflag)];
                            end
                        end
                    end
                    
                end
            end
        end
        
        % ----- Subfunction getNodeData -----
        function [text,name,attr,childs,textflag] = getNodeData(theNode)
            % Create structure of node info.
            
            %make sure name is allowed as structure name
            name = toCharArray(getNodeName(theNode))';
            name = strrep(name, '-', '_dash_');
            name = strrep(name, ':', '_colon_');
            name = strrep(name, '.', '_dot_');
            
            attr = xmlManagement.parseAttributes(theNode);
            if (isempty(fieldnames(attr)))
                attr = [];
            end
            
            %parse child nodes
            [childs,text,textflag] = xmlManagement.parseChildNodes(theNode);
            
            if (isempty(fieldnames(childs)) && isempty(fieldnames(text)))
                %get the data of any childless nodes
                % faster than if any(strcmp(methods(theNode), 'getData'))
                % no need to try-catch (?)
                % faster than text = char(getData(theNode));
                text.(textflag) = toCharArray(getTextContent(theNode))';
            end
            
        end
        
        % ----- Subfunction parseAttributes -----
        function attributes = parseAttributes(theNode)
            % Create attributes structure.
            
            attributes = struct;
            if hasAttributes(theNode)
                theAttributes = getAttributes(theNode);
                numAttributes = getLength(theAttributes);
                
                for count = 1:numAttributes
                    %attrib = item(theAttributes,count-1);
                    %attr_name = regexprep(char(getName(attrib)),'[-:.]','_');
                    %attributes.(attr_name) = char(getValue(attrib));
                    
                    %Suggestion of Adrian Wanner
                    str = toCharArray(toString(item(theAttributes,count-1)))';
                    k = strfind(str,'=');
                    attr_name = str(1:(k(1)-1));
                    attr_name = strrep(attr_name, '-', '_dash_');
                    attr_name = strrep(attr_name, ':', '_colon_');
                    attr_name = strrep(attr_name, '.', '_dot_');
                    attributes.(attr_name) = str((k(1)+2):(end-1));
                end
            end
        end
        
        
        function varargout = struct2xml( s, varargin )
            
            % source:
            % https://www.mathworks.com/matlabcentral/fileexchange/28639-struct2xml
            % might need to fiand a better source bacuse of the license...
            % https://www.mathworks.com/matlabcentral/fileexchange/59411-struct2xml-with-bug-fix
            % http://freesourcecode.net/matlabprojects/72023/struct2xml-in-matlab#.XjQF5imYVhE
            

            %Convert a MATLAB structure into a xml file
            % [ ] = struct2xml( s, file )
            % xml = struct2xml( s )
            %
            % A structure containing:
            % s.XMLname.Attributes.attrib1 = "Some value";
            % s.XMLname.Element.Text = "Some text";
            % s.XMLname.DifferentElement{1}.Attributes.attrib2 = "2";
            % s.XMLname.DifferentElement{1}.Text = "Some more text";
            % s.XMLname.DifferentElement{2}.Attributes.attrib3 = "2";
            % s.XMLname.DifferentElement{2}.Attributes.attrib4 = "1";
            % s.XMLname.DifferentElement{2}.Text = "Even more text";
            %
            % Will produce:
            % <XMLname attrib1="Some value">
            %   <Element>Some text</Element>
            %   <DifferentElement attrib2="2">Some more text</Element>
            %   <DifferentElement attrib3="2" attrib4="1">Even more text</DifferentElement>
            % </XMLname>
            %
            % Please note that the following strings are substituted
            % '_dash_' by '-', '_colon_' by ':' and '_dot_' by '.'
            %
            % Written by W. Falkena, ASTI, TUDelft, 27-08-2010
            % On-screen output functionality added by P. Orth, 01-12-2010
            % Multiple space to single space conversion adapted for speed by T. Lohuis, 11-04-2011
            % Val2str subfunction bugfix by H. Gsenger, 19-9-2011
            
            % Modifications by Tibor Novák:
            % - All "num2str()" was replaced with "mat2str()"
            % - Added support for boolean values
            
            if (nargin ~= 2)
                if(nargout ~= 1 || nargin ~= 1)
                    error(['Supported function calls:' sprintf('\n')...
                        '[ ] = struct2xml( s, file )' sprintf('\n')...
                        'xml = struct2xml( s )']);
                end
            end
            if(nargin == 2)
                file = varargin{1};
                if (isempty(file))
                    error('Filename can not be empty');
                end
                if (isempty(strfind(file,'.xml')))
                    file = [file '.xml'];
                end
            end
            
            if (~isstruct(s))
                error([inputname(1) ' is not a structure']);
            end
            
            if (length(fieldnames(s)) > 1)
                error(['Error processing the structure:' sprintf('\n') 'There should be a single field in the main structure.']);
            end
            xmlname = fieldnames(s);
            xmlname = xmlname{1};
            
            %substitute special characters
            xmlname_sc = xmlname;
            xmlname_sc = strrep(xmlname_sc,'_dash_','-');
            xmlname_sc = strrep(xmlname_sc,'_colon_',':');
            xmlname_sc = strrep(xmlname_sc,'_dot_','.');
            %create xml structure
            docNode = com.mathworks.xml.XMLUtils.createDocument(xmlname_sc);
            %process the rootnode
            docRootNode = docNode.getDocumentElement;
            %append childs
            xmlManagement.parseStruct(s.(xmlname),docNode,docRootNode,[inputname(1) '.' xmlname '.']);
            if(nargout == 0)
                % convert the file path if it contains the short notation of the home directory
                file = xmlManagement.homeStringReplace(file);
                %save xml file
                xmlwrite(file,docNode);
            else
                varargout{1} = xmlwrite(docNode);
            end
        end
        % ----- Subfunction parseStruct -----
        function [] = parseStruct(s,docNode,curNode,pName)
            
            fnames = fieldnames(s);
            for i = 1:length(fnames)
                curfield = fnames{i};
                
                %substitute special characters
                curfield_sc = curfield;
                curfield_sc = strrep(curfield_sc,'_dash_','-');
                curfield_sc = strrep(curfield_sc,'_colon_',':');
                curfield_sc = strrep(curfield_sc,'_dot_','.');
                
                if (strcmp(curfield,'Attributes'))
                    %Attribute data
                    if (isstruct(s.(curfield)))
                        attr_names = fieldnames(s.Attributes);
                        for a = 1:length(attr_names)
                            cur_attr = attr_names{a};
                            
                            %substitute special characters
                            cur_attr_sc = cur_attr;
                            cur_attr_sc = strrep(cur_attr_sc,'_dash_','-');
                            cur_attr_sc = strrep(cur_attr_sc,'_colon_',':');
                            cur_attr_sc = strrep(cur_attr_sc,'_dot_','.');
                            
                            [cur_str,succes] = xmlManagement.val2str(s.Attributes.(cur_attr));
                            if (succes)
                                curNode.setAttribute(cur_attr_sc,cur_str);
                            else
                                disp(['Warning. The text in ' pName curfield '.' cur_attr ' could not be processed.']);
                            end
                        end
                    else
                        disp(['Warning. The attributes in ' pName curfield ' could not be processed.']);
                        disp(['The correct syntax is: ' pName curfield '.attribute_name = ''Some text''.']);
                    end
                elseif (strcmp(curfield,'Text'))
                    %Text data
                    [txt,succes] = xmlManagement.val2str(s.Text);
                    if (succes)
                        curNode.appendChild(docNode.createTextNode(txt));
                    else
                        disp(['Warning. The text in ' pName curfield ' could not be processed.']);
                    end
                else
                    %Sub-element
                    if (isstruct(s.(curfield)))
                        %single element
                        curElement = docNode.createElement(curfield_sc);
                        curNode.appendChild(curElement);
                        xmlManagement.parseStruct(s.(curfield),docNode,curElement,[pName curfield '.'])
                    elseif (iscell(s.(curfield)))
                        %multiple elements
                        for c = 1:length(s.(curfield))
                            curElement = docNode.createElement(curfield_sc);
                            curNode.appendChild(curElement);
                            if (isstruct(s.(curfield){c}))
                                xmlManagement.parseStruct(s.(curfield){c},docNode,curElement,[pName curfield '{' mat2str(c) '}.'])
                            else
                                disp(['Warning. The cell ' pName curfield '{' mat2str(c) '} could not be processed, since it contains no structure.']);
                            end
                        end
                    else
                        %eventhough the fieldname is not text, the field could
                        %contain text. Create a new element and use this text
                        curElement = docNode.createElement(curfield_sc);
                        curNode.appendChild(curElement);
                        [txt,succes] = xmlManagement.val2str(s.(curfield));
                        if (succes)
                            curElement.appendChild(docNode.createTextNode(txt));
                        else
                            disp(['Warning. The text in ' pName curfield ' could not be processed.']);
                        end
                    end
                end
            end
        end
        %----- Subfunction val2str -----
        function [str,succes] = val2str(val)
            
            succes = true;
            str = [];
            
            if (isempty(val))
                % modified by NT:
                val = mat2str(val);
                %return; %bugfix from H. Gsenger
            elseif (ischar(val))
                %do nothing
            elseif (isnumeric(val))
                val = mat2str(val);
            elseif (islogical(val))
                % This condition for the boolean variables was added by Tibor
                % Novák.
                val = mat2str(val);
            else
                succes = false;
            end
            
            if (ischar(val))
                %add line breaks to all lines except the last (for multiline strings)
                lines = size(val,1);
                val = [val char(sprintf('\n')*[ones(lines-1,1);0])];
                
                %transpose is required since indexing (i.e., val(nonspace) or val(:)) produces a 1-D vector.
                %This should be row based (line based) and not column based.
                valt = val';
                
                remove_multiple_white_spaces = true;
                if (remove_multiple_white_spaces)
                    %remove multiple white spaces using isspace, suggestion of T. Lohuis
                    whitespace = isspace(val);
                    nonspace = (whitespace + [zeros(lines,1) whitespace(:,1:end-1)])~=2;
                    nonspace(:,end) = [ones(lines-1,1);0]; %make sure line breaks stay intact
                    str = valt(nonspace');
                else
                    str = valt(:);
                end
            end
        end
        
        
        %% unfinished functions:
        
        function xmlTreeObject = parseString(xmlString)            % TODO!!!!!!!!!
            % source for parsing raw string to xml:
            % http://undocumentedmatlab.com/articles/parsing-xml-strings
            % https://stackoverflow.com/questions/9489713/parsing-xml-strings-in-matlab
            
            try
                % The following avoids the need for file I/O:
                inputObject = java.io.StringBufferInputStream(xmlString);  % or: org.xml.sax.InputSource(java.io.StringReader(xmlString))
                try
                    % Parse the input data directly using xmlread's core functionality
                    parserFactory = javaMethod('newInstance','javax.xml.parsers.DocumentBuilderFactory');
                    p = javaMethod('newDocumentBuilder',parserFactory);
                    xmlTreeObject = p.parse(inputObject);
                catch
                    % Use xmlread's semi-documented inputObject input feature
                    xmlTreeObject = xmlread(inputObject);
                end
            catch
                % Fallback to standard xmlread usage, using a temporary XML file:
                % Store the XML data in a temp *.xml file
                filename = [tempname '.xml'];
                fid = fopen(filename,'Wt');
                fwrite(fid,xmlString);
                fclose(fid);
                % Read the file into an XML model object
                xmlTreeObject = xmlread(filename);
                % Delete the temp file
                delete(filename);
            end
            % Parse the XML model objec
            
        end
        
        
        
        function struct=structure_from_text()
            %TODO it will neead a proper input and output argument, and a bit of
            %testing
            
            
            % source:
            % https://stackoverflow.com/questions/10880754/how-to-read-a-structure-from-txt-file-in-matlab
            
            %%
            
            %# read lines
            fid = fopen('file.txt','rt');
            C = textscan(fid, '%s', 'Delimiter',''); C = C{1};
            fclose(fid);
            
            %# start/end of each structure
            startIdx = find(ismember(C, 'struct'));
            endIdx = find(ismember(C, 'endstruct'));
            
            %# array of strucutres
            N = numel(startIdx);
            arr = struct('studentname','', 'notes','', 'n1',0, 'n2',0, 'average',0);
            arr = repmat(arr,[N 1]);
            
            %# parse and store each structure in the array
            for i=1:numel(startIdx)
                %# parse key/value of struct
                s = C(startIdx(i)+1:endIdx(i)-1);
                s = regexp(s, '(\w+)\s*[:=]\s*([^%$]*)(?:%[^$]*)?', 'tokens', 'once');
                s = vertcat(s{:});
                
                %# try to parse as numbers
                v = str2double(s(:,2));
                s(~isnan(v),2) = num2cell(v(~isnan(v)));
                
                %# store: struct.key = value
                for j=1:size(s,1)
                    arr(i).(s{j,1}) = s{j,2};
                end
            end
            
            
        end
        
        function fullPath_replaced = homeStringReplace(fullPath)
        
            %%
            % convert the file path if it contains the short notation of the home directory
            
            % source: https://www.mathworks.com/matlabcentral/fileexchange/15885-get-user-home-directory

            splittedPath = strsplit(fullPath, filesep);

            if numel(splittedPath) > 0

                if ispc
                    
                    homeString = '%userprofile%';
                    

                    userDir = winqueryreg('HKEY_CURRENT_USER',...
                        ['Software\Microsoft\Windows\CurrentVersion\' ...
                        'Explorer\Shell Folders'],'Personal');
                    
                else
                    
                    homeString = '~';
                    
                    userDir = char(java.lang.System.getProperty('user.home'));
                    
                end
                
                if strcmp(homeString, splittedPath(1))
                    
                    fullPath_replaced = strrep(fullPath, homeString, userDir);
                else
                    fullPath_replaced = fullPath;
                end
            else
                fullPath_replaced = fullPath;
            end
        
        end
        
    end
end

