classdef customColormap
    
    methods(Static)

        function definitionNames = getDefinitionNames()
        % Returns the napes of the custom (non-Matlab built-in) colormaps.
        % Keep it in synch with the "definitions" function!

        definitionNames = {...
            'Green',...
            'Red',...
            'Blue',...
            'Yellow',...
            'Cyan',...
            'Magenta',...
            'Grey',...
            'Hot',...
            'Inv. Hot',...
            'Div 1',...
            'Div 2',...
            'Div 3',...
            'Div 4',...
            'Div 5',...
            'Div 1 inv',...
            'Div 2 inv',...
            'Div 3 inv',...
            'Div 4 inv',...
            'Div 5 inv',...
            'Fine Green',...
            'Fine Red'};

        end

        function [colormapVector, colormapRange] = getColormap(colormapName, colormap_minValue, colormap_maxValue)

            % discretize the colormap range
            colormapRange = [floor(colormap_minValue), ceil(colormap_maxValue)];
            
            % check the validity of the colormap ranges:
            colormapRange(1) = max([0, min(colormapRange)]);
            colormapRange(2) = max([1, max(colormapRange)]);

            colormapDefinition = customColormap.definitions(colormapName);
            colormapVector = customColormap.createColormap_int(colormapDefinition, colormapRange);

        end

    end
    
    methods (Static, Access=private)
        
        function newColormap_values = createColormap_int(colormapDefinition, colormapRange)
        % This function creates a (discretized) colormap for 8-bit grayscale images. For every subsequent pixel values (0,1,2...) the colors must be defined.

            % discretized values in a range for which the colormap should be created.
            newColormap_points = double(floor(colormapRange(1)):1:ceil(colormapRange(2)));

            definition_intervals = colormapDefinition.intervals;
            definition_colors = colormapDefinition.colors;
            definition_function = colormapDefinition.function;


            newColormap_values = definition_function(definition_intervals, definition_colors, newColormap_points);

        end


        function confinedPoints = confineColormapPoints(colormapPoints)
        % Confines the point or which the colormap is defines or should be calculated to the [0,1] range

            colormap_leftBorder = colormapPoints(1);
            colormap_rightBorder = colormapPoints(end);

            confinedPoints = (colormapPoints - colormap_leftBorder) / (colormap_rightBorder - colormap_leftBorder);

        end


        function newColormap_values = linearInterpolation(definition_intervals, definition_colors, newColormap_points)
        % This function performs linear interpoaltion for the 3 color components separately.

            definition_points = cumsum(definition_intervals);
            definition_points = [0, definition_points] + 1;

            % confine values

            newColormap_points = customColormap.confineColormapPoints(newColormap_points);
            definition_points = customColormap.confineColormapPoints(definition_points);

            newColormap_values = interp1(definition_points, definition_colors, newColormap_points);

        end

        function newColormap_values = divergingColormap(definition_intervals, definition_colors, newColormap_points)
        % It uses an external function to perform the interpolation. It uses a colorspace other than RGB for the task.

            background = definition_colors(1,:);
            color1 = definition_colors(2, :);
            color2 = definition_colors(3, :);

            s = linspace(0,1,numel(newColormap_points)-1);
            divColormap = diverging_map(s, color1, color2);

            newColormap_values = [background; divColormap];
        end



        function definition = definitions(colormapName)

        % definition.color: RGB color definitions
        % definition.intervals: relative "length" (sampling points) of intervals betweent the RGB color definitions
        % definition.function: function to calculate the colors between the defined RGB colors (within the "intervals")

        % TODO: consider defining the background color separately


        switch colormapName
            case 'Green'
                definition.colors = [
                    0   0   0;     % Black
                    0   1   0;     % Green
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Red'
                definition.colors = [
                    0   0   0;     % Black
                    1   0   0;     % Red
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Blue'
                definition.colors = [
                    0   0   0;     % Black
                    0   0   1;     % Blue
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Yellow'
                definition.colors = [
                    0   0   0;     % Black
                    1   1   0;     % Yellow
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Cyan'
                definition.colors = [
                    0   0   0;     % Black
                    0   1   1;     % Cyan
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Magenta'
                definition.colors = [
                    0   0   0;     % Black
                    1   0   1;     % Magenta
                    1   1   1;     % White
                ];
                definition.intervals = [1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Grey'
                definition.colors = [
                    0   0   0;     % Black
                    1   1   1;     % White
                ];
                definition.intervals = [1];
                definition.function = @customColormap.linearInterpolation;

            case 'Hot'
                % Sets colors to use default 'hot' colormap.
                definition.colors = [
                    0   0   0;     % Black
                    1   0   0;     % Red
                    1   1   0;     % Yellow
                    1   1   1;     % White
                    ];
                definition.intervals = [1,1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Inv. Hot'
                % Sets colors to use default 'hot' colormap.
                definition.colors = [
                    1   1   1;     % White
                    1   0   0;     % Red
                    1   1   0;     % Yellow
                    0   0   0;     % Black
                    ];
                definition.intervals = [1,1,1];
                definition.function = @customColormap.linearInterpolation;

            case 'Div 1'
                definition.colors = [
                    0   0   0;      % Black
                    0.230 0.299 0.754;
                    0.706 0.016 0.150;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 2'
                definition.colors = [
                    0   0   0;      % Black
                    0.436 0.308 0.631;
                    0.759 0.334 0.046;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 3'
                definition.colors = [
                    0   0   0;      % Black
                    0.085 0.532 0.201;
                    0.436 0.308 0.631;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 4'
                definition.colors = [
                    0   0   0;      % Black
                    0.217 0.525 0.910;
                    0.677 0.492 0.093;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 5'
                definition.colors = [
                    0   0   0;      % Black
                    0.085 0.532 0.201;
                    0.758 0.214 0.233;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 1 inv'
                definition.colors = [
                    1   1   1;     % White
                    0.706 0.016 0.150;
                    0.230 0.299 0.754;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 2 inv'
                definition.colors = [
                    1   1   1;     % White
                    0.759 0.334 0.046;
                    0.436 0.308 0.631;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 3 inv'
                definition.colors = [
                    1   1   1;     % White
                    0.436 0.308 0.631;
                    0.085 0.532 0.201;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 4 inv'
                definition.colors = [
                    1   1   1;     % White
                    0.677 0.492 0.093;
                    0.217 0.525 0.910;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Div 5 inv'
                definition.colors = [
                    1   1   1;     % White
                    0.758 0.214 0.233;
                    0.085 0.532 0.201;
                ];
                definition.intervals = nan;
                definition.function = @customColormap.divergingColormap;

            case 'Fine Green'
                definition.colors = [
                    255 255 255;
                    0 255 0;
                    0 51 0;
                    0 0 0;
                    ]/255;
                points = [1, 52, 248, 256];
                definition.intervals = points(2:end)-points(1:end-1);
                definition.function = @customColormap.linearInterpolation;

            case 'Fine Red'
                definition.colors = [
                    255 255 255;
                    255 0 0;
                    51 0 0;
                    0 0 0;
                    ]/255;
                points = [1, 78, 248, 256];
                definition.intervals = points(2:end)-points(1:end-1);
                definition.function = @customColormap.linearInterpolation;

            otherwise
                try
                    definition.colors = colormap(colormapName);
                    definition.intervals = ones(size(colors, 1)-1,1);
                    definition.function = @customColormap.linearInterpolation;
                catch
                    error('No custom defined or Matlab bult-in colormap named "%s".', colormapName)
                end

        end

        end

    end
end
