% write the pixel size info that can be read with ImageJ:
% https://forum.image.sc/t/simplest-way-to-save-tif-in-matlab-with-resolution-in-um/67893/3
% there is no built-in Matlab function to write EXIF or TIFF headers:
% https://www.mathworks.com/matlabcentral/answers/152559-writing-exif-data-to-jpg
%https://www.mathworks.com/matlabcentral/answers/446658-how-to-write-metadata-information-to-a-jpg-file

function imageMatrixExport(imageMatrix, colorMap, pixelSize_nm, outputFullFileName)
    % Exports the image matrix into a tiff file with pixel size and
    % colormap informations.

    pixelSize_um = pixelSize_nm/1000;

    colormapDivision = size(colorMap, 1);
    if colormapDivision>256
       warning('The requested colormap for tiff export contans more divisions than the tiff can handle (255). Normalizing both the colormap and the image.');
       originalColormap_points = (0:1:(colormapDivision-1))/(colormapDivision-1);
       newColormap_points = (0:1:255)/255;
       %interpolate colormap:
       colorMap = interp1(originalColormap_points, colorMap, newColormap_points);
       
       normalizationFactor  = 255/colormapDivision;
       imageMatrix = imageMatrix*normalizationFactor;
    end
    
    image_maximum = max(max(imageMatrix));
    if image_maximum>255
        warning('The image matrix for tiff with colormap can only be uint8 type (maximal value: 255), but the maximal value of the current image is %d. All values higher than 255 will be set to 255.', image_maximum);
    end
    
    % must convert to uint8 for the colormap to work
    imageMatrix = uint8(imageMatrix);
    
    imwrite(imageMatrix, colorMap, outputFullFileName, 'TIF',...
        'Resolution', [1/pixelSize_um 1/pixelSize_um],...
        'Description', sprintf('ImageJ=1.53q\nunit=um\npixel size = %f um\n', pixelSize_um));

    % ImageJ does not reads the "pixel size" line in the "Description".
    % It is intended to be human readable because some image viewers cannot
    % interpret the "Resolution" data but able to display the
    % "Description".
end
