classdef filteringOperation


    methods(Static)

        function [pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new, filterBoolVect_new_noFrameIndex, metadata, sharedParameters] = filter(pointillisticData_accepted_old, pointillisticData_rejected_old, filterBooleanVect_old, method, settings)
    % make the (re)filtering in the "rainSTORM" data convention, in which the
    % the data is handled internally
    % no need to convert the data to different convention, only has to do
    % the conversion of the filtering thresholds to match the defaults
    % data units
    % performs the filtering without the "frame index", too, as some
    % algorithms (drift correction) need a data that has no such filtering
    % (at least for further refiltering it is required)

    filteringThresholds = settings.thresholds;
    
    cameraSignalConversion = settings.cameraSignalConversion;
    
    
    % if there were no filtering yet, the "rejected" data might be empty
    if isempty(pointillisticData_rejected_old)
        pointillisticData_rejected_old = pointillisticData_fieldManagement.createEmptyData(pointillisticData_accepted_old);
        filterBooleanVect_old = true(size(pointillisticData_accepted_old.frame_idx));
    end
        
    % convert the thresholds to units defined in the convention:
    fieldNames = {filteringThresholds.field};
    [unitsList_rainSTORM] = getDataFieldUnits.default(fieldNames);
    [filteringThreshold_converted] = filteringOperation.convertFiltreingThresholdsToDefaultUnits(filteringThresholds, unitsList_rainSTORM, cameraSignalConversion);

    % because of the drift correction, perform the filtering without the
    % "frame_idx" field, too...
    filteringFields = {filteringThreshold_converted.field};
    thresholds_noFrame = filteringThreshold_converted(~strcmp(filteringFields, 'frame_idx'));
    thresholds_onlyFrame = filteringThreshold_converted(strcmp(filteringFields, 'frame_idx'));

    filterBoolVect_accepted_noFrame = pointillisticData_filtering.getBooleanVect(pointillisticData_accepted_old, thresholds_noFrame);
    filterBoolVect_accepted_onlyFrame = pointillisticData_filtering.getBooleanVect(pointillisticData_accepted_old, thresholds_onlyFrame);

    filterBoolVect_rejected_noFrame = pointillisticData_filtering.getBooleanVect(pointillisticData_rejected_old, thresholds_noFrame);
    filterBoolVect_rejected_onlyFrame = pointillisticData_filtering.getBooleanVect(pointillisticData_rejected_old, thresholds_onlyFrame);

    filterBoolVect_accepted = filterBoolVect_accepted_noFrame & filterBoolVect_accepted_onlyFrame;
    filterBoolVect_rejected = filterBoolVect_rejected_noFrame & filterBoolVect_rejected_onlyFrame;
    
    
    [pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new] = pointillisticData_filtering.refilter(pointillisticData_accepted_old, pointillisticData_rejected_old, filterBooleanVect_old, filterBoolVect_accepted, filterBoolVect_rejected);


    % if we didn't need to "no frame" foltering for the drift correction,
    % it would be this simple:
    %[pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new] = pointillisticData_filtering.rethreshold(poinstillisticData_accepted_old, pointillisticData_rejected_old, filterBoolVect_old, filteringThreshold_converted);

    fieldNames = pointillisticData_fieldNameManagement.getFieldNames(pointillisticData_accepted_new);

    metadata.acceptedNumber = numel(pointillisticData_accepted_new.(fieldNames{1}));
    metadata.rejectedNumber = numel(pointillisticData_rejected_new.(fieldNames{1}));

    sharedParameters = struct([]);

    % for the drift correction, disable the frame index based filtering
    filterBoolVect_new_noFrameIndex=false(size(filterBooleanVect_old));
    filterBoolVect_new_noFrameIndex(filterBooleanVect_old)=filterBoolVect_accepted_noFrame;
    filterBoolVect_new_noFrameIndex(~filterBooleanVect_old)=filterBoolVect_rejected_noFrame;

end


function [pointillisticData_accepted_new, pointillisticData_rejected_new, filterBoolVect_new, filterBoolVect_new_noFrame, metadata, sharedParameters] = filter_convention(poinstillisticData_accepted_old, pointillisticData_rejected_old, filterBoolVect_old, method, settings)
    % make the filtering in some other data convention, i.e. the data must be converted before the filtering
    % TODO: this function is not tested!!!
    % TODO: also filter without the frame index, for the drift correction

    convention = settings.originalDataConvention.name;
    algorithm = settings.originalDataConvention.algorithm;
    analyses = settings.originalDataConvention.analyses;
    filteringThresholds = settings.thresholds;
    
    cameraSignalConversion = settings.cameraSignalConversion;
    
    % convert the thresholds to units defined in the convention:
    fieldNames = {filteringThresholds.field};
    [unitsList_convention] = getDataFieldUnits.convention(fieldNames, convention, algorithm, analyses);
    [filteringThreshold_converted] = filteringOperation.convertFiltreingThresholdsToDefaultUnits(filteringThresholds, unitsList_convention, cameraSignalConversion);
    
    [dataMatrix_accepted_old, headerList] = convertDataToExport(poinstillisticData_accepted_old, convention, algorithm, analyses, cameraSignalConversion);
    dataCell_accepted_old = {headerList, dataMatrix_accepted_old};

    [dataMatrix_rejected_old, headerList] = convertDataToExport(pointillisticData_rejected_old, convention, algorithm, analyses, cameraSignalConversion);
    dataCell_rejected_old = {headerList, dataMatrix_rejected_old};

    [dataCell_accepted_new, dataCell_rejected_new, filterBoolVect_new] = pointillisticData_filtering.rethreshold(dataCell_accepted_old, dataCell_rejected_old, filterBoolVect_old, filteringThreshold_converted);
    
    dataMatrix_accepted_new = dataCell_accepted_new{2};
    dataMatrix_rejected_new = dataCell_rejected_new{2};

    pointillisticData_accepted_new = convertImportedData(dataMatrix_accepted_new, headerList, convention, algorithm, analyses, cameraSignalConversion);
    pointillisticData_rejected_new = convertImportedData(dataMatrix_rejected_new, headerList, convention, algorithm, analyses, cameraSignalConversion);
    

    fieldNames = pointillisticData_fieldNameManagement.getFieldNames(pointillisticData_accepted_new);

    metadata.acceptedNumber = numel(pointillisticData_accepted_new.(fieldNames{1}));
    metadata.rejectedNumber = numel(pointillisticData_rejected_new.(fieldNames{1}));

    sharedParameters = struct([]);

    % for the drift correction, disable the frame index based filtering
    filterBoolVect_new_noFrame = true(size(filterBoolVect_new));

end


function [filteringThreshold_converted] = convertFiltreingThresholdsToDefaultUnits(filteringThresholds, unitsList_convention, cameraSignalConversion)
    % Converts the bounds of the thresholding to the given units
   

    filteringThreshold_converted = filteringThresholds;
    for idxField=1:numel(filteringThresholds)
        
        filteringThreshold_act = filteringThresholds(idxField);
        
        % convert the threshold bounds and not the whole data as it
        % is less calculation
        valueToConvert = filteringThreshold_act.bounds;
        unitString_input = filteringThreshold_act.unit;
        unitString_output = unitsList_convention{idxField};
        if ~isempty(unitString_input)
            [thresholds_bounds, ~] = unitConversion.convertValue(valueToConvert, unitString_input, unitString_output, cameraSignalConversion);
        else
            thresholds_bounds = valueToConvert;
        end

        filteringThreshold_converted(idxField).bounds = thresholds_bounds;
        filteringThreshold_converted(idxField).unit = unitString_output;
        
    end 
end


function filteringSettings = initializeFilteringSettings(fieldNames)
    % Creates a "filteringThresholds" variables based  on the field
    % names given in a cell array with unset ([-Inf, Inf]) bounds.

    if ~iscell(fieldNames)
        error("The given field names for the filtering thresholds initialization is not a cell array, but a %s.", class(fieldNames))
    end

    [unitsList_rainSTORM] = getDataFieldUnits.default(fieldNames);

    filteringThresholds = struct();
    for idxField = 1:numel(fieldNames)
        filteringThresholds(idxField).field = fieldNames{idxField};
        filteringThresholds(idxField).bounds = [-Inf, Inf];
        filteringThresholds(idxField).unit = unitsList_rainSTORM{idxField};
    end

    filteringSettings.method = 'All data fields';
    filteringSettings.operationType='pointillistic data filtering';
    filteringSettings.note='filtering the data in the rainSTORM convention';
    filteringSettings.functionName='filteringOperation.filter';
    filteringSettings.settings.thresholds = filteringThresholds;


    filteringSettings.settingsLinking(1).target='originalDataConvention';
    filteringSettings.settingsLinking(1).sourceParameter='convention';
    filteringSettings.settingsLinking(2).target='stackSize';
    filteringSettings.settingsLinking(2).sourceParameter='stackSize';
    filteringSettings.settingsLinking(3).target='frameRange';
    filteringSettings.settingsLinking(3).sourceParameter='frameRange';
    filteringSettings.settingsLinking(4).target='ROI';
    filteringSettings.settingsLinking(4).sourceParameter='ROI';
    filteringSettings.settingsLinking(5).target='cameraSignalConversion';
    filteringSettings.settingsLinking(5).sourceParameter='cameraSignalConversion';
    

end

function filteringSettings = initializeFilteringSettings_originalConvention(sharedParameters)
    % Create filtering threshold based on the fields defines for
    % the given data convention. The filtering names can be chosen
    % to be the original data names or the ones "converted" ones
    % that the rainSTORM algorithms use.

    convention = sharedParameters.convention.name;
    algorithm = sharedParameters.convention.algorithm;
    analyses = sharedParameters.convention.analyses;

    [algorithmFields, analysesFields] = conventionFunctions.conventionFields(convention, algorithm, analyses);
    jointFields = [algorithmFields, analysesFields{:}];

    % remove the definitions where the "fieldname" is empty
    jointFields = jointFields(~strcmp({jointFields.fieldname}, ''));
    
    filteringThresholds = struct();
    for idx = 1:numel(jointFields)
        filteringThresholds(idx).field = jointFields(idx).header;
        filteringThresholds(idx).bounds = [-Inf Inf];
        filteringThresholds(idx).unit = jointFields(idx).unit;
    end
    

    filteringSettings.method = 'All original data fields';
    filteringSettings.type='pointillistic data filtering';
    filteringSettings.note='filtering the data in the original convention';
    filteringSettings.functionName='filteringOperation.filter';
    filteringSettings.settings.thresholds = filteringThresholds;

    filteringSettings.settingsLinking(1).target='originalDataConvention';
    filteringSettings.settingsLinking(1).sourceParameter='convention';
    filteringSettings.settingsLinking(2).target='stackSize';
    filteringSettings.settingsLinking(2).sourceParameter='stackSize';
    filteringSettings.settingsLinking(3).target='frameRange';
    filteringSettings.settingsLinking(3).sourceParameter='frameRange';
    filteringSettings.settingsLinking(4).target='ROI';
    filteringSettings.settingsLinking(4).sourceParameter='ROI';
    filteringSettings.settingsLinking(5).target='cameraSignalConversion';
    filteringSettings.settingsLinking(5).sourceParameter='cameraSignalConversion';

end


function filteringSettings_updated = adjustFilteringSettings(filteringSettings, sharedParameters)
    % Updates some of the filtering threshold (x and y coordinates and the
    % frame index) according to the actual data. The update is only performed
    % if the data has stricter bounds than the preset ones.

    % TODO: it only works if the filtering thresholds are given in the
    % "rainSTORM" data convention
    % TODO: convert the filtering threshold between the data convention


    filteringThresholds = filteringSettings.settings.thresholds;
    fieldNames = {filteringThresholds.field};

    stackSize = sharedParameters.stackSize;
    frameRange = sharedParameters.frameRange;
    ROI = sharedParameters.ROI;
    cameraSignalConversion = sharedParameters.cameraSignalConversion;


    if any(strcmp(fieldNames, "x_coord"))
        indices = find(strcmp(fieldNames, "x_coord"));
        idx = indices(1);
        x_lower = -Inf;
        x_upper = Inf;
        if ~isempty(ROI)
            ROI_x = unitConversion.convertValue(ROI.x, ROI.unit, filteringThresholds(idx).unit, cameraSignalConversion);
            x_lower = ROI_x(1);
            x_upper = ROI_x(2);
        elseif ~ismepty(stackSize)
            stackSize_x = unitConversion.convertValue(stackSize(1), 'camera pixel length', filteringThresholds(idx).unit, cameraSignalConversion);
            x_lower = 0;
            x_upper = stackSize_x;
        end
        filteringThresholds(idx).bounds = [max(filteringThresholds(idx).bounds(1), x_lower), min(filteringThresholds(idx).bounds(2), x_upper)];
    end

    if any(strcmp(fieldNames, "y_coord"))
        indices = find(strcmp(fieldNames, "y_coord"));
        idx = indices(1);
        y_lower = -Inf;
        y_upper = Inf;
        if ~isempty(ROI)
            ROI_y = unitConversion.convertValue(ROI.y, ROI.unit, filteringThresholds(idx).unit, cameraSignalConversion);
            y_lower = ROI_y(1);
            y_upper = ROI_y(2);
        elseif ~ismepty(stackSize)
            stackSize_y = unitConversion.convertValue(stackSize(2), 'camera pixel length', filteringThresholds(idx).unit, cameraSignalConversion);
            y_lower = 0;
            y_upper = stackSize_y;
        end
        filteringThresholds(idx).bounds = [max(filteringThresholds(idx).bounds(1), y_lower), min(filteringThresholds(idx).bounds(2), y_upper)];
    end

    if any(strcmp(fieldNames, "frame_idx"))
        indices = find(strcmp(fieldNames, "frame_idx"));
        idx = indices(1);
        frame_lower = 1;
        frame_upper = Inf;
        if ~isempty(frameRange)
            frameRange_converted = unitConversion.convertValue(frameRange, 'camera frame interval', filteringThresholds(idx).unit, cameraSignalConversion);
            frame_lower = frameRange_converted(1);
            frame_upper = frameRange_converted(2);
        elseif ~ismepty(stackSize)
            stackSize_frame = unitConversion.convertValue(stackSize(3), 'camera frame interval', filteringThresholds(idx).unit, cameraSignalConversion);
            frame_lower = 1;
            frame_upper = stackSize_frame;
        end
        filteringThresholds(idx).bounds = [max(filteringThresholds(idx).bounds(1), frame_lower), min(filteringThresholds(idx).bounds(2), frame_upper)];
    end

    filteringSettings_updated = filteringSettings;
    filteringSettings_updated.settings.thresholds = filteringThresholds;

end


end
end