classdef imageStackReading
% This class is only a collection of different image stack format reading methods

    methods (Static)
        
        function [imageStack, originalMetadata, omeMetadata, metadata, sharedParameters] = read(fullFileName, imageStackFormat, method, imageStackReadingSettings)
            % This function reads the image data from the given path (let
            % it be image stack or folder containing several single frame
            % files) with the specified read method, which depends on image
            % format.
            
            % the switch-case options must be consistent with these:
            %imageFormatFilterList=imageStack_selection.getStackFormatFilterList();
            %folderTypeList=imageStack_selection.getImageFolderTypeList();
            
            switch imageStackFormat
                case {'TIFF image stack (*.tif)', 'TIFF image stack (*.tiff)'}
                    % TIF/TIFF file:
                    [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.multipage_tiff(fullFileName);
                case 'RAW image stack (*.raw)'
                    % RAW file:
                    [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.raw(fullFileName);
                case 'TIFF folder'
                    % folder containing tif files:
                    [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.per_frame_tiff(fullFileName);
                case 'Nikon image stack (*.nd2)'
                    % Nikon ND2 file format:
                    [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.nd2(fullFileName);
                otherwise
                    % try using the Bio-Formats library on unrecognized format:
                    [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.bioformats(fullFileName);
                    
            end
            metadata.imageStackFormat = imageStackFormat;
            
            
            cameraPrameters = metadata.cameraPrameters;
            cameraCalibrationFile = imageStackReadingSettings.cameraCalibrationFile;
            additionalMagnification = imageStackReadingSettings.additionalMagnification;
            
            extractedCameraSignalConversion = imageStack_getConversionFactors.cameraSignal(cameraPrameters, additionalMagnification, cameraCalibrationFile);
            
            % overwrite the parametrs that could not be read out from the
            % metadata with the default ones:
            missingOnlyBoolean = true;
            definedCameraSignalConversion = imageStackReadingSettings.missingCameraSignalConversionParameterOverwrite;
            cameraSignalConversion = imageStack_getConversionFactors.overwriteParameters(extractedCameraSignalConversion, definedCameraSignalConversion, missingOnlyBoolean);
            
            sharedParameters.cameraSignalConversion = cameraSignalConversion;
            sharedParameters.stackSize = metadata.stackSize;
            dataPath = '';
            dataName = '';
            if isfile(fullFileName)
                [dataPath, dataName, ~] = fileparts(fullFileName); 
            elseif isfolder(fullFileName)
                % let the name of the folder containing the image files be
                % the data name and its parent folder be the data path
                splittedPath = split(fullFileName, fileparts);
                dataPath = splittedPath{end};
                dataName = fullfile(splittedPath{1:end-1});
            else
                error('The image stack path given for reading is neither a file nor a folder.')
            end
            sharedParameters.originalDataPath = dataPath;
            sharedParameters.originalDataName = dataName;
        end
        
        
        function [imageStack, originalMetadata, omeMetadata, metadata] = bioformats(fullFileName)
            % This function use the Bio-Formats Matlab library to open
            % image stack and extract their metadata.
            
            % see:
            % https://docs.openmicroscopy.org/bio-formats/latest/developers/matlab-dev.html#retrieving-metadata
            
            if ~exist('bfCheckJavaPath.m','file') == 2
                error('The Bio-Formats is unavailable. Please add the Bio-Formats library to the Matlab work path. Can not open image stack.' )
            end
            
            
            
            BioFormatsData = bfopen(fullFileName);
            % get the metadata:
            omeMetadata=BioFormatsData{4};
            originalMetadata=BioFormatsData{2};
            
            if isempty(fullFileName)
                % if the given file parts is empty, the bioformats library
                % creates a pop-up window where a file can be selected
                % update the filename in this case
                bioformats_plane_info_splitted = split(BioFormatsData{1}{1,2}, ';');
                fullFileName = bioformats_plane_info_splitted{1};
            end
            [~,~, extension ]=fileparts(fullFileName);
                
            
            % get the image stack:
            imageStack = cat(3, BioFormatsData{1}{1:end, 1});
                
            metadata.stackSize=size(imageStack);
            
            % extract the acquisition parameters from the metadata:
            try
                metadata.cameraPrameters=imageStack_extractCameraParameters(omeMetadata, originalMetadata, extension);
            catch
                % TODO: decide on it
                warning('Could not extract the acquisition settings from the image stack metadata, returning empty camera settings data. Fill the fields manually.')
                metadata.cameraPrameters=struct([]);
            end
            
            % TODO:
            % - when opening tiff files saved with Nikon NIS Elements, the
            % original metadata hash table is empty, why????
            
        end
        
        
        function [imageStack, originalMetadata, omeMetadata, metadata] = multipage_tiff(fullFileName)
            % Loads a multipage *.tiff/*.tif file.
            
            try
                % use the bioformats library:
                [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.bioformats(fullFileName);
            catch
                % if the Bio-Formats library fail for some reason
                % Matlab's "imInfo" function could be used, but if it the
                % image metadata is not in OME format (that the bioformats
                % library can read) or other standard format, it is
                % hopeless...
                % its only use would be to read OME TIFF files without the
                % Bioformats library, but just install it...
                
                error('Could not open the "%s" *.tiff image stack with the Bioformats library. Check the stack with ImageJ and convert it to tiff or check whether the Bioformats library is installed.', fullFileName)
            end
            
        end
        
        
        function [imageStack, originalMetadata, omeMetadata, metadata] = per_frame_tiff(fullPath)
        % Loads all files (intended for single page TIFF files) in the
        % directory, and store the number of matching files as number of frames.
        
            [path,~,extension]=fileparts(fullPath);
            if ischar(path)
                if ~isempty(path)
                    % the last seperator need for the "dir()" function
                    path=[path, filesep];
                    % I do not understand why Matlab removes the last
                    % fielseperator from the path....
                else
                    % do nothing
                    % In this case the reading should be performed from the
                    % working directory.
                end
            else
               error('You did not give a string as a file name or path') 
            end
            
            if isempty(extension)
                if ~isempty(dir([path,'*','.tif']))
                    extension='.tif';
                elseif ~isempty(dir([path,'*','.tiff']))
                    extension='.tiff';
                else
                   error("There are no TIFF files in the given folder.") 
                end
            elseif srtcmp(extension, '.tif') || srtcmp(extension, '.tiff')
                % do nothing
            else
                error(['You gave other file than TIFF. The chosen file type is: ', extension])
            end
            listOfFiles = dir([path,'*',extension]);
            numOfIm = size(listOfFiles, 1);
            
            
            firstFileName = fullfile(path,listOfFiles(1).name);
            imInfo = imfinfo(firstFileName);
            width = imInfo(1).Width;
            height = imInfo(1).Height;
            stackSize = [height, width, numOfIm];
            
            %reader = bfGetReader(firstFileName);
            %omeMeta = reader.getMetadataStore();
            % could not find method for the "original metadata" extraction
            % without the "bfopen" function
            % extract the metadata from the first file:
            BioFormatsData = bfopen(fullFileName);
            omeMetadata=BioFormatsData{4};
            originalMetadata=BioFormatsData{2};
            try
                metadata.cameraPrameters=imageStack_extractCameraPrameters(omeMetadata, originalMetadata, extension);
            catch
                % TODO: decide on it
                warning('Could not extract the acquisition settings from the image stack metadata, returning empty camera settings data. Fill the fields manually that will be used later.')
                metadata.cameraPrameters=struct([]);
            end
            
            imageStack = zeros(stackSize, 'uint16');
            for i=1:numOfIm
                imageStack(:,:,i) = imread(fullfile(path,listOfFiles(i).name), 'Info', imInfo);
                if (~(mod(i,100)))
                    disp( [ 'Frame: ' num2str(i) '/' num2str(numOfIm)] );
                end
            end
            
            metadata.stackSize=stackSize;
            metadata.imInfo=imInfo;
            
        end
        
        
        function [imageStack, originalMetadata, omeMetadata, metadata] = raw(fullFileName)
            % Loads a raw image file.
            
            try
                % use the bioformats library:
                [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.bioformats(fullFileName);
            catch
                
                error('Could not open the "%s" *.raw image stack with the Bioformats library. Check the stack with ImageJ and convert it to tiff or check whether the Bioformats library is installed.', fullFileName)
                
            end
            
        end
        
        
        function [imageStack, originalMetadata, omeMetadata, metadata] = nd2(fullFileName)
            % Loads a Nikon *.nd2 image file.
            
            % use the bioformats library:
            [imageStack, originalMetadata, omeMetadata, metadata] = imageStackReading.bioformats(fullFileName);
            
        end
        
    end
end
