function [algorithmFields, analysesFields] = ThunderSTORM_dataConversionDefinitions()

% ThunderSTORM conventions:
% - ThunderSTORM calcualtes everything in physical units (pixel size and photoelectrons (not the incident photons as it is shown in the data headers...))
% - the "x" axis is the vertical axis starting at the top left corner
% - the "y" axis is the horizontal axis starting at the top left corner
%   - the naming of the axes are basicly swapped with that of the raiNSOTRM, the starting pont and the directions match  
% - the sigma1 and sigma2 corresponds to the "x" and "y" axes, respectively
% - the "weighted least squares" and "least squares" algorithms returns a
%   fit error field (chi2), the "maximum likelihood" algorithm does not 

commonFields = [...
    struct(...
        'fieldname', 'frame_idx',...
        'header', 'frame',...
        'unit', 'camera frame interval',...
        'description','camera frame index',...
        'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'frame')),...
        'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.frame_idx...
        ),...
    struct(...
        'fieldname', 'x_coord',...
        'header', 'y [nm]',...
        'unit', 'nm',...
        'description','vertical distance from the top-left corner',...
        'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'y [nm]'))/cameraSignalConversion.pixelSize_nm,...
        'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.x_coord*cameraSignalConversion.pixelSize_nm...
        ),...
    struct(...
        'fieldname', 'y_coord',...
        'header', 'x [nm]',...
        'unit', 'nm',...
        'description','horizontal distance from the top-left corner',...
        'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'x [nm]'))/cameraSignalConversion.pixelSize_nm,...
        'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.y_coord*cameraSignalConversion.pixelSize_nm...
        )...
 ];

fit_error = [...
    struct(...
        'fieldname', 'chi2',...
        'header', 'chi2',...
        'unit', 'camera count^2',...
        'description','measure of the error of the fit',...
        'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'chi2')),...
        'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.chi2...
        )... 
];

GaussianGeneralFields = [
            struct(...
                'fieldname', 'background',...
                'header', 'offset [photon]',...
                'unit', 'photoelectron',...
                'description','fluorescent background',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'offset [photon]'))*cameraSignalConversion.countsPerElectron,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.background/cameraSignalConversion.countsPerElectron...
                ),...
            struct(...
                'fieldname', 'backgroundSTD',...
                'header', 'bkgstd [photon]',...
                'unit', 'photoelectron',...
                'description','fluorescent background noise',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'bkgstd [photon]'))*cameraSignalConversion.countsPerElectron,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.backgroundSTD/cameraSignalConversion.countsPerElectron...
                ),...
            struct(...
                'fieldname', 'sum_signal',...
                'header', 'intensity [photon]',...
                'unit', 'photoelectron',...
                'description','sum signal of the spots',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'intensity [photon]'))*cameraSignalConversion.countsPerElectron,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.sum_signal/cameraSignalConversion.countsPerElectron...
                ),...
            struct(...
                'fieldname', 'std',...
                'header', 'uncertainty [nm]',...
                'unit', 'nm',...
                'description','localization precision',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'uncertainty [nm]'))/cameraSignalConversion.pixelSize_nm,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.std*cameraSignalConversion.pixelSize_nm...
                )...
        ];

nonAstigmaticGaussianFields = [...
            struct(...
                'fieldname', 'sig',...
                'header', 'sigma [nm]',...
                'unit', 'nm',...
                'description','fitted width',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'sigma [nm]'))/cameraSignalConversion.pixelSize_nm,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.sig*cameraSignalConversion.pixelSize_nm...
                )...
                ];

astigmaticFields = [...
            struct(...
                'fieldname', 'z_coord',...
                'header', 'z [nm]',...
                'unit', 'nm',...
                'description','axial position',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'z [nm]'))/cameraSignalConversion.pixelSize_nm,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.z_coord*cameraSignalConversion.pixelSize_nm...
                ),...
            struct(...
                'fieldname', 'sig_x',...
                'header', 'sigma2 [nm]',...
                'unit', 'nm',...
                'description','fitted vertical PSF width',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'sigma2 [nm]'))/cameraSignalConversion.pixelSize_nm,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.sig_x*cameraSignalConversion.pixelSize_nm...
                ),...
            struct(...
                'fieldname', 'sig_y',...
                'header', 'sigma1 [nm]',...
                'unit', 'nm',...
                'description','fitted horizontal PSF width',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'sigma1 [nm]'))/cameraSignalConversion.pixelSize_nm,...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.sig_y*cameraSignalConversion.pixelSize_nm...
                )...
            ];



algorithmFields = [...
    struct(...
        'name', 'no estimator',...
        'fields',[...
            commonFields...
            ]...
    ),...
    struct(...
        'name', 'Radial symmetry',...
        'fields',[...
            commonFields...
            ]...
    ),...
    struct(...
        'name', 'Centroid of local neighborhood',...
        'fields',[...
            commonFields...
            ]...
    ),...
    struct(...
        'name', 'PSF: Gaussian, Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Gaussian, Weighted Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Gaussian, Maximum likelihood',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields...
            ]...
    ),...
    struct(...
        'name', 'PSF: Integrated Gaussian, Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Integrated Gaussian, Weighted Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Integrated Gaussian, Maximum likelihood',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            nonAstigmaticGaussianFields...
            ]...
    ),...
    struct(...
        'name', 'PSF: Elliptical Gaussian (3D astigmatism), Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            astigmaticFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Elliptical Gaussian (3D astigmatism), Weighted Least squares',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            astigmaticFields,...
            fit_error...
            ]...
    ),...
    struct(...
        'name', 'PSF: Elliptical Gaussian (3D astigmatism), Maximum likelihood',...
        'fields',[...
            commonFields,...
            GaussianGeneralFields,...
            astigmaticFields...
            ]...
    )...
    ];


analysesFields = [...
    struct(...
        'name', 'merging',...
        'fields',[...
            struct(...
                'fieldname', 'frameN',...
                'header', 'detections',...
                'unit', '',...
                'description','frame number of concatenated localizations',...
                'function_forward', @(dataMatrix, dataHeaders, cameraSignalConversion) dataMatrix(:, strcmp(dataHeaders, 'detections')),...
                'function_backward', @(pointillisticData, cameraSignalConversion) pointillisticData.frameN...
                )...
            ]...
    )...
    ];



end

% data headers with the differnt ThunderSTORM algorithms: 
% name: "Centroid of local neighborhood",
% data headers: "id","frame","x [nm]","y [nm]"
% 
% name: - (no estimator)
% data headers: "id","frame","x [nm]","y [nm]"
% 
% name: "Radial symmetry"
% data headers: "id","frame","x [nm]","y [nm]"
% 
% name: "PSF: Gaussian", "PSF: Integrated Gaussian"
% method: "Least squares", "Weighted Least squares"
% data headers: "id","frame","x [nm]","y [nm]","sigma [nm]","intensity [photon]","offset [photon]","bkgstd [photon]","chi2","uncertainty [nm]"
% 
% name: "PSF: Integrated Gaussian"
% method: "Maximum likelihood"
% data headers: "id","frame","x [nm]","y [nm]","sigma [nm]","intensity [photon]","offset [photon]","bkgstd [photon]","uncertainty [nm]"
% 
% name: "PSF: Elliptical Gaussian (3D astigmatism)"
% method: "Least squares"
% data headers: "id","frame","x [nm]","y [nm]","z [nm]","sigma1 [nm]","sigma2 [nm]","intensity [photon]","offset [photon]","bkgstd [photon]","chi2","uncertainty [nm]"
% 
% After merging
% method: "Maximum likelihood"
% data headers: "id","frame","x [nm]","y [nm]","sigma [nm]","intensity [photon]","offset [photon]","bkgstd [photon]","uncertainty [nm]","detections"