function [pointillisticData, analysisParameters] = ThunderSTORM_dataImport(dataFileFullName, definedCameraSignalConversion)

    [dataPath, dataName, ~] = fileparts(dataFileFullName);
    
    [dataHeaders, dataMatrix] = pointillisticData_load.headerAndMatrixData(dataFileFullName);
    dataHeaders = strrep(dataHeaders, '"', '');

    % the frame number and size information is not saved ny ThunderSTORM, so guess it.... 
    frameNumber = int32(max(dataMatrix(:, strcmp(dataHeaders, 'frame'))));
    frameSize_nm=ceil([max(dataMatrix(:, strcmp(dataHeaders, 'y [nm]'))), max(dataMatrix(:, strcmp(dataHeaders, 'x [nm]')))]);

    infoFileName = [dataName, '-protocol.txt'];
    infoFileFullName = fullfile(dataPath, infoFileName);

    infoText = pointillisticData_load.infoText(infoFileFullName);
    [analysisParameters, algorithm] = processInfoText(infoText, frameSize_nm, frameNumber);

    
    [analyses] = inferDataTypes(dataHeaders);
    convention = 'ThunderSTORM';
    
    analysisParameters.sharedParameters.convention.name = convention;
    analysisParameters.sharedParameters.convention.algorithm = algorithm;
    analysisParameters.sharedParameters.convention.analyses = analyses;

    % overwrite the parametrs that could not be read out from the
    % metadata with the default ones:
    loadedCameraSignalConversion = analysisParameters.sharedParameters.cameraSignalConversion;
    missingOnlyBoolean = true;
    overwrittenCameraSignalConversion = imageStack_getConversionFactors.overwriteParameters(loadedCameraSignalConversion, definedCameraSignalConversion, missingOnlyBoolean);
    analysisParameters.sharedParameters.cameraSignalConversion = overwrittenCameraSignalConversion;

    analysisParameters.sharedParameters.originalDataPath = dataPath;
    analysisParameters.sharedParameters.originalDataName = dataName;

    pointillisticData = convertImportedData(dataMatrix, dataHeaders, convention, algorithm, analyses, overwrittenCameraSignalConversion);
end


function [analysisParameters, algorithm] = processInfoText(infoText, frameSize_nm, frameNumber)


versionString = extractBefore(infoText, 'Input:');
inputCell = extractBetween(infoText, 'Input:', 'Camera settings:');
inputString = inputCell{1};

cameraSettingsCell = extractBetween(infoText, 'Camera settings:', 'Analysis:');
cameraSettingsString = cameraSettingsCell{1};

analysisCell = extractBetween(infoText, 'Analysis:', 'Post-processing:');
analysisString = analysisCell{1};
splitPositions = strfind(analysisString, ['}',newline,'{']);
if numel(splitPositions)==2
    analysisString_part1 = extractBetween(analysisString, 1, splitPositions(1)+1);
    analysisString_part1 = analysisString_part1{1};
    imageFiltering = jsondecode(analysisString_part1);
    
    analysisString_part2 = extractBetween(analysisString, splitPositions(1)+1, splitPositions(2)+1);
    analysisString_part2 = analysisString_part2{1};
    approximateLocalizationOfMolecules = jsondecode(analysisString_part2);

    analysisString_part3 = extractBetween(analysisString, splitPositions(2)+1, numel(analysisString));
    analysisString_part3 = analysisString_part3{1};
    subPixelLocalizationOfMolecules = jsondecode(analysisString_part3);
else
    error('Something is wrong with the ThunderSTORM protocols file, probably. This case should never happen, as I tested.')
end
postprocessingString = extractAfter(infoText, 'Post-processing:');


settings_TS.version = strtrim(versionString);
settings_TS.input = jsondecode(inputString);
settings_TS.cameraSettings = jsondecode(cameraSettingsString);

settings_TS.imageFiltering = imageFiltering;
settings_TS.approximateLocalizationOfMolecules = approximateLocalizationOfMolecules;
settings_TS.subPixelLocalizationOfMolecules = subPixelLocalizationOfMolecules;
%settings_TS.visualizationOfTheResults

settings_TS.postprocessing = jsondecode(postprocessingString);

imageStackReading_settings.dataFullFileName = settings_TS.input;
imageStackReading_settings.imageType='';
imageStackReading_settings.cameraPropertiesFile='';
imageStackReading_settings.additionalMagnification=NaN;

frameSize = ceil(frameSize_nm / settings_TS.cameraSettings.pixelSize);
imageStackReading_metadata.stackSize = [frameSize, frameNumber];

imageStackReading_metadata.cameraSettings.manufacturer = '';
imageStackReading_metadata.cameraSettings.model = '';
imageStackReading_metadata.cameraSettings.serialNumber = '';
imageStackReading_metadata.cameraSettings.exposureTime_ms = NaN;
imageStackReading_metadata.cameraSettings.frameInterval_ms = NaN;
imageStackReading_metadata.cameraSettings.pixelSize_nm  = settings_TS.cameraSettings.pixelSize;
imageStackReading_metadata.cameraSettings.emissionWavelength_nm = NaN;

imageStackReading_metadata.cameraSettings.settings.readOutRate = NaN;
imageStackReading_metadata.cameraSettings.settings.conversionGain = NaN;
imageStackReading_metadata.cameraSettings.settings.amplificationGain = settings_TS.cameraSettings.gain;
imageStackReading_metadata.cameraSettings.settings.emMultiplication = settings_TS.cameraSettings.isEmGain;

imageStackReading_metadata.cameraSignalConversion.pixelSize_nm  = settings_TS.cameraSettings.pixelSize;
imageStackReading_metadata.cameraSignalConversion.exposureTime_ms = NaN;
imageStackReading_metadata.cameraSignalConversion.frameInterval_ms = NaN;
imageStackReading_metadata.cameraSignalConversion.countsPerPhoton = NaN;
imageStackReading_metadata.cameraSignalConversion.countsPerElectron = settings_TS.cameraSettings.gain/settings_TS.cameraSettings.photons2ADU;
imageStackReading_metadata.cameraSignalConversion.baseline_count = settings_TS.cameraSettings.offset;

if isfield(settings_TS.subPixelLocalizationOfMolecules, 'name')
    TS_method_PSF = settings_TS.subPixelLocalizationOfMolecules.name;
else
    TS_method_PSF = 'no estimator';
end
if contains(TS_method_PSF, 'PSF:')
    
    TS_fitting_method = settings_TS.subPixelLocalizationOfMolecules.parameters.trackedParameters.method.value;
    
    algorithm = [TS_method_PSF, ', ', TS_fitting_method];

else
    algorithm = TS_method_PSF;
end


localizationSettings.imageFiltering = settings_TS.imageFiltering;
localizationSettings.approximateLocalizationOfMolecules = settings_TS.approximateLocalizationOfMolecules;
localizationSettings.subPixelLocalizationOfMolecules = settings_TS.subPixelLocalizationOfMolecules;
localizationSettings.visualizationOfTheResults = struct(); % it is not stored in the *protocol.txt

localizationMetadata.frameRange = [1, frameNumber];
localizationMetadata.ROI.x = [0, frameSize(1)];
localizationMetadata.ROI.y = [0, frameSize(2)];
localizationMetadata.ROI.unit = 'camera pixel length';

sharedParameters.cameraSignalConversion  = imageStackReading_metadata.cameraSignalConversion;
sharedParameters.stackSize = imageStackReading_metadata.stackSize;
sharedParameters.frameRange = localizationMetadata.frameRange;
sharedParameters.ROI = localizationMetadata.ROI;



analysisParameters_imageStack.dataType = 'imageStack';
analysisParameters_imageStack.note = 'ThunderSTORM localization data import';
analysisParameters_imageStack.creationOperation = {...
    struct(...
    'operationType','image stack loading',...
    'note','ThunderSTORM localization data import',...
    'function','',...
    'settings',imageStackReading_settings,...
    'metadata',imageStackReading_metadata...
    )...
};
analysisParameters_imageStack.processingOperations = {};
analysisParameters_imageStack.derivativeDatas = {};

% the localization data is the upper in the hierarchy within the
% "analysisParameters" in case of the legacy rainSTORM
analysisParameters.dataType = 'localizationData';
analysisParameters.note = 'ThunderSTORM localization data import';
analysisParameters.creationOperation = {...
    struct(...
    'operationType','localization',...
    'note','ThunderSTORM localization data import',...
    'function',algorithm,...
    'settings',localizationSettings,...
    'metadata',localizationMetadata...
    )...
    };
analysisParameters.processingOperations = {};
analysisParameters.derivativeDatas = {analysisParameters_imageStack};
analysisParameters.sharedParameters = sharedParameters;


end

function [analyses] = inferDataTypes(dataHeaders)

if any(strcmp(dataHeaders, 'detections'))
    analyses = {'merging'};
else
    analyses = {};
end


end