classdef rainSTORM_GUI_astigmatic3D_calculation < handle
    
    properties
        
        specificElements
        
        pointillisticDataObject
        calibrationData
        
        methodList
        formulaList
        availableSettings
        chosenMethod
        
        rowPositions
        elementHeight
        
    end
    
    methods

        function pointillisticDataObject = create(someStoringHandleObject, pointillisticDataObject, astigmaticSettings)

            %  place it in the calling function
            % xx.pointillistidDataObject = somesStoringHandleObject.pointillistidDataObject;
            
            rowNumber = 8;

            margin = 0.05;

            rowheight = (1-margin)/rowNumber;
            rowPositions = ((1:rowNumber)-1)*rowheight + margin;
            elementHeight = (1-(rowNumber+1)*margin)/rowNumber;
            
            someStoringHandleObject.rowPositions = rowPositions;
            someStoringHandleObject.elementHeight = elementHeight;

            windowObject = dialog();
            % for debugging:
            %windowObject = figure();

            % replace it with some other handle class!!!!!!!
            someStoringHandleObject.pointillisticDataObject = pointillisticDataObject;
            someStoringHandleObject.calibrationData = [];

            someStoringHandleObject.methodList = rainSTORM_GUIsettings_astigmatic3D.methods();
            someStoringHandleObject.formulaList = rainSTORM_GUIsettings_astigmatic3D.ellipticityFormulas();
            someStoringHandleObject.availableSettings = {};
            someStoringHandleObject.chosenMethod = astigmaticSettings.method;


            for idxMethod = 1:numel(someStoringHandleObject.methodList)

              method = someStoringHandleObject.methodList{idxMethod};

              if strcmp(method, astigmaticSettings)
                someStoringHandleObject.availableSettings{idxMethod} = astigmaticSettings;
              else
                someStoringHandleObject.availableSettings{idxMethod} = rainSTORM_GUIsettings_astigmatic3D.settings(method);
              end

            end



            uicontrol('Style','text','Parent',windowObject,...
                'String', 'Calibration measurement file:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'Position', [0.05 rowPositions(8) 0.4 elementHeight]);

            calibrationFile_text = uicontrol('Style','text','Parent',windowObject,...
                'String', '',....
                'Tooltip', 'calibration file path',...
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'Position', [0.05 rowPositions(7) + elementHeight/2 0.9 elementHeight/2]);
            
            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'calibrationImport_button',...
                'Callback',@(ObjH, EventData) import_callback(calibrationFile_text, someStoringHandleObject),...
                'String', 'Import',....
                'Tooltip', 'Import an astigmatic calibration file.',...
                'Units', 'normal',...
                'Position', [0.55 rowPositions(8) 0.4 elementHeight]);


            uicontrol('Style','text','Parent',windowObject,...
                'String', 'Method:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'Position', [0.05 rowPositions(6) 0.4 elementHeight]);

            uicontrol('Style','popupmenu','Parent',windowObject,...
                'String', someStoringHandleObject.methodList,....
                'Value', find(strcmp(someStoringHandleObject.chosenMethod, someStoringHandleObject.methodList)),...
                'CreateFcn', @(ObjH, EventData) method_createfcn(windowObject, someStoringHandleObject),...
                'Callback', @(ObjH, EventData) method_callback(windowObject, someStoringHandleObject),...
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'Position', [0.55 rowPositions(6) 0.4 elementHeight]);

            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'calibrationShow_button',...
                'Callback',@(ObjH, EventData) show_callback(someStoringHandleObject),...
                'String', 'Show Calibration',....
                'Tooltip', 'Visualize the calibration data.',...
                'Units', 'normal',...
                'Position', [0.05 rowPositions(2) 0.3 elementHeight]);

            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'calculate_button',...
                'Callback',@(ObjH, EventData) calculate_callback(someStoringHandleObject),...
                'String', 'Calculate',....
                'Tooltip', 'Calculate the axial (z) positions.',...
                'Units', 'normal',...
                'Position', [0.35 rowPositions(2) 0.3 elementHeight]);

            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'remove_button',...
                'Callback',@(ObjH, EventData) remove_callback(someStoringHandleObject),...
                'String', 'Remove',....
                'Tooltip', 'Remove the axial (z) positions data field.',...
                'Units', 'normal',...
                'Position', [0.65 rowPositions(2) 0.3 elementHeight]);



            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'cancel_button',...
                'Callback',@(ObjH, EventData) cancel_callback(someStoringHandleObject, pointillisticDataObject, windowObject),...
                'String', 'Cancel',....
                'Tooltip', 'Cancel the astigmatic 3D calculation.',...
                'Units', 'normal',...
                'Position', [0.35 rowPositions(1) 0.3 elementHeight]);

            uicontrol('Style','pushbutton','Parent',windowObject,...
                'Tag', 'ok_button',...
                'Callback','delete(gcf)',...
                'String', 'Ok',....
                'Tooltip', 'Apply the astigmatic 3D calculation.',...
                'Units', 'normal',...
                'Position', [0.65 rowPositions(1) 0.3 elementHeight]);


            uiwait(windowObject);

        end

    end
    
end


function common_settings(windowObject, someStoringHandleObject)
       
rowPositions = someStoringHandleObject.rowPositions;
elementHeight = someStoringHandleObject.elementHeight;

uicontrol('Style','text','Parent',windowObject,...
    'String', 'Polynomial order:',....
    'HorizontalAlignment', 'left',...
    'Units', 'normal',...
    'Position', [0.05 rowPositions(5) 0.4 elementHeight]);

uicontrol('Style','edit','Parent',windowObject,...
    'Tag', 'polynomialOrder_edit',...
    'String', get_settingsField_numeric(someStoringHandleObject, 'polynomialOrder.value', 1),....
    'Callback', @(ObjH, EventData) set_settingsField_numeric(someStoringHandleObject, 'polynomialOrder', 1),...
    'Tooltip', 'Order of the polynomial to be fitted on the calibration data.',...
    'Units', 'normal',...
    'Position', [0.55 rowPositions(5) 0.2 elementHeight]);



uicontrol('Style','text','Parent',windowObject,...
    'String', ['Z bounds [', get_settingsField_text(someStoringHandleObject, 'zBounds.unit'), ']:'],....
    'HorizontalAlignment', 'left',...
    'Units', 'normal',...
    'Position', [0.05 rowPositions(4) 0.4 elementHeight]);

uicontrol('Style','edit','Parent',windowObject,...
    'Tag', 'lowerBound_edit',...
    'String', get_settingsField_numeric(someStoringHandleObject, 'zBounds.value', 1),....
    'Callback', @(ObjH, EventData)  set_settingsField_numeric(someStoringHandleObject, 'zBounds.value', 1),...
    'Tooltip', 'Lower bound for the Z (axial) position calculation in nm.',...
    'Units', 'normal',...
    'Position', [0.55 rowPositions(4) 0.2 elementHeight]);

uicontrol('Style','edit','Parent',windowObject,...
    'Tag', 'upperBound_edit',...
    'String', get_settingsField_numeric(someStoringHandleObject, 'zBounds.value', 2),....
    'Callback', @(ObjH, EventData) set_settingsField_numeric(someStoringHandleObject, 'zBounds.value', 2),...
    'Tooltip', 'Upper bound for the Z (axial) position calculation in nm.',...
    'Units', 'normal',...
    'Position', [0.75 rowPositions(4) 0.2 elementHeight]);
        
end


function method_elements(windowObject, someStoringHandleObject, chosenMethod)

    rowPositions = someStoringHandleObject.rowPositions;
    elementHeight = someStoringHandleObject.elementHeight;

  for i = 1:numel(someStoringHandleObject.specificElements)
    delete(someStoringHandleObject.specificElements{i});
  end
  % create the GUI elements for the method settings:
  switch chosenMethod
    case 'sigma squared error'

      common_settings(windowObject, someStoringHandleObject)
        
      someStoringHandleObject.specificElements{1} = uicontrol('Style','text','Parent',windowObject,...
          'String', 'resampling number:',....
          'HorizontalAlignment', 'left',...
          'Units', 'normal',...
          'Position', [0.05 rowPositions(3) 0.4 elementHeight]);

      someStoringHandleObject.specificElements{2} = uicontrol('Style','edit','Parent',windowObject,...
          'Tag', 'sampling_number_edit',...
          'String', get_settingsField_numeric(someStoringHandleObject, 'resamplingNumber.value', 1),....
          'Callback', @(ObjH, EventData) set_settingsField_numeric(someStoringHandleObject, 'resamplingNumber', 1),...
          'Tooltip', 'Number of sampling points of the fitted function for the "sigma squared error" method.',...
          'Units', 'normal',...
          'Position', [0.55 rowPositions(3) 0.2 elementHeight]);

    case 'fit on ellipticity'

      common_settings(windowObject, someStoringHandleObject)
        
      someStoringHandleObject.specificElements{1} = uicontrol('Style','text','Parent',windowObject,...
          'String', 'ellipticity formula:',....
          'HorizontalAlignment', 'left',...
          'Units', 'normal',...
          'Position', [0.05 rowPositions(3) 0.4 elementHeight]);

      someStoringHandleObject.specificElements{2} = uicontrol('Style','popupmenu','Parent',windowObject,...
          'Tag', 'formula_popupmenu',...
          'String', someStoringHandleObject.formulaList,....
          'Value', get_settingsField_list(someStoringHandleObject, someStoringHandleObject.formulaList, 'formula'),...
          'Callback', @(ObjH, EventData) set_settingsField_list(someStoringHandleObject, 'formula'),...
          'Tooltip', 'Formula for the ellipticity calculation for the "fit on ellipticity" method..',...
          'Units', 'normal',...
          'Position', [0.55 rowPositions(3) 0.2 elementHeight]);

    case 'fit on sigmas'
      common_settings(windowObject, someStoringHandleObject)
    otherwise
      error('Unknown astigmatic 3D method: %s', chosenMethod)
  end
    
end


function text = get_settingsField_text(someStoringHandleObject, referencedSubField)

  idxMethod = strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod);

  referencedSubField_parts = strsplit(referencedSubField, '.');
  
  settingsStruct = someStoringHandleObject.availableSettings{idxMethod}.settings;

  %mainField = referencedSubField_parts{1};
  %subField = referencedSubField_parts{2};
  %text = settingsStruct.(mainField).(subField);
  text = getfield(settingsStruct, referencedSubField_parts{:});

end


function valueText = get_settingsField_numeric(someStoringHandleObject, referencedSubField, valueIndex)

  idxMethod = strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod);
  
  referencedSubField_parts = strsplit(referencedSubField, '.');
  
  settingsStruct = someStoringHandleObject.availableSettings{idxMethod}.settings;

  %mainField = referencedSubField_parts{1};
  %subField = referencedSubField_parts{2};
  %valueText = num2str(settingsStruct.(mainField).(subField)(valueIndex));
  values = getfield(settingsStruct, referencedSubField_parts{:});
  valueText = values(valueIndex);

end


function set_settingsField_numeric(someStoringHandleObject, fieldNames, index)
    
  value = str2double(get(gcbo, 'String'));

  idxMethod = find(strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod));

  fieldName_list = strsplit(fieldNames, '.');
  
  % get the original value and modify one of its element:
  settingsValue = getfield(someStoringHandleObject.availableSettings{idxMethod}.settings, fieldName_list{:});
  settingsValue(index) = value;
  
  % the "setfield"doues not changes the handle object, but converts it to
  % struct...
  modifiedSettings = setfield(someStoringHandleObject.availableSettings{idxMethod}.settings, fieldName_list{:}, settingsValue);
  someStoringHandleObject.availableSettings{idxMethod}.settings = modifiedSettings;
end

function index = get_settingsField_list(someStoringHandleObject, list, fieldName)

  idxMethod = strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod);
  
  settingsStruct = someStoringHandleObject.availableSettings{idxMethod}.settings.(fieldName);

  index = find(strcmp(list, settingsStruct.value));

end


function set_settingsField_list(someStoringHandleObject, fieldName)

  contents = cellstr(get(gcbo,'String'));
  chosenElement = contents{get(gcbo,'Value')};

  idxMethod = strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod);

  someStoringHandleObject.availableSettings{idxMethod}.settings.(fieldName).value = chosenElement;

end

function method_createfcn(windowObject, someStoringHandleObject)

    chosenMethod = someStoringHandleObject.chosenMethod;
    
    method_elements(windowObject, someStoringHandleObject, chosenMethod);
    
end

function method_callback(windowObject, someStoringHandleObject)

    contents = cellstr(get(gcbo,'String'));
    chosenMethod = contents{get(gcbo,'Value')};

    someStoringHandleObject.chosenMethod = chosenMethod;

    method_elements(windowObject, someStoringHandleObject, chosenMethod);

end


function import_callback(calibrationFile_text, someStoringHandleObject)

  pointillisticDataObject = someStoringHandleObject.pointillisticDataObject;
  
  dataPath = pointillisticDataObject.analysisParameters.sharedParameters.originalDataPath;

  [fileName, filePath] = uigetfile({"*.csv", "Coma Separated Value files";"*.*", "All Files"}, 'Import astigmatic 3D calibration file', dataPath);

  calibrationFullFileName = fullfile(filePath, fileName);

  calibrationData = readtable(calibrationFullFileName);
  someStoringHandleObject.calibrationData = calibrationData;

  set(calibrationFile_text, 'String', fileName);
  set(calibrationFile_text, 'Tooltip', calibrationFullFileName);

end


function show_callback(someStoringHandleObject)

  if isempty(someStoringHandleObject.calibrationData)

    warning('Cannot visualize the astigmatic 3D calibration data. Import an astigmatic 3D calibration data first.');
    return
  end

  rainSTORM_astigmatic3D_calibration.visualize(someStoringHandleObject.calibrationData)

  % by default, Matlab places the windows too high, its header (graggable
  % area, close button) is outs of the screen....
  Position = get(gcf, 'Position');
  Position(1) = 400;
  Position(2) = 200;
  set(gcf, 'Position', Position);
  
  
end


function calculate_callback(someStoringHandleObject)

  if isempty(someStoringHandleObject.calibrationData)

    warning('Cannot calculate the astigmatic 3D axial positions. Import an astigmatic 3D calibration data first.');
    return
  end

  calibrationData = someStoringHandleObject.calibrationData;


  astigmaticSettings.method = someStoringHandleObject.chosenMethod;
  idxMethod = strcmp(someStoringHandleObject.methodList, someStoringHandleObject.chosenMethod);

  astigmaticSettings = someStoringHandleObject.availableSettings{idxMethod};

  astigmaticSettings = rainSTORM_GUI_settings_extract.operation(astigmaticSettings);

  someStoringHandleObject.pointillisticDataObject.calculateAstigmatic3D(calibrationData, astigmaticSettings);

  disp('Axial coordinate calculation is finished.')
  
end


function remove_callback(someStoringHandleObject)

  someStoringHandleObject.pointillisticDataObject.undoAstigmatic3D();

end

function cancel_callback(someStoringHandleObject, pointillisticDataObject, windowObject)

% revert to the original data
someStoringHandleObject.pointillisticDataObject = pointillisticDataObject;
% TODO: the pointillisticDataClass is a handle class, so it is not reverted
% to the original

delete(windowObject)

end
