classdef rainSTORM_GUI_concatenation < handle
    
    properties
        
        concatenationSettings
        pixelizationOperationSettings
        visualizationOperationSettings
        
        localizationDataObject
        trajectoryRawData
        simpleTrajectories
        complexTrajectories
        
        trajectoryImageMatrix
        pixelCenters
        
        uipanel_trajReconSettings
        uipanel_Traj_Settings_slider
        uipanel_trajGraphSettingsPanel
        uipanel_trajGraphPlotPanel
        
        distThresholdField
        ZPrecConstField
        maxNumberOfNoNewLocsInARowField
        
        filterNPhotonBoundsField
        filterFrameDurationBoundsField
        filterEmPowerBoundsField
        filterLifeTimeBoundsField

        exposureTimeField
        frameTimeField
        photonPerCountField
        NBinsField
        thresPixValNField
        mapPixelSizeField
        
        reconTrajPlotButtonGroup
        scatterMapRadioButton
        imageMapRadioButton
        reconTrajExpectationValueButtonGroup
        trajExpFitRadioButton
        trajFitResRadioButton
        trajMeanForDecayRadioButton
        trajMeanOfValRadioButton

        traj_qualityHistButton
        traj_qualityHistTrackerButton
        traj_photoPhysicsHistButton
        traj_photoPhysicsHistTrackerButton
        traj_NPhotonPlotButton
        traj_frameDurationPlotButton
        traj_emPowerPlotButton
        traj_lifeTimePlotButton
        traj_recalcButton
        traj_importButton
        traj_exportButton
        traj_exportFilteredToCSVButton
        
        includeComplexTrajsCheckbox
        
    end
    
    methods
        
        function create(someStoringHandleObject, pointillisticDataObject, operationGUISettings, pixelizationOperationSettings, visualizationOperationSettings)
            % 	Summary:
            %	Version:    3.2
            %	Note:
            %	Package:	rainSTORM v3.1+
            %	Developer:	novakt
            %	Name:		Novák Tibor
            %	Company:	University of Szeged
            %	Date of the merger: 2017.11.08.
            
            % This function creatres the GUI elements on the trajectory fitting's
            % "uipanel_trajGraphSettingsPanel" (for setting the maps and histograms)
            % and  on the "uipanel_trajGraphPlotPanel" (for plotting the maps and
            % histograms) and handles its the action calls
            
            
            %handles = guidata(gcbo);
            
            someStoringHandleObject.localizationDataObject = pointillisticDataObject;
            concatenationSettings = rainSTORM_GUI_settings_extract.operation(operationGUISettings);
            someStoringHandleObject.concatenationSettings = concatenationSettings;
            someStoringHandleObject.pixelizationOperationSettings = pixelizationOperationSettings;
            someStoringHandleObject.visualizationOperationSettings = visualizationOperationSettings;
            
            analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
            cameraSignalConversion = analysisParameters.sharedParameters.cameraSignalConversion;
            
            % some of the window geometry's definition:
            windowWidth=500;
            windowHeight=200;
            windowObject = figure('Position',[300 300 windowWidth windowHeight]);
            
            panel_pos = get(windowObject,'Position');    % "x", "y" positins of the bottom left corner and the width and height of the panel
            panel_unit = get(windowObject,'Unit');       % units the GUI element positions, e.g. 'Pixel' (I guess 'Pixel' is set), 'Normalized'
            panelHeightRatio=0.85;  % the height of the panel on which the input fields and other GUI elements should be placed is that much less than the height of the original panel, needed because of the radiobuttons
            
            panel_line_space = 30;
            panel_first_line_pos = panel_pos(4)*panelHeightRatio-panel_line_space/2;
            panel_max_lines = floor(panel_pos(4)*panelHeightRatio / panel_line_space)-1;
            
            
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            % PANELS
            
            % create panel for the 'Trajectory Fitting' algorithm settings
            someStoringHandleObject.uipanel_trajReconSettings=uipanel(windowObject,...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.0 0.0 1 panelHeightRatio],...
                'Visible', 'On');
            
            someStoringHandleObject.uipanel_Traj_Settings_slider=uicontrol('Style','Slider','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'Units','normalized','Position',[0.95 0 0.05 1],...
                'Visible','Off',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Value',1);
            
            someStoringHandleObject.uipanel_trajGraphSettingsPanel=uipanel(windowObject,...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.0 0.0 1 panelHeightRatio],...
                'Visible', 'Off');
            
            someStoringHandleObject.uipanel_trajGraphPlotPanel=uipanel(windowObject,...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.0 0.0 1 panelHeightRatio],...
                'Visible', 'Off');
            
            
            % button group for the radio buttons, for the two radioubuttons
            % could not be checked at once
            reconTrajButtonGroup = uibuttongroup(windowObject,...
                'Visible','on',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0 panelHeightRatio 1 1-panelHeightRatio]...
                );
            %'SelectionChangedFcn',@(reconTrajButtonGroup, event) reconTrajButtonGroup_SelectionChangedFcn(reconTrajButtonGroup, event));
            %'SelectionChangedFcn',@(hObject, handles) bselection(hObject, handles));
            %'SelectionChangedFcn',@(hObject, handles) reconTrajButtonGroup_SelectionChangedFcn(hObject, handles));
            
            % Create three radio buttons in the button group.
            uicontrol(reconTrajButtonGroup,...
                'Style','radiobutton',...
                'String','Concatenation settings',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.1 0.0 0.4 1],...
                'Callback', @(ObjH, EventData) reconTrajReconSettingsRadioButton_Callback(someStoringHandleObject),...
                'Tag', 'reconTrajSettingsRadioButton',...
                'HandleVisibility','off');
            
            uicontrol(reconTrajButtonGroup,...
                'Style','radiobutton',...
                'String','Visualization settings',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.4 0.0 0.4 1],...
                'Callback', @(ObjH, EventData) reconTrajGraphSettingsRadioButton_Callback(someStoringHandleObject),...
                'Tag', 'reconTrajGraphsRadioButton',...
                'HandleVisibility','off');
            
            uicontrol(reconTrajButtonGroup,...
                'Style','radiobutton',...
                'String','Calculate/Visualize',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.7 0.0 0.4 1],...
                'Callback', @(ObjH, EventData) reconTrajGraphPlotRadioButton_Callback(someStoringHandleObject),...
                'Tag', 'reconTrajGraphsRadioButton',...
                'HandleVisibility','off');
            
            
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            % PANEL 1
            
            concatenationGUISettings = operationGUISettings.settings;
            
            
            %% Concatenation threshold value
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'String', [concatenationGUISettings.distThreshold_nm.name, ' [nm]:'],....
                'Tooltip', concatenationGUISettings.distThreshold_nm.description,....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.225 0.65 0.25 0.1]);
            
            someStoringHandleObject.distThresholdField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'Tag', 'distThreshold',...
                'String', num2str(concatenationGUISettings.distThreshold_nm.value),....
                'Tooltip', concatenationGUISettings.distThreshold_nm.description,...
                'Units', 'normal',...
                'Position', [0.5 0.65 0.2 0.1]);
            
            %% Z threshold scaling
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'String', [concatenationGUISettings.ZPrecConst.name, ':'],....
                'Tooltip', concatenationGUISettings.ZPrecConst.description,....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.225 0.45 0.25 0.1]);
            
            someStoringHandleObject.ZPrecConstField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'Tag', 'ZPrecConst',...
                'String', num2str(concatenationGUISettings.ZPrecConst.value),....
                'Tooltip', concatenationGUISettings.ZPrecConst.description,...
                'Units', 'normal',...
                'Position', [0.5 0.45 0.2 0.1]);
            
            %% Max. empty frames
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'String', [concatenationGUISettings.maxNumberOfNoNewLocsInARow.name, ':'],....
                'Tooltip', concatenationGUISettings.maxNumberOfNoNewLocsInARow.description,....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.225 0.25 0.25 0.1]);
            
            someStoringHandleObject.maxNumberOfNoNewLocsInARowField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajReconSettings,...
                'Tag', 'maxNumberOfNoNewLocsInARow',...
                'String', num2str(concatenationGUISettings.maxNumberOfNoNewLocsInARow.value),....
                'Tooltip', concatenationGUISettings.maxNumberOfNoNewLocsInARow.description,...
                'Units', 'normal',...
                'Position', [0.5 0.25 0.2 0.1]);
            
            
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            % PANEL 2
            
            %% Buttons for plotting histograms
            someStoringHandleObject.traj_qualityHistButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_qualityHistButton',...
                'Callback',@(ObjH, EventData) traj_plotHist(someStoringHandleObject, 'traj_qualityHist'),...
                'String', 'Quality Histogram',....
                'Tooltip', 'Plot the histogram of the localization quality.',...
                'Units', 'normal',...
                'Position', [0.05 0.85 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_qualityHistTrackerButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_qualityHistTrackerButton',...
                'Callback',@(ObjH, EventData) traj_plotHist(someStoringHandleObject, 'traj_qualityHistTracker'),...
                'String', 'Quality Hist. Tracker',....
                'Tooltip', 'Plot the histogram of the localization quality of the region to be selected.',...
                'Units', 'normal',...
                'Position', [0.05 0.7 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_photoPhysicsHistButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_photoPhysicsHistButton',...
                'Callback',@(ObjH, EventData) traj_plotHist(someStoringHandleObject, 'traj_photoPhysicsHist'),...
                'String', 'Photophys. Histogram',....
                'Tooltip', 'Plot the histogram of the photophysics parameters.',...
                'Units', 'normal',...
                'Position', [0.525 0.85 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_photoPhysicsHistTrackerButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_photoPhysicsHistTrackerButton',...
                'Callback',@(ObjH, EventData) traj_plotHist(someStoringHandleObject, 'traj_photoPhysicsHistTracker'),...
                'String', 'Photophys. Hist. Tracker',....
                'Tooltip', 'Plot the histogram of the photophysics parameters of the region to be selected.',...
                'Units', 'normal',...
                'Position', [0.525 0.7 0.425 0.1],...
                'Enable','off');
            
            %% Buttons for the the maps of different quantities
            someStoringHandleObject.traj_NPhotonPlotButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_NPhotonPlotButton',...
                'Callback',@(ObjH, EventData) traj_plotMap(someStoringHandleObject, 'traj_NPhotonPlot'),...
                'String', ' N Photon Map',....
                'Tooltip', 'Plot the photon numbers of localized events.',...
                'Units', 'normal',...
                'Position', [0.05 0.55 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_frameDurationPlotButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_frameDurationPlotButton',...
                'Callback',@(ObjH, EventData) traj_plotMap(someStoringHandleObject, 'traj_frameDurationPlot'),...
                'String', ' Frame Duration Map',....
                'Tooltip', 'Plot the frames duration of localized events.',...
                'Units', 'normal',...
                'Position', [0.05 0.4 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_emPowerPlotButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_emPowerPlotButton',...
                'Callback',@(ObjH, EventData) traj_plotMap(someStoringHandleObject, 'traj_emPowerPlot'),...
                'String', 'Emission Power Map',....
                'Tooltip', 'Plot emission powers (photon/sec) of localized events (at leat 3 frames duration).',...
                'Units', 'normal',...
                'Position', [0.525 0.55 0.425 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_lifeTimePlotButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Tag', 'traj_lifeTimePlotButton',...
                'Callback',@(ObjH, EventData) traj_plotMap(someStoringHandleObject, 'traj_lifeTimePlot'),...
                'String', 'Lifetime Map',....
                'Tooltip', 'Plot lifetimes of localized events (at leat 3 frames duration).',...
                'Units', 'normal',...
                'Position', [0.525 0.4 0.425 0.1],...
                'Enable','off');
            
            %% Button for calculating/recalculating trajectory fitting
            someStoringHandleObject.traj_recalcButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Callback',@(ObjH, EventData) traj_recalc(someStoringHandleObject),...
                'String', 'Calc/Recalc',....
                'Tooltip', 'Calculate or recalculate trajectry fitting.',...
                'Units', 'normal',...
                'Position', [0.35 0.25 0.3 0.1]);
            
            %% Button for exporting or importing the fitted trajectory data
            someStoringHandleObject.traj_importButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Callback',@(ObjH, EventData) traj_import(someStoringHandleObject),...
                'String', 'Import',....
                'Tooltip', 'Import previouly calculated trajectry fit data.',...
                'Units', 'normal',...
                'Position', [0.05 0.1 0.2666 0.1],...
                'Enable','on');
            
            someStoringHandleObject.traj_exportButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Callback',@(ObjH, EventData) traj_export(someStoringHandleObject),...
                'String', 'Export',....
                'Tooltip', 'Export newly calculated trajectry fit data.',...
                'Units', 'normal',...
                'Position', [0.3666 0.1 0.2666 0.1],...
                'Enable','off');
            
            someStoringHandleObject.traj_exportFilteredToCSVButton=uicontrol('Style','pushbutton','Parent',someStoringHandleObject.uipanel_trajGraphPlotPanel,...
                'Callback',@(ObjH, EventData) traj_exportFilteredToCSV(someStoringHandleObject),...
                'String', 'Exp. filt. csv',....
                'Tooltip', 'Export filtered trajectry fit data to *.csv.',...
                'Units', 'normal',...
                'Position', [0.683 0.1 0.2666 0.1],...
                'Enable','off');
            
            %% Photon number lower and upper bounds for filtering
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Photon N:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.45 0.25 0.1]);
            
            someStoringHandleObject.filterNPhotonBoundsField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'filterNPhotonBoundsField',...
                'String', num2str([-Inf Inf]),....
                'Tooltip', 'Upper and lower photon number bounds for filtering the trajectories.',...
                'Units', 'normal',...
                'Position', [0.3 0.45 0.2 0.1]);
            
            %% Frame number lower and upper bounds for filtering
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Frame N:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.35 0.25 0.1]);
            
            someStoringHandleObject.filterFrameDurationBoundsField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'filterFrameDurationBoundsField',...
                'String', num2str([-Inf Inf]),....
                'Tooltip', 'Upper and lower frame duration bounds for filtering the trajectories.',...
                'Units', 'normal',...
                'Position', [0.30 0.35 0.2 0.1]);
            
            %% Emission power lower and upper bounds for filtering
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Em. power:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.25 0.25 0.1]);
            
            someStoringHandleObject.filterEmPowerBoundsField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'filterEmPowerBoundsField',...
                'String', num2str([-Inf Inf]),....
                'Tooltip', 'Upper and lower emission power bounds for filtering the trajectories.',...
                'Units', 'normal',...
                'Position', [0.3 0.25 0.2 0.1]);
            
            %% Lifetime lower and upper bounds for filtering
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Lifetime:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.15 0.25 0.1]);
            
            someStoringHandleObject.filterLifeTimeBoundsField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'filterLifeTimeBoundsField',...
                'String', num2str([-Inf Inf]),....
                'Tooltip', 'Upper and lower lifetime bounds for filtering the trajectories.',...
                'Units', 'normal',...
                'Position', [0.3 0.15 0.2 0.1]);
            
            
            %% Exposure time
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Expos. time [s]:',...
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.85 0.3 0.1]);
            
            someStoringHandleObject.exposureTimeField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'exposureTimeField',...
                'String', num2str(cameraSignalConversion.exposureTime_ms/1000),...
                'Tooltip', 'Exposure time of the camera.',...
                'Units', 'normal',...
                'Position', [0.35 0.85 0.15 0.1]);
            
            %% Frame time
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Frame time [s]:',...
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.75 0.3 0.1]);
            
            someStoringHandleObject.frameTimeField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'frameTimeField',...
                'String', num2str(cameraSignalConversion.frameInterval_ms/1000),...
                'Tooltip', 'Frame (acquisition) time of the camera.',...
                'Units', 'normal',...
                'Position', [0.35 0.75 0.15 0.1]);
            
            %% Photon per count conversion factor
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Photons/Count:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.65 0.3 0.1]);
            
            someStoringHandleObject.photonPerCountField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'photonPerCountField',...
                'String', num2str(1/cameraSignalConversion.countsPerPhoton),....
                'Tooltip', 'Photons per count conversion factor of imaging system.',...
                'Units', 'normal',...
                'Position', [0.35 0.65 0.15 0.1]);
            
            
            %% Number of bins for the histograms
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Bins min. N:',...
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.525 0.85 0.25 0.1]);
            
            someStoringHandleObject.NBinsField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'NBinsField',...
                'String', num2str([50 5]),...
                'Tooltip', 'Number of bins for the histograms and for the exponential fit in case of image map, respectively.',...
                'Units', 'normal',...
                'Position', [0.80 0.85 0.2 0.1]);
            
            %% Minimal number of localizations in each pixel
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Pix. Val. N min.:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.525 0.75 0.3 0.1]);
            
            someStoringHandleObject.thresPixValNField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'thresPixValNField',...
                'String', num2str(1),...
                'Tooltip', 'Minimal number of localizations in each pixel (in case of image map)',...
                'Units', 'normal',...
                'Position', [0.85 0.75 0.15 0.1]);
            
            %% Pixel size
            uicontrol('Style','text','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'String', 'Pixel size [nm]:',....
                'HorizontalAlignment', 'left',...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.525 0.65 0.25 0.1]);
            
            someStoringHandleObject.mapPixelSizeField=uicontrol('Style','edit','Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Tag', 'mapPixelSizeField',...
                'String', num2str([480 480]),...
                'Tooltip', 'Pixel size for the image map.',...
                'Units', 'normal',...
                'Position', [0.80 0.65 0.2 0.1]);
            
            %% Button group for choosing the map type
            someStoringHandleObject.reconTrajPlotButtonGroup = uibuttongroup('Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Visible','on',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.55 0.4 0.4 0.2]...
                );
            
            % Create three radio buttons in the button group.
            someStoringHandleObject.scatterMapRadioButton = uicontrol(someStoringHandleObject.reconTrajPlotButtonGroup,...
                'Style','radiobutton',...
                'String','Scatter map',...
                'Tooltip', 'Create scatter map of trajectories with color code.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.5 0.9 0.4],...
                'Tag', 'trajScatterMapRadioButton',...
                'HandleVisibility','off');
            
            someStoringHandleObject.imageMapRadioButton = uicontrol(someStoringHandleObject.reconTrajPlotButtonGroup,...
                'Style','radiobutton',...
                'String','Image map',...
                'Tooltip', 'Create pixelized image map of trajectories with color code.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.0 0.9 0.4],...
                'Tag', 'trajImageMapRadioButton',...
                'HandleVisibility','off');
            
            %% buton group for choosing the method about how the calculation of the expectation/mean values of different quantites should be calculaed
            someStoringHandleObject.reconTrajExpectationValueButtonGroup = uibuttongroup('Parent',someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Visible','On',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.55 0.01 0.40 0.39]...
                );
            
            someStoringHandleObject.trajExpFitRadioButton=uicontrol(someStoringHandleObject.reconTrajExpectationValueButtonGroup,...
                'Style','radiobutton',...
                'String','Exponential fit',...
                'Tooltip', 'Visualize the fitted exponential decay rate on the image map.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.75 0.9 0.25],...
                'Tag', 'trajExpFitRadioButton',...
                'HandleVisibility','off');
            
            someStoringHandleObject.trajFitResRadioButton=uicontrol(someStoringHandleObject.reconTrajExpectationValueButtonGroup,...
                'Style','radiobutton',...
                'String','Residuum of fit',...
                'Tooltip', 'Visualize the residuum of the fitted exponential on the image map.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.5 0.9 0.25],...
                'Tag', 'trajFitResRadioButton',...
                'HandleVisibility','off');
            
            someStoringHandleObject.trajMeanForDecayRadioButton=uicontrol(someStoringHandleObject.reconTrajExpectationValueButtonGroup,...
                'Style','radiobutton',...
                'String','Mean for decay',...
                'Tooltip', 'Visualize the mean value minus the lower filter bound of the choosen quantity on the image map.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.25 0.9 0.25],...
                'Tag', 'trajMeanForDecayRadioButton',...
                'HandleVisibility','off');
            
            someStoringHandleObject.trajMeanOfValRadioButton=uicontrol(someStoringHandleObject.reconTrajExpectationValueButtonGroup,...
                'Style','radiobutton',...
                'String','Mean of values',...
                'Tooltip', 'Visualize the mean value of the choosen quantity on the image map.',...
                'Units', 'Normalized',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position',[0.05 0.00 0.9 0.25],...
                'Tag', 'trajMeanOfValRadioButton',...
                'HandleVisibility','off');
            
            %% Checkbox for setting whether the comlex trajectories should be included
            
            someStoringHandleObject.includeComplexTrajsCheckbox=uicontrol(someStoringHandleObject.uipanel_trajGraphSettingsPanel,...
                'Style','check',...
                'String', 'Include complex trajs',...
                'Tooltip', 'Include the complex trajectories beside the simple ones when creating histograms and scatter or image maps.',...
                'Value',1,...
                'Units', 'normal',...
                'BackgroundColor',[0.8 0.8 0.8],...
                'Position', [0.025 0.02 0.475 0.1],...
                'Tag', 'complexTrajsCheckBox');
            
            uiwait(windowObject);
            
        end
        
        function reconTrajReconSettingsRadioButton_Callback(someStoringHandleObject)
            
            set(someStoringHandleObject.uipanel_trajReconSettings, 'Visible', 'On')
            set(someStoringHandleObject.uipanel_trajGraphSettingsPanel, 'Visible', 'Off')
            set(someStoringHandleObject.uipanel_trajGraphPlotPanel, 'Visible', 'Off')
            
        end
        
        
        function reconTrajGraphSettingsRadioButton_Callback(someStoringHandleObject)
            
            set(someStoringHandleObject.uipanel_trajReconSettings, 'Visible', 'Off')
            set(someStoringHandleObject.uipanel_trajGraphSettingsPanel, 'Visible', 'On')
            set(someStoringHandleObject.uipanel_trajGraphPlotPanel, 'Visible', 'Off')
            
        end
        
        
        function reconTrajGraphPlotRadioButton_Callback(someStoringHandleObject)
            
            set(someStoringHandleObject.uipanel_trajReconSettings, 'Visible', 'Off')
            set(someStoringHandleObject.uipanel_trajGraphSettingsPanel, 'Visible', 'Off')
            set(someStoringHandleObject.uipanel_trajGraphPlotPanel, 'Visible', 'On')
            
        end
        
        
        % Hiding/making visible the image map properties
        function trajScatterMapRadioButton_Callback(someStoringHandleObject)
            
            set(someStoringHandleObject.trajExpFitRadioButton, 'Enable', 'Off')
            set(someStoringHandleObject.trajFitResRadioButton, 'Enable', 'Off')
            set(someStoringHandleObject.trajMeanForDecayRadioButton, 'Enable', 'Off')
            set(someStoringHandleObject.trajMeanOfValRadioButton, 'Enable', 'Off')
        end
        
        function trajImageMapRadioButton_Callback(someStoringHandleObject)
            
            set(someStoringHandleObject.trajExpFitRadioButton, 'Enable', 'On')
            set(someStoringHandleObject.trajFitResRadioButton, 'Enable', 'On')
            set(someStoringHandleObject.trajMeanForDecayRadioButton, 'Enable', 'On')
            set(someStoringHandleObject.trajMeanOfValRadioButton, 'Enable', 'On')
        end
        
    end
    
end

function [calcParamsStruct, concatComplexTrajsBool] = getAnalysisSettings(someStoringHandleObject)

    %% Parameters required for calculating and filtering the quantites belonging to the trajectories
    % camera parameters
    calcParamsStruct.exposureTime=str2num(get(someStoringHandleObject.exposureTimeField,'String'));
    calcParamsStruct.frameTime=str2num(get(someStoringHandleObject.frameTimeField,'String'));
    calcParamsStruct.photonPerCount=str2num(get(someStoringHandleObject.photonPerCountField,'String'));
    % filtering bounds
    calcParamsStruct.NPhotonBounds=str2num(get(someStoringHandleObject.filterNPhotonBoundsField,'String'));
    calcParamsStruct.frameDurationBounds=str2num(get(someStoringHandleObject.filterFrameDurationBoundsField,'String'));
    calcParamsStruct.emPowerBounds=str2num(get(someStoringHandleObject.filterEmPowerBoundsField,'String'));
    calcParamsStruct.lifeTimeBounds=str2num(get(someStoringHandleObject.filterLifeTimeBoundsField,'String'));
    % histogram parameters
    NBins=str2num(get(someStoringHandleObject.NBinsField,'String'));
    calcParamsStruct.NBins=NBins(1);

    % getting the checkbox value whether the complex trajectories should be
    % included
    if get(someStoringHandleObject.includeComplexTrajsCheckbox, 'Enable')
        concatComplexTrajsBool=true;
    else
        concatComplexTrajsBool=false;
    end

end


% Plotting the chosen histogram
function traj_plotHist(someStoringHandleObject, histogramType)


    %% selecting the histogram type and whether the tracker should be used based on which button has been pushed
    switch histogramType
        case 'traj_photoPhysicsHist'
            histType='photophysics';
            text_figure_name='Photophysics histogram';
            quantityType={'photon number', 'emission power', 'frame duration', 'lifetime' };
        case 'traj_photoPhysicsHistTracker'
            histType='photophysics';
            text_figure_name='Photophysics histogram tracker';
            quantityType={'photon number', 'emission power', 'frame duration', 'lifetime' };
        case 'traj_qualityHist'
            histType='quality';
            text_figure_name='Quality histogram';
            quantityType={'PSF residuum', 'PSF width X', 'PSF width Y' };
        case 'traj_qualityHistTracker'
            histType='quality';
            text_figure_name='Quality histogram tracker';
            quantityType={'PSF residuum', 'PSF width X', 'PSF width Y' };
    end

    [calcParamsStruct, concatComplexTrajsBool] = getAnalysisSettings(someStoringHandleObject);

    %% Calculating and filtering the quantites belonging to the trajectories
    switch histogramType
        % create a set of histagrams from the whole image
        case {'traj_photoPhysicsHist', 'traj_qualityHist'}
            mapBool=false;
            frameIdxBool=false;
            [trajQuantitesAccepted, trajQuantitesRejected]=rainSTORM_trajectoryFitting.filter(someStoringHandleObject.simpleTrajectories, someStoringHandleObject.complexTrajectories, concatComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool);
            
        % create a set of histagrams from only a rectangular cut-out part of
        % the image
        case {'traj_photoPhysicsHistTracker', 'traj_qualityHistTracker'}

            analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
            dataType = analysisParameters.dataType;
            operations.(dataType).processing.('visualization') = someStoringHandleObject.visualizationOperationSettings;
            [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters, someStoringHandleObject.trajectoryImageMatrix, someStoringHandleObject.pixelCenters);
            figure(localizationFigure)


            rect = getrect(localizationFigure);

            % "x" and "y" bounds of the selected region
            yBounds = [rect(1), rect(1) + rect(3) ];
            xBounds = [rect(2), rect(2) + rect(4) ];

            mapBool=false;
            frameIdxBool=false;
            [trajQuantitesAccepted, trajQuantitesRejected]=rainSTORM_trajectoryFitting.filter(someStoringHandleObject.simpleTrajectories, someStoringHandleObject.complexTrajectories, concatComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool, xBounds, yBounds);
    end


    %% Calling the functions to draw the selected set of histograms
    switch histogramType
        % create a set of histagrams containing photophysics information
        case {'traj_photoPhysicsHist', 'traj_photoPhysicsHistTracker'}
            figHistsHandle = rainSTORM_traj_photophysicsHist( trajQuantitesAccepted, trajQuantitesRejected, text_figure_name, calcParamsStruct );

            % create a set of histagrams containing localization quality
        case {'traj_qualityHist', 'traj_qualityHistTracker'}
            figHistsHandle = rainSTORM_traj_qualityHist( trajQuantitesAccepted, trajQuantitesRejected, text_figure_name, calcParamsStruct );
    end
    figure(figHistsHandle)
end


%% Plotting the map of the chosen quantity
function traj_plotMap(someStoringHandleObject, mapType)


    %% selecting the histogram type and whether the tracker should be used based on which button has been pushed
    switch mapType
        case 'traj_NPhotonPlot'
            quantityType='photon number';
            text_figure_name='Photon number map';

        case 'traj_emPowerPlot'
            quantityType='emission power';
            text_figure_name='Emission power map';

        case 'traj_frameDurationPlot'
            quantityType='frame duration';
            text_figure_name='Frame duration map';

        case 'traj_lifeTimePlot'
            quantityType='lifetime';
            text_figure_name='ON lifetime map';
    end

    [calcParamsStruct, catComplexTrajsBool] = getAnalysisSettings(someStoringHandleObject);
    
    %% Calculating and filtering the quantites belonging to the trajectories
    mapBool=true;
    frameIdxBool=false;
    [trajQuantitesAccepted, ~]=rainSTORM_trajectoryFitting.filter(someStoringHandleObject.simpleTrajectories, someStoringHandleObject.complexTrajectories, catComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool);

    %% Parameters required for visualizing the results
    analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
    cameraSignalConversion = analysisParameters.sharedParameters.cameraSignalConversion;
    % number of camera pixels on the frames
    plotParamsStruct.sizeOfCCDFrame=analysisParameters.sharedParameters.stackSize;
    % size of camera pixels in nm
    plotParamsStruct.camPixelWidth=cameraSignalConversion.pixelSize_nm;
    %
    plotParamsStruct.quantityType=quantityType;
    % title of the map
    plotParamsStruct.figureName=text_figure_name;
    % examining whether scatter or image map was selected
    selectedTrajPlotRadioButton=get(get(someStoringHandleObject.reconTrajPlotButtonGroup,'SelectedObject'), 'Tag');
    switch selectedTrajPlotRadioButton
        case 'trajScatterMapRadioButton'
            % map type
            plotParamsStruct.figureType='scatter';
        case 'trajImageMapRadioButton'
            % map type
            plotParamsStruct.figureType='image';
            % size of pixels in "x" and "y" directions for the image map
            plotParamsStruct.mapPixelSize=str2num(get(someStoringHandleObject.mapPixelSizeField,'String'));
            % minimum number of trajectories in an image map pixel
            plotParamsStruct.thresPixValN=str2num(get(someStoringHandleObject.thresPixValNField,'String'));

            % selecting what kind of eopration shoud be performed on the
            % quantites of the trajectories in the pixels, what the pixels
            % should depict
            selectedExpectationValueTypeRadioButton=get(get(someStoringHandleObject.reconTrajExpectationValueButtonGroup,'SelectedObject'), 'Tag');
            switch selectedExpectationValueTypeRadioButton
                % the decay rate of the fitted exponentials
                case 'trajExpFitRadioButton'
                    plotParamsStruct.mapType='fitted exponential decay';

                    % the residuum rate of the fitted exponentials
                case 'trajFitResRadioButton'
                    plotParamsStruct.mapType='fitted exponential residuum';

                    % the mean value by supposing exponential distribution
                case 'trajMeanForDecayRadioButton'
                    plotParamsStruct.mapType='mean for decay';

                    % the mean value of the quantites
                case 'trajMeanOfValRadioButton'
                    plotParamsStruct.mapType='mean of values';
            end
    end

    %% Preparing the chosen quantity for plotting
    switch quantityType
        case 'photon number'
            isNotNaNQuantVect=~isnan(trajQuantitesAccepted.NPhoton);
            quantity=trajQuantitesAccepted.NPhoton(isNotNaNQuantVect);

        case 'emission power'
            isNotNaNQuantVect=~isnan(trajQuantitesAccepted.emPower);
            quantity=trajQuantitesAccepted.emPower(isNotNaNQuantVect);

        case 'frame duration'
            isNotNaNQuantVect=~isnan(trajQuantitesAccepted.frame_N);
            quantity=trajQuantitesAccepted.frame_N(isNotNaNQuantVect);

        case 'lifetime'
            isNotNaNQuantVect=~isnan(trajQuantitesAccepted.lifeTime);
            quantity=trajQuantitesAccepted.lifeTime(isNotNaNQuantVect);
    end
    posits=[trajQuantitesAccepted.x_coord(isNotNaNQuantVect,:), trajQuantitesAccepted.y_coord(isNotNaNQuantVect,:)];

    %% Calling the function for creating the scatter or image map for the chosen quantity
    quantity = double(quantity);
    mapFigureHAndle = rainSTORM_traj_plotQuantity(plotParamsStruct, posits, quantity, calcParamsStruct);
    figure(mapFigureHAndle);

end
        

%% Calculating/recalculating trajectory fitting
function traj_recalc(someStoringHandleObject)

    % update the settings from the GUI
    settings = someStoringHandleObject.concatenationSettings.settings;
    settings.distThreshold_nm=str2num(get(someStoringHandleObject.distThresholdField,'String'));
    settings.ZPrecConst=str2num(get(someStoringHandleObject.ZPrecConstField,'String'));
    settings.maxNumberOfNoNewLocsInARow=str2num(get(someStoringHandleObject.maxNumberOfNoNewLocsInARowField,'String'));
    
    localizationData = someStoringHandleObject.localizationDataObject.accepted;

    analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
    cameraPixelSize_nm = analysisParameters.sharedParameters.cameraSignalConversion.pixelSize_nm;
    
    settings.cameraPixelSize_nm = cameraPixelSize_nm;
    method = 'branching trajectories';
    [simpleTrajs, complexTrajs, trajsInfo, ~, ~] = rainSTORM_trajectoryFitting.perform(localizationData, method, settings);

    someStoringHandleObject.simpleTrajectories = simpleTrajs;
    someStoringHandleObject.complexTrajectories = complexTrajs;
    someStoringHandleObject.trajectoryRawData = trajsInfo;

    % create a super resolced image from the concatenated coordiantes
    dataType = analysisParameters.dataType;
    operations.(dataType).processing.('pixelization') = someStoringHandleObject.pixelizationOperationSettings;
    
    pixelSize_nm = analysisParameters.sharedParameters.cameraSignalConversion.pixelSize_nm;
    [analysisParameters, trajectoryImageMatrix, pixelCenters] = runOperation.processing(operations, 'pixelization', analysisParameters, simpleTrajs);
    someStoringHandleObject.trajectoryImageMatrix = trajectoryImageMatrix;
    someStoringHandleObject.pixelCenters = pixelCenters;
    
    operations.(dataType).processing.('visualization') = someStoringHandleObject.visualizationOperationSettings;
    [analysisParameters, localizationFigure, colormapVector] = runOperation.processing(operations, 'visualization', analysisParameters, trajectoryImageMatrix, pixelCenters);
    figure(localizationFigure)
    
    set(someStoringHandleObject.traj_qualityHistButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_qualityHistTrackerButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_photoPhysicsHistButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_photoPhysicsHistTrackerButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_NPhotonPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_frameDurationPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_emPowerPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_lifeTimePlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_recalcButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_exportButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_exportFilteredToCSVButton, 'Enable', 'on');
    
end

% Export the newly calculated trajectories
function traj_export(someStoringHandleObject)

    analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
    dataName = analysisParameters.sharedParameters.originalDataName;
    dataPath = analysisParameters.sharedParameters.originalDataPath;

    defaultName=fullfile(dataPath, [dataName, '_trajectory.mat']);

    [trajExportFileName, trajExportPath] = uiputfile('*.mat','Save Trajectory Data As', defaultName);
    trajExportName=fullfile(trajExportPath, trajExportFileName);

    % getting the determined trajecties
    trajFilteredOutputData.simpleTrajectories=someStoringHandleObject.simpleTrajectories;
    trajFilteredOutputData.complexTrajectories=someStoringHandleObject.complexTrajectories;

    if isempty(trajFilteredOutputData.simpleTrajectories) && isempty(trajFilteredOutputData.complexTrajectories)
        disp('Empty trajectory data, not exporting. Please run the trajectory fitting.')
    else
        %save(trajExportName, 'trajFilteredOutputData');
        save(trajExportName, 'trajFilteredOutputData', '-v7.3');
    end

end

% Eport the filtered trajectory data to *.csv
function traj_exportFilteredToCSV(someStoringHandleObject)

    analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
    dataName = analysisParameters.sharedParameters.originalDataName;
    dataPath = analysisParameters.sharedParameters.originalDataPath;

    defaultName=fullfile(dataPath, [dataName, '_filteredTraj.csv']);

    [trajExportFileName, trajExportPath] = uiputfile('*.csv','Save Filtered Trajectory Data As', defaultName);
    
    quantityType={'photon number', 'emission power', 'frame duration', 'lifetime', 'PSF residuum', 'PSF width X', 'PSF width Y' };
    mapBool=true;
    frameIdxBool=true;

    [calcParamsStruct, concatComplexTrajsBool] = getAnalysisSettings(someStoringHandleObject);
    
    [trajQuantitesAccepted, trajQuantitesRejected]=rainSTORM_trajectoryFitting.filter(someStoringHandleObject.simpleTrajectories, someStoringHandleObject.complexTrajectories, concatComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool);

    pointillisticData_save.tabularData(trajExportPath, trajExportFileName, trajQuantitesAccepted);

end


% Import the previouly calculated trajectories
function traj_import(someStoringHandleObject)

    analysisParameters = someStoringHandleObject.localizationDataObject.analysisParameters;
    dataPath = analysisParameters.sharedParameters.originalDataPath;

    [trajImportFileName, trajImportPath] = uigetfile('*.mat','Open Trajectory Data', dataPath);
    trajImportName=fullfile(trajImportPath, trajImportFileName);

    % getting the trajecties to be imported
    importData=load(trajImportName,'-mat');

    someStoringHandleObject.simpleTrajectories = importData.trajFilteredOutputData.simpleTrajectories;
    someStoringHandleObject.complexTrajectories = importData.trajFilteredOutputData.complexTrajectories;
    
    
    set(someStoringHandleObject.traj_qualityHistButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_qualityHistTrackerButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_photoPhysicsHistButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_photoPhysicsHistTrackerButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_NPhotonPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_frameDurationPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_emPowerPlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_lifeTimePlotButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_recalcButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_exportButton, 'Enable', 'on');
    set(someStoringHandleObject.traj_exportFilteredToCSVButton, 'Enable', 'on');
end