function [ localizationData_accepted, localizationData_rejected, metadata, sharedParameters] = rainSTORM_astigmatic3D(localizationData_accepted, localizationData_rejected, calibrationData, astigmaticMethod, astigmaticSettings )
% This function calculates the blinking event's "z" positions based on
% their PSF widths and the calibration data. The first part prepares the
% calculation, defines the necessary variables. The second part calls the
% desired atigmatic 3D method, preapres the calibration data and "fits" the
% blinkings on it to fing the "z" positions. In the next step localizations
% can be filtered by their I parameters or by astigmatic 3D method specific
% values. The next part visualized the reconstructed positions in 3D plots,
% and finally writes out the filtered, "z" position complemented
% localization data.

if ~pointillisticData_fieldNameManagement.isMemberField(localizationData_accepted, {'sig_x', 'sig_y'})
    error('The data given for astigmatic 3D calculation does not contain the neccessary "sig_x" and "sig_y" fields, probably not an astigmatic data.') 
end

cameraSignalConversion = astigmaticSettings.cameraSignalConversion;

zBounds_nm = unitConversion.convertScalarQuantity(astigmaticSettings.zBounds, 'nm', cameraSignalConversion);
zBounds_nm = zBounds_nm.value;

% cuttig the calibration data that falls within the given z bounds, plus 1 element to the left and right
filterBooleanVect_original = (calibrationData.z_nm>=zBounds_nm(1)) & (calibrationData.z_nm<=zBounds_nm(2));
filterBooleanVect_left = false(size(filterBooleanVect_original));
filterBooleanVect_left(1:end-1) = filterBooleanVect_original(2:end);
filterBooleanVect_right = false(size(filterBooleanVect_original));
filterBooleanVect_right(2:end) = filterBooleanVect_original(1:end-1);
filterBooleanVect = filterBooleanVect_original | filterBooleanVect_left | filterBooleanVect_right;


zPos_calib=unitConversion.convertValue(calibrationData.z_nm(filterBooleanVect), 'nm', 'camera pixel length', cameraSignalConversion);
sigX_calib=unitConversion.convertValue(calibrationData.sig_x_mean_nm(filterBooleanVect), 'nm', 'camera pixel length', cameraSignalConversion);
sigY_calib=unitConversion.convertValue(calibrationData.sig_y_mean_nm(filterBooleanVect), 'nm', 'camera pixel length', cameraSignalConversion);

%% applying the chosen axial position calculation method

% get the fitted widths of the astigmatic measurement
sigmaX_accepted = localizationData_accepted.sig_x;   % fitted widths in the "x" direction
sigmaY_accepted = localizationData_accepted.sig_y;   % fitted widths in the "y" direction
z_coord_accepted = [];
if ~isempty(sigmaX_accepted)
    switch astigmaticMethod
        case 'sigma squared error'
          z_coord_accepted = sigma_squared_error(sigmaX_accepted, sigmaY_accepted, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        case 'fit on ellipticity'
          z_coord_accepted = fit_on_ellipticity(sigmaX_accepted, sigmaY_accepted, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        case 'fit on sigmas'
          z_coord_accepted = fit_on_sigmas(sigmaX_accepted, sigmaY_accepted, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        otherwise

    end
end
localizationData_accepted = pointillisticData_fieldManagement.add(localizationData_accepted, 'z_coord', z_coord_accepted);

sigmaX_rejected = localizationData_rejected.sig_x;   % fitted widths in the "x" direction
sigmaY_rejected = localizationData_rejected.sig_y;   % fitted widths in the "y" direction
z_coord_rejected = [];
if ~isempty(sigmaX_rejected)
    switch astigmaticMethod
        case 'sigma squared error'
          z_coord_rejected = sigma_squared_error(sigmaX_rejected, sigmaY_rejected, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        case 'fit on ellipticity'
          z_coord_rejected = fit_on_ellipticity(sigmaX_rejected, sigmaY_rejected, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        case 'fit on sigmas'
          z_coord_rejected = fit_on_sigmas(sigmaX_rejected, sigmaY_rejected, zPos_calib, sigX_calib, sigY_calib, astigmaticSettings);
        otherwise

    end
end
localizationData_rejected = pointillisticData_fieldManagement.add(localizationData_rejected, 'z_coord', z_coord_rejected);

metadata.zBounds_nm = zBounds_nm;
sharedParameters = struct();


end


function zPos = sigma_squared_error(sigmaX, sigmaY, zPos_calib, sigX_calib, sigY_calib, settings)
    % This method calculates the axial positions of the blinking events based
    % on their widths in the "x" and "y" directions. Finds the "z" positions of the
    % calibration data where the squared error of the measured and calibrated sigma
    % values are minimal, then fits a parabola on this and its neighbouring
    % values and calculates the minimum position of the fitted
    % parabola. The calibration file should be a matrix containing the
    % PSF widths in "x" and "y" directions within a selected "z" range.

    % disard the calibration data where it is nan, i.e. there might have been no
    % localizations is some frames
    nan_sigX = isnan(sigX_calib);
    nan_sigY = isnan(sigY_calib);
    calib_nan = nan_sigX | nan_sigY;
    zPos_calib = zPos_calib(~calib_nan);
    sigX_calib = sigX_calib(~calib_nan);
    sigY_calib = sigY_calib(~calib_nan);


    zBounds = unitConversion.convertScalarQuantity(settings.zBounds, 'camera pixel length', settings.cameraSignalConversion);
    zBounds = zBounds.value;
    
    resamplingNumber = settings.resamplingNumber;

    % fitting a polynomials on the fitted width data
    fitPoly_sigX = polyfit(zPos_calib, sigX_calib, settings.polynomialOrder);     % coefficients of the fitted polynomials in decreasing power order
    fitPoly_sigY = polyfit(zPos_calib, sigY_calib, settings.polynomialOrder);     % coefficients of the fitted polynomials in decreasing power order

    % preparing the resampling
    zMin=min(zBounds);            % "lower" "z" boundary
    zMax=max(zBounds);            % "upper" "z" boundary
    delta_z=(zMax-zMin)/(resamplingNumber-1);   % "z" position change of the resampling
    zCal=(zMin:delta_z:zMax);   % "z" positions after the resampling

    % resampling the calibration data with the fitted polynomial
    sigCalXVect=polyval(fitPoly_sigX, zCal);    % resampling the sigma values in "x" direction
    sigCalYVect=polyval(fitPoly_sigY, zCal);    % resampling the sigma values in "y" direction


    sigLocXMat=repmat(sigmaX, 1, resamplingNumber);     % resampling the sigma values of localized blinking events in "x" direction
    sigLocYMat=repmat(sigmaY, 1, resamplingNumber);     % resampling the sigma values of localized blinking events in "y" direction

    % calculating the sigma values and the minimal measured "z" position for
    % the fitting
    sigVar=(sigCalXVect-sigLocXMat).^2+(sigCalYVect-sigLocYMat).^2;    % calculating the variance of all measured sigma value compared to all sigma values of the calibration
    %sigVar=(sigCalXVect./sigCalYVect-sigLocXMat./sigLocYMat).^2;    % calculating the variance of all measured sigma value compared to all sigma values of the calibration

    [sigVar_0, idx_0_orig]=min(sigVar, [], 2);          % finding where the variances of sigma values are minimal

    idx_0=idx_0_orig;                   % original indices where the sigma variance is minimal
    idx_0(idx_0_orig==1)=2;             % if the sigma variance is minimal at the first element of the calibration file
    idx_0(idx_0_orig==resamplingNumber)=resamplingNumber-1;     % if the sigma variance is minimal at the last element of the calibration file

    localizationNumber = numel(sigmaX);
    z_0=zeros(localizationNumber,1);                  % "z" positions where the measured sigma values have minimal variance
    sigVar_left=zeros(localizationNumber,1);          % sigma variance value left to mentioned "z" position
    sigVar_right=zeros(localizationNumber,1);         % sigma variance value right to this mentioned "z" position

    for idxLoc=1:localizationNumber
        % assigning the zero position and the neighbouring sigma values
        z_0(idxLoc,1)=zCal(idx_0(idxLoc,1));
        %sigVar_0(idxLoc,1)=sigVar(idxLoc, idx_0(idxLoc));
        sigVar_left(idxLoc,1)=sigVar(idxLoc, idx_0(idxLoc)-1);
        sigVar_right(idxLoc,1)=sigVar(idxLoc, idx_0(idxLoc)+1);
    end

    % fitting the parabola
    zPos=z_0-(sigVar_right-sigVar_left)./(sigVar_right-2*sigVar_0+sigVar_left)*delta_z*0.5;   % minimal position of the fitted parabola
    %zPos=z_0;
    zPos(idx_0_orig==1)=NaN;        % if the sigma variance is minimal at the first element of the calibration file, it should be invalid
    zPos(idx_0_orig==resamplingNumber)=NaN;     % if the sigma variance is minimal at the last element of the calibration file, it should be invalid
    sigVar_zPos=sigVar_0-(zPos-z_0).^2.*(sigVar_right-2*sigVar_0+sigVar_left)/(2*delta_z^2);        % minimal value of the fitted parabola
    %sigVar_zPos=(zPos-z_0).^2.*(sigVar_right-2*sigVar_0+sigVar_left)/(2*delta_z^2)+(zPos-z_0).*(sigVar_right-sigVar_left)/(2*delta_z)+sigVar_0; % old, wrong formula        %
    %sigVar_zPos=sigVar_0;
end


function zPos = fit_on_ellipticity(sigmaX, sigmaY, zPos_calib, sigX_calib, sigY_calib, settings)

    zBounds = unitConversion.convertScalarQuantity(settings.zBounds, 'camera pixel length', settings.cameraSignalConversion);
    zBounds = zBounds.value;
    
    % disard the calibration data where it is nan, i.e. there might have been no
    % localizations is some frames
    nan_sigX = isnan(sigX_calib);
    nan_sigY = isnan(sigY_calib);
    calib_nan = nan_sigX | nan_sigY;
    zPos_calib = zPos_calib(~calib_nan);
    sigX_calib = sigX_calib(~calib_nan);
    sigY_calib = sigY_calib(~calib_nan);
    
    % calculating the ellipticities
    ellipticity_calib=ellipticity_formulas(settings.formula, sigX_calib, sigY_calib);    % vector containing the ellipticities belonging to each frame

    % fitting a polynomial on the ellipticity vector data
    polynomialCoefficients=polyfit(zPos_calib, ellipticity_calib, settings.polynomialOrder);     % coefficients of the fitted polynomials in decreasing power order

    % vector required for substracting the actual localized sigma from
    % the zeroth order term for finding the roots
    zerothCoeffVect=zeros(1,settings.polynomialOrder+1);
    zerothCoeffVect(end)=1;

    % coefficont of the different terms of the polynomial of the fitted
    % calibration curve
    zMin=min(zBounds);
    zMax=max(zBounds);

    % selection the definition of the ellipticity
    ellipticity=ellipticity_formulas(settings.formula, sigmaX, sigmaY );

    localizationNumber = numel(sigmaX);
    zPos = zeros(localizationNumber, 1);
    % going through the localizations
    injectiveBoolean = true;
    for idxLoc=1:localizationNumber

        % all roots of the "z" position
        rootsZ = roots(polynomialCoefficients-zerothCoeffVect*ellipticity(idxLoc));

        % invalid roots should be deleted
        rootsZ(imag(rootsZ)~=0)=[];     % if the root is complex
        rootsZ(rootsZ>zMax | rootsZ<zMin)=[];       % root is out of the chosen region of the calibration

        % assigning the actual axial position
        if length(rootsZ)>1     % if more than one valid roots remained
            injectiveBoolean = injectiveBoolean & false;
          zPos(idxLoc)=rootsZ(1);   % let it be
        elseif isempty(rootsZ)  % if no valid roots remained
          zPos(idxLoc)=NaN;   % it should be NaN
        else                    % if only one valid root remained
          zPos(idxLoc)=rootsZ;    % a valid axial position
        end
    end
    if ~ injectiveBoolean
        warning('The calibration curve is not injective in the selected region. Some axial coordinate results are ambiguous. Set proper upper and lower Z bounds.');
    end
end


function zPos = fit_on_sigmas(sigmaX, sigmaY, zPos_calib, sigX_calib, sigY_calib, settings)

    zBounds = unitConversion.convertScalarQuantity(settings.zBounds, 'camera pixel length', settings.cameraSignalConversion);
    zBounds = zBounds.value;
    
    % remove every data point that is NaN
    NaN_dataIndices = isnan(sigX_calib) | isnan(sigY_calib);
    sigX_calib = sigX_calib(~NaN_dataIndices);
    sigY_calib = sigY_calib(~NaN_dataIndices);
    zPos_calib = zPos_calib(~NaN_dataIndices);
    
    % fitting a polynomial on the ellipticity vector data
    polynomialCoefficients_x = polyfit(zPos_calib, sigX_calib, settings.polynomialOrder);     % coefficients of the fitted polynomials in decreasing power order
    polynomialCoefficients_y = polyfit(zPos_calib, sigY_calib, settings.polynomialOrder);     % coefficients of the fitted polynomials in decreasing power order

    % vector required for subtracting the actual localized sigma from
    % the zeroth order term for finding the roots
    zerothCoeffVect=zeros(1, settings.polynomialOrder+1);
    zerothCoeffVect(end)=1;

    % coefficont of the different terms of the polynomial of the fitted
    % calibration curve
    zMin=min(zBounds);
    zMax=max(zBounds);

    localizationNumber = numel(sigmaX);
    dist=zeros(localizationNumber,1); % distance of the "z" positions fitted on the sigma values in the "x" and "y" directions

    zPos = zeros(localizationNumber, 1);
    % going through the localizations
    for idxLoc=1:localizationNumber
        % all roots of the "z" position
        rootsZ.X = roots(polynomialCoefficients_x-zerothCoeffVect*sigmaX(idxLoc));
        rootsZ.Y = roots(polynomialCoefficients_y-zerothCoeffVect*sigmaY(idxLoc));

        % invalid roots should be changed to NaN
        rootsZ.X (imag(rootsZ.X )~=0)=NaN;      % if the root is complex
        rootsZ.X (rootsZ.X >zMax | rootsZ.X <zMin)=NaN;     % root is out of the chosen region of the calibration

        rootsZ.Y(imag(rootsZ.Y)~=0)=NaN;       % if the root is complex
        rootsZ.Y(rootsZ.Y>zMax | rootsZ.Y<zMin)=NaN;     % root is out of the chosen region of the calibration

        distRoots=abs(rootsZ.X - transpose(rootsZ.Y));   % get all the pairwise distance
        [actDist,indRoot]=min(distRoots(:));       % find the minimum distance
        [indX,indY] = ind2sub(settings.polynomialOrder,indRoot); % get the index of the roots belonging to the minimum difference

        zPos(idxLoc)=mean([rootsZ.X(indX),rootsZ.Y(indY)]); % if distance < tolerance value -> accepted z position
        dist(idxLoc)=actDist;       % store the minimal distance of the actual localization's roots
    end
end



function [ ellip ] = ellipticity_formulas( type, sigX, sigY )
% This function contains the different methods of calculating the
% ellipticity. Its inputs are the ellipticity "type" and vectors containing
% PSF width in "x" and "y" directions. The ourpur is the vector containing
% the ellipticity values.

switch type
	case 'ratio'
        % ellipticity based on the width ratio
        ellip=sigX./sigY;
	case 'symmetrical'
        % ellipticity based on the symmetrized width ratio
        ellip=zeros(size(sigX));
        ellip(sigX>sigY)=(sigX(sigX>sigY)./sigY(sigX>sigY)) -1;
        ellip(sigX<=sigY)=-(sigY(sigX<=sigY)./sigX(sigX<=sigY)) +1;
	case 'difference'
        % ellipticity based on the width difference
        ellip=sigX-sigY;
	case 'n.square difference'
        % more complicated definition
        ellip=sigX.^3./sigY-sigY.^3./sigX;
        %ellip=sigX.^(0.10)./sigY-sigY.^(0.10)./sigX;
	case 'n.difference'
        % ellipticity based on the normalized width difference
        ellip=(sigX-sigY)./(sigX+sigY);
end


end

