classdef rainSTORM_astigmatic3D_calibration

methods(Static)

    function calibrationData = calibrate(pointillisticData, frameRange, firstFrameZ_nm, lastFrameZ_nm, pixelSize_nm)

      if ~pointillisticData_fieldNameManagement.isMemberField(pointillisticData, {'sig_x', 'sig_y'})
         error('The data given for astigmatic 3D calibration does not contain the neccessary "sig_x" and "sig_y" fields, probably not an astigmatic data.') 
      end
        
      frameIdx_min = min(frameRange);
      frameIdx_max = max(frameRange);
      frameIndices = frameIdx_min:1:frameIdx_max;
      frameN = frameIdx_max - frameIdx_min + 1;

      frameZ = linspace(firstFrameZ_nm, lastFrameZ_nm, frameN);

      frameIdx=pointillisticData.frame_idx;      % frame indices belonging to each localization
      sigX=pointillisticData.sig_x;          % "x" widths belonging to each localization
      sigY=pointillisticData.sig_y;       	% "y" widths belonging to each localization

      fieldNames = {'z_nm', 'loc_N', 'sig_x_mean_nm', 'sig_y_mean_nm', 'sig_x_std_nm', 'sig_y_std_nm'};
      calibrationData = array2table(zeros(frameN,numel(fieldNames)), 'VariableNames',fieldNames);

      % calculating the mean and variance values
      for idx=1:frameN
          actualFrameIndex = frameIndices(idx);
          calibrationData.z_nm(idx,1)=frameZ(idx);
          calibrationData.loc_N(idx,1) = sum(frameIdx==actualFrameIndex);
          calibrationData.sig_x_mean_nm(idx,1) = mean(sigX(frameIdx==actualFrameIndex))*pixelSize_nm;
          calibrationData.sig_y_mean_nm(idx,1) = mean(sigY(frameIdx==actualFrameIndex))*pixelSize_nm;
          calibrationData.sig_x_std_nm(idx,1) = std(sigX(frameIdx==actualFrameIndex))*pixelSize_nm;
          calibrationData.sig_y_std_nm(idx,1) = std(sigY(frameIdx==actualFrameIndex))*pixelSize_nm;
      end

    end



    function [figureObject] = visualize(calibrationData, varargin)


      z_nm = calibrationData.z_nm;
      loc_N = calibrationData.loc_N;
      sig_x_mean_nm = calibrationData.sig_x_mean_nm;
      sig_y_mean_nm = calibrationData.sig_y_mean_nm;
      sig_x_std_nm = calibrationData.sig_x_std_nm;
      sig_y_std_nm  =calibrationData.sig_y_std_nm;

      %%
      % plotting figures to determine the appropriate range for fitting

      % plotting the mean values of the PSF widths belonging to each frame
      figureObject = figure();
      figureObject.Position(3:4) = [840 560];
      subplot(2,1,1);
      errorbar(z_nm, sig_x_mean_nm, sig_x_std_nm, "s", "MarkerSize",10,"MarkerEdgeColor","blue","MarkerFaceColor",[0.65, 0.85, 0.90]);
      %scatter(zPosFrame, sigMeanFrameX,10, 'b','.');
      hold on
      errorbar(z_nm, sig_y_mean_nm, sig_y_std_nm, "s", "MarkerSize",10,"MarkerEdgeColor","red","MarkerFaceColor",[255, 114, 118]/255);
      %scatter(zPosFrame, sigMeanFrameY,10, 'r','.');
      hold off
      xLimits = xlim;
      ticks = xLimits(1):200:xLimits(2);
      set(gca, 'xtick', ticks);
      grid on
      title('Fitted Gaussian widths');
      ylabel('Sigma [px]');
      xlabel('Z position [nm]');

      if numel(varargin) == 1
      xline(varargin{1}(1))
      xline(varargin{1}(2))
      end



      % plotting the number of localizations belonging to each frame
      subplot(2,1,2);
      plot(z_nm, loc_N)
      xLimits = xlim;
      ticks = xLimits(1):200:xLimits(2);
      set(gca, 'xtick', ticks);
      grid on
      title('Localization number');
      ylabel('Number');
      xlabel('Z position [nm]');

      if numel(varargin) == 1
      xline(varargin{1}(1))
      xline(varargin{1}(2))
      end

    end




  end

end
