function [ outputData, nRows, nRows_mod, nCols, nCols_mod ] = rainSTORM_traj_calcImagePlotQuantity( quantity, mapOrHist, plotParamsStruct, calcParamsStruct, varargin )
% 	Summary:	
%	Version:    3.2	
%	Note:		
%	Package:	rainSTORM v3.1+
%	Developer:	novakt
%	Name:		Novák Tibor
%	Company:	University of Szeged
%	Date of the merger: 2017.11.08.

%  This function calculated the pixel values for the image map creation.	

% parents: rainSTORM_traj_plotQuantity, rainSTORM_traj_photophysicsHist
% children: rainSTORM_traj_calcHistData, rainSTORM_traj_exponentialToFit

%% Initialization
singleExpBool=true;

%% Provide the required parameters
% type of the quantity to be plotted
quantityType=plotParamsStruct.quantityType;
% type of image map plot
mapType=plotParamsStruct.mapType;
% threshold for visualization; the number of localizations belonging one pixel of 2D histogram must be equal or greater than this value
thresPixValN=plotParamsStruct.thresPixValN;

% number of bins for the histogram (used for exponential fitting), not used
% for the "frame duration" plots
NBins=calcParamsStruct.NBins;

% determining the histogram bounds for the different type of quantites
switch quantityType
    case 'photon number'
        plotValBoundMin=calcParamsStruct.NPhotonBounds(1);
        if plotValBoundMin==-inf    % if it is the default, was not set manually
            plotValBoundMin=0;      % photon number can not be lower than this
        end
        plotValBoundMax=calcParamsStruct.NPhotonBounds(2);
        
    case 'emission power'
        plotValBoundMin=calcParamsStruct.emPowerBounds(1);
        plotValBoundMax=calcParamsStruct.emPowerBounds(2);
        
    case 'frame duration'
        plotValBoundMin=calcParamsStruct.frameDurationBounds(1);
        if plotValBoundMin==-inf    % if it is the default, was not set manually
            plotValBoundMin=1;      % frame duration can not be lower than this
        end
        plotValBoundMax=calcParamsStruct.frameDurationBounds(2);
        
        singleExpBool=true;
        
    case 'lifetime'
        plotValBoundMin=calcParamsStruct.lifeTimeBounds(1);
        if plotValBoundMin==-Inf    % if it is the default, was not set manually
            plotValBoundMin=2.0*calcParamsStruct.frameTime;     % the ON lifetime should be cut, because only if a trajectory longer than this can be used for ON lifetime calculation for sure
        end
        plotValBoundMax=calcParamsStruct.lifeTimeBounds(2);
        
        singleExpBool=true;
end




switch mapOrHist
    case 'map'
        
        SupResPosits=varargin{1};
        
        % size of pixels in "x" and "y" directions for the image map
        pixelSizeX=plotParamsStruct.mapPixelSize(1);
        pixelSizeY=plotParamsStruct.mapPixelSize(2);

        % number of camera pixels on the frames
        sizeOfCCDFrame=plotParamsStruct.sizeOfCCDFrame;
        % size of camera pixels in nm
        pixelWidth=plotParamsStruct.camPixelWidth;
        
        % calculating number of rows of map pixels
        nRows_int=floor(sizeOfCCDFrame(1)*pixelWidth/pixelSizeX);   % integer part of the ratio of the frame and map pixel size
        nRows_mod=mod(sizeOfCCDFrame(1)*pixelWidth/pixelSizeX, 1);     % fractional part of the ratio of the frame and map pixel size
        nRows=nRows_int+2*ceil(nRows_mod);          % number of rows of map pixels
        
        % calculating number of culomns of map pixels
        nCols_int=floor(sizeOfCCDFrame(2)*pixelWidth/pixelSizeY);   % integer part of the ratio of the frame and map pixel size
        nCols_mod=mod(sizeOfCCDFrame(2)*pixelWidth/pixelSizeY, 1);     % fractional part of the ratio of the frame and map pixel size
        nCols=nCols_int+2*ceil(nCols_mod);          % number of culomns of map pixels
        
        % assign trajectory positions to the map pixels
        SupResImC = cell(nRows,nCols); % each pixel is a cell
        myPosits(:,2) = SupResPosits(:,2)*nCols/sizeOfCCDFrame(2)+(1-nCols_mod); % Scale localizations to super-res grid
        myPosits(:,1) = SupResPosits(:,1)*nRows/sizeOfCCDFrame(1)+(1-nRows_mod); % Scale localizations to super-res grid
        myPosits = ceil(myPosits);      % Digitise localizations to super-res grid
        
        % assign the chosen trajectory quantity to the map pixels
        for lpSRI = 1:size(myPosits,1)
            if ( myPosits(lpSRI,1)<1 || myPosits(lpSRI,2)<1 || ...
                    myPosits(lpSRI,1)>nRows || myPosits(lpSRI,2)>nCols )
                continue; % Skip over positions if they are outside reconstruction
            end
            if quantity(lpSRI)>=plotValBoundMin && quantity(lpSRI)<=plotValBoundMax
                SupResImC(myPosits(lpSRI,1), myPosits(lpSRI,2)) = ...
                    {[SupResImC{myPosits(lpSRI,1), myPosits(lpSRI,2)}, quantity(lpSRI)]};  % attaching the new ones
            end
        end
        
        % number of trajctories on each map pixel, double array
        mapPixN=cellfun(@numel,SupResImC);
        % calculation the double array from which the image is created, mean values
        % of a trajectory quantity on each pixel
        meanOfValMap=cellfun(@(x) mean(x, 'omitnan'), SupResImC);
        meanOfValMap(mapPixN<thresPixValN)=NaN;   % the behaviour of NaN values in "imagesc" function is not defined...

    case 'hist'
        nRows=1;
        nCols=1;
        SupResImC{1, 1}=quantity;
        meanOfValMap=mean(quantity, 'omitnan');
        mapPixN=numel(quantity);
end

% selecting the chosen image map type
switch mapType
    case {'mean of values', 'emission power histogram'}
        % if mean values should be plotted
        switch mapOrHist
            case 'map'
                outputData=meanOfValMap;
            case 'hist'
                outputData.mean=meanOfValMap;
        end
    case {'mean for decay', 'fitted exponential decay', 'fitted exponential residuum', 'exponential histogram'}
        % if the exponential decay rate should be determined (or the residuum of the exponential fit)
        
        % exponential decay calculation does not have much use in case of
        % "emission power" in normal case
        if strcmp(quantityType,'emission power')
            warning('You wanted to evaluate the exponential decay of the goodness of the exponential fit for the emission power. It might not have much meaning.');
        end
        
        % the mean value of the (corrigated) quantity supposedly following
        % exponential distribution (corrigation is needed to get the decay rate)
        if plotValBoundMin==-Inf
            switch quantityType
                case 'frame duration'
                    meanForExpLifeTimeMap=meanOfValMap-0.5;
                otherwise
                    meanForExpLifeTimeMap=meanOfValMap;
            end 
        else
            switch quantityType
                case 'frame duration'
                    meanForExpLifeTimeMap=meanOfValMap-plotValBoundMin+0.5;   % the correction
                otherwise
                    meanForExpLifeTimeMap=meanOfValMap-plotValBoundMin;   % the correction
            end
        end
        
        switch mapType
            case 'mean for decay'
                % get the exponential decay from the corrigated mean value
                outputData=meanForExpLifeTimeMap;
                
            case {'fitted exponential decay', 'fitted exponential residuum', 'exponential histogram'}
                % get the exponential decay from an exponential fit or plot
                % its residuum
                switch mapOrHist
                    case 'map'
                        outputData=zeros(nRows, nCols);
                    case 'hist'
                        outputData.mean=0;
                        outputData.meanForExpLifeTime=0;
                        outputData.expectedLifeTime=0;
                        outputData.fitResiduum=0;
                end
                
                % loop through the pixels of the image map
                for idxCol=1:nCols
                    for idxRow=1:nRows
                        if mapPixN(idxRow, idxCol)>=thresPixValN
                            
                            % bin properties
                            if plotValBoundMin==-Inf
                                binMin=min(SupResImC{idxRow, idxCol});
                            else
                                binMin=plotValBoundMin;
                            end
                            if plotValBoundMax==Inf
                            	binMax=max(SupResImC{idxRow, idxCol});
                                plotValBoundMax=binMax;
                            else
                                binMax=plotValBoundMax;
                            end
                            
                            if isempty(binMin) || isnan(binMin) || isempty(binMax) || isnan(binMax) || binMin==binMax     % is not meaningful
                                %outputData(idxRow, idxCol)=NaN;       % let it be
                            else
                                % if the bin values can be interpreted
                                if ~strcmp(quantityType, 'frame duration')
                                    frameDurationBool=false;
                                    [quantDistribution, histEdges]=rainSTORM_traj_calcHistData(SupResImC{idxRow, idxCol}, frameDurationBool, binMin, binMax, NBins);
                                else
                                    frameDurationBool=true;
                                    % becuse of the "'BinMethod', 'integers'" option in the "histogram"
                                    % function
                                    binMin=binMin-0.5;
                                    binMax=binMax+0.5;
                                    [quantDistribution, histEdges]=rainSTORM_traj_calcHistData(SupResImC{idxRow, idxCol}, frameDurationBool, binMin, binMax);
                                end
                                
                                % number of trajectories on the current
                                % image map pixel
                                histTrajN=sum(quantDistribution);
                                
                                % initial decay value for the exponential
                                % fit
                                expDecay_init=1/meanForExpLifeTimeMap(idxRow, idxCol);
                                options = optimset('Display','off');
                                
                                % fit the exponential by minimizing the
                                % residuum
                                if ~strcmp(quantityType, 'frame duration')
                                    if singleExpBool
                                        expParams_init=expDecay_init;
                                        [fittedParams, fitRes, exitFlag]=fminsearch(@(expParams) rainSTORM_traj_exponentialToFit(expParams, frameDurationBool, histTrajN, quantDistribution, true, histEdges), expParams_init, options);
                                        fittedDecay=fittedParams;
                                        exponentialInitVal=1;
                                    else
                                        expParams_init(3)=0.5;   % initial weight of the second exponential
                                        expParams_init(1)=expDecay_init*0.5;  % initial decay rate of the first exponential, let it be
                                        expParams_init(2)=expDecay_init*1.5;  % initial decay rate of the second exponential, let it be
                                        [fittedParams, fitRes, exitFlag]=fminsearch(@(expParams) rainSTORM_traj_exponentialToFit(expParams, frameDurationBool, histTrajN, quantDistribution, true, histEdges), expParams_init, options);
                                        fittedDecay=[fittedParams(1),fittedParams(2)];
                                        exponentialInitVal=rainSTORM_traj_exponentialToFit(fittedParams, frameDurationBool, histTrajN, quantDistribution, false, histEdges);
                                        outputData.multiExp.decay1=fittedDecay(1);
                                        outputData.multiExp.decay2=fittedDecay(2);
                                        outputData.multiExp.weight=fittedParams(3);
                                    end
                                else
                                    if singleExpBool
                                        expParams_init=expDecay_init;
                                        [fittedParams, fitRes, exitFlag]=fminsearch(@(expParams) rainSTORM_traj_exponentialToFit(expParams, frameDurationBool, histTrajN, quantDistribution, true, plotValBoundMin, plotValBoundMax), expParams_init, options);
                                        fittedDecay=fittedParams;
                                        %exponentialInitVal=1;
                                        exponentialInitVal=rainSTORM_traj_exponentialToFit(fittedParams, frameDurationBool, histTrajN, quantDistribution, false, plotValBoundMin, plotValBoundMax);
                                    else
                                        expParams_init(3)=0.5;   % initial weight of the second exponential
                                        expParams_init(1)=expDecay_init*0.5;  % initial decay rate of the first exponential, let it be
                                        expParams_init(2)=expDecay_init*1.5;  % initial decay rate of the second exponential, let it be
                                        [fittedParams, fitRes, exitFlag]=fminsearch(@(expParams) rainSTORM_traj_exponentialToFit(expParams, frameDurationBool, histTrajN, quantDistribution, true, plotValBoundMin, plotValBoundMax), expParams_init, options);
                                        fittedDecay=[fittedParams(1),fittedParams(2)];
                                        exponentialInitVal=rainSTORM_traj_exponentialToFit(fittedParams, frameDurationBool, histTrajN, quantDistribution, false, plotValBoundMin, plotValBoundMax);
                                        outputData.multiExp.lifetime1=1/fittedDecay(1);
                                        outputData.multiExp.lifetime2=1/fittedDecay(2);
                                        outputData.multiExp.weight=fittedParams(3);
                                    end
                                end
                                
                                if exitFlag==1
                                    % if the fitting was successful
                                    switch mapOrHist
                                        case 'map'
                                            switch mapType
                                                case 'fitted exponential decay'
                                                    outputData(idxRow, idxCol)=sum(exponentialInitVal./fittedDecay.^2)/sum(exponentialInitVal./fittedDecay);        % expection value of the fitted exponential
                                                case 'fitted exponential residuum'
                                                    outputData(idxRow, idxCol)=fitRes;
                                            end
                                        case 'hist'
                                            outputData.mean=meanOfValMap;
                                            outputData.meanForExpLifeTime=meanForExpLifeTimeMap;
                                            outputData.expectedLifeTime=sum(exponentialInitVal./fittedDecay.^2)/sum(exponentialInitVal./fittedDecay);               % expection value of the fitted exponential
                                            outputData.fitResiduum=fitRes;
                                    end
                                else
                                    switch mapOrHist
                                        case 'map'
                                            outputData(idxRow, idxCol)=NaN;
                                        case 'hist'
                                            outputData.mean=NaN;
                                            outputData.meanForExpLifeTime=NaN;
                                            outputData.expectedLifeTime=NaN;
                                            outputData.fitResiduum=NaN;
                                    end
                                end
                            end
                        else
                            switch mapOrHist
                                case 'map'
                                    outputData(idxRow, idxCol)=NaN;
                                case 'hist'
                                    outputData.mean=NaN;
                                    outputData.meanForExpLifeTime=NaN;
                                    outputData.expectedLifeTime=NaN;
                                    outputData.fitResiduum=NaN;
                            end
                        end
                    end
                end
        end
end

end

