classdef rainSTORM_trajectoryFitting
    % TODO: description
    % This function performs the trajectory fitting on the localizations
    % belonging to different frames.
    % First, in the "concatenateLocsFunc" creates clusters from the
    % trajectories. Every trajectory contains exactly one element on each frame
    % during their existence. The actaul position of the trajectory is the
    % position of the last localization of the trajectory. To determine the
    % which localization might belong to the trajectory on the next frame,
    % calculates the distances of the localization and the trajectory positions,
    % and check whether a distance falls within the given limit (set constans
    % value, or based on the localization precision). If no new localization
    % belong to a trajectory, then terminates it. If several trajectories might
    % belong to a trajectory, concatenates the closest one, and connect the
    % trajectory to the other trajectory to which the other satisfactory
    % localizations were concatenated. If a single trajectory satisfy the
    % distance condition for more than one trajectories, concatenates it to one
    % of it and connects these trajectories.
    % After it, select the "simple" trajectories from those that are not
    % connected to any another one. The other trajectories forms clusters.
    % In the next step, finds the all the clusters, and based on the distances,
    % fitted Gaussian peak, area under the fitted Gaussian or on the Gaussian
    % PSF widths creates new trajectories and seperates them from the cluster.
    % These are called "complex trajectories".
    
    methods (Static)
        
        
        function [simpleTrajs, complexTrajs, trajsInfo, metadata, sharedParameters] = perform(localizationData, method, settings)
            % This function runs the selected trajectory fitting algorithm
                    
            if ~strcmp(method, 'branching trajectories')
               error('This is not the concatenation method you want to use: % ', method) 
            end
            
            disp('Preparing trajectory fitting.')

            % pass the settigns
            cameraPixelSize_nm = settings.cameraPixelSize_nm;

            % create "number of localizations"x1x3 dimensional double array for localization positions
            % this can be unsafe as it depend on the dimensions of the source data:
            % locPos(:,1,:)= [localizationData.x_coord, localizationData.y_coord, localizationData.z_coord];
            % the safer way:
            locPos(:,1,1)= double(localizationData.x_coord)*cameraPixelSize_nm;
            locPos(:,1,2)= double(localizationData.y_coord)*cameraPixelSize_nm;
            if pointillisticData_fieldNameManagement.isMemberField(localizationData, 'z_coord')
                locPos(:,1,3)= double(localizationData.z_coord)*cameraPixelSize_nm; % start with "z" to initialize the whole array to avoid resizing it
            else
                locPos(:,1,3)= zeros(size(localizationData.x_coord));
            end


            % create a vector for the frame indices of the localizations
            locFrames=[localizationData.frame_idx];

            % create vectors for the other parameters of the localized blinking events
            try
                locProperties.X_STD(:,1)=double(localizationData.std_x)*cameraPixelSize_nm;
                locProperties.Y_STD(:,1)=double(localizationData.std_y)*cameraPixelSize_nm;
            catch
                locProperties.X_STD(:,1)=double(localizationData.std)*cameraPixelSize_nm;
                locProperties.Y_STD(:,1)=double(localizationData.std)*cameraPixelSize_nm;
            end
            try
                locProperties.SigX(:,1)=double(localizationData.sig_x)*cameraPixelSize_nm;
                locProperties.SigY(:,1)=double(localizationData.sig_y)*cameraPixelSize_nm;
            catch
                locProperties.SigX(:,1)=double(localizationData.sig)*cameraPixelSize_nm;
                locProperties.SigY(:,1)=double(localizationData.sig)*cameraPixelSize_nm;
            end
            locProperties.SumSignal(:,1)=double(localizationData.sum_signal);
            % Gaussian peak heights:
            locProperties.GaussianPeak(:,1)=locProperties.SumSignal./(2*pi*locProperties.SigX.*locProperties.SigY);


            %%

            disp('Trajectory fitting has been started.')
            tic

            % catenating the localizations
            trajsInfo=rainSTORM_trajectoryFitting.concatenateLocalizations(locPos, locFrames, settings, locProperties.X_STD, locProperties.Y_STD);
            % "trajsInfo" contains information on the indices of localization belonging
            % to the trajectory, whether the trajectory is simple or not, and on the
            % trajectory position

            % calculate the trajectory properties
            if ~pointillisticData_fieldNameManagement.isMemberField(localizationData, 'z_coord')
                localizationData.z_coord = zeros(size(localizationData.x_coord));
            end
            if ~pointillisticData_fieldNameManagement.isMemberField(localizationData, 'I')
                localizationData.I = locProperties.GaussianPeak;
            end
            if ~pointillisticData_fieldNameManagement.isMemberField(localizationData, 'res')
                localizationData.res = zeros(size(localizationData.x_coord));
            end
            if ~pointillisticData_fieldNameManagement.isMemberField(localizationData, 'sig_x')
                localizationData.sig_x = localizationData.sig;
            end
            if ~pointillisticData_fieldNameManagement.isMemberField(localizationData, 'sig_y')
                localizationData.sig_y = localizationData.sig;
            end
            [simpleTrajs, complexTrajs] = rainSTORM_trajectoryFitting.calculateProperties(localizationData, trajsInfo);

            % seperate trajectories from the complex ones and concatenate
            % them with the simple ones
            %trajsInfo_seperated=rainSTORM_trajectoryFitting.seperateTrajectories(trajsInfo, locPos, locFrames, locProperties, settings);

            % calculate the trajectory properties
            %[acceptedTrajs, rejectedTrajs] = trajectoryFitting.calculateProperties(localizationData, trajsInfo_seperated);
            
            metadata = struct([]);
            sharedParameters = struct([]);
            
        end
        
        
        function [simpleTrajs, complexTrajs] = calculateProperties(localizationData, trajsInfo)
        % This function calculates the trajectory properties from the
        % properties of the localizations belonging t the trajectories 
        
        
        % number of all trajectories
        trajN=numel(trajsInfo.simpleTrajBool);
        % number of simple trajectories
        trajN_simple=sum([trajsInfo(:).simpleTrajBool]);
        trajN_complex=trajN-trajN_simple;
        
        simpleTrajIndices=find([trajsInfo(:).simpleTrajBool]);
        complexTrajIndices=find(~[trajsInfo(:).simpleTrajBool]);
        
        % allocating memory space for the trajectory output parameters
        simpleTrajs.frame_idx=uint32(zeros(trajN,1));
        simpleTrajs.frame_N=uint32(zeros(trajN,1));
        simpleTrajs.sum_signal=single(zeros(trajN,1));
        simpleTrajs.signalPerFrame=single(zeros(trajN,1));
        simpleTrajs.x_coord=single(zeros(trajN,1));
        simpleTrajs.y_coord=single(zeros(trajN,1));
        simpleTrajs.z_coord=single(zeros(trajN,1));
        %acceptedTrajs.isSimple=single(zeros(trajN,1));
        simpleTrajs.max_I=single(zeros(trajN,1));
        simpleTrajs.maxSignal=single(zeros(trajN,1));
        simpleTrajs.meanSigX=single(zeros(trajN,1));
        simpleTrajs.meanSigY=single(zeros(trajN,1));
        simpleTrajs.meanRes=single(zeros(trajN,1));
        
        for idxTraj=trajN_simple:-1:1
            % localization indices belonging to the actual trajectory
            actTraj_locIndices=trajsInfo(simpleTrajIndices(idxTraj)).locIndices;
            
            % simple trajectory output parameters
            simpleTrajs.frame_idx(idxTraj,1)=min(localizationData.frame_idx(actTraj_locIndices));
            simpleTrajs.frame_N(idxTraj,1)=length(actTraj_locIndices);
            simpleTrajs.sum_signal(idxTraj,1)=sum(localizationData.sum_signal(actTraj_locIndices));
            simpleTrajs.signalPerFrame(idxTraj,1)=mean(localizationData.sum_signal(actTraj_locIndices(2:end-1)));
            simpleTrajs.x_coord(idxTraj,1)=sum(localizationData.x_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            simpleTrajs.y_coord(idxTraj,1)=sum(localizationData.y_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            simpleTrajs.z_coord(idxTraj,1)=sum(localizationData.z_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            %acceptedTrajs.isSimple(idxTraj,1)=trajsInfo_seperated.isSimple;
            simpleTrajs.max_I(idxTraj,1)=max(localizationData.I(actTraj_locIndices(1:end)));
            if isempty(actTraj_locIndices(2:end-1))
                %acceptedTrajs.max_I(idxTraj,1)=NaN;
                simpleTrajs.maxSignal(idxTraj,1)=NaN;
            else
                %acceptedTrajs.max_I(idxTraj,1)=max(localizationData.I(actTraj_locIndices(2:end-1)));
                simpleTrajs.maxSignal(idxTraj,1)=max(localizationData.sum_signal(actTraj_locIndices(2:end-1)));
            end
            simpleTrajs.meanSigX(idxTraj,1)=sum(localizationData.sig_x(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            simpleTrajs.meanSigY(idxTraj,1)=sum(localizationData.sig_y(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            simpleTrajs.meanRes(idxTraj,1)=sum(localizationData.res(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/simpleTrajs.sum_signal(idxTraj,1);
            
        end
        
        % create empty data for the rejected trajectories
        complexTrajs = pointillisticData_fieldManagement.createEmptyData(simpleTrajs);
        
        for idxTraj=trajN_complex:-1:1
            % localization indices belonging to the actual trajectory
            actTraj_locIndices=trajsInfo(complexTrajIndices(idxTraj)).locIndices;
            
            % simple trajectory output parameters
            complexTrajs.frame_idx(idxTraj,1)=min(localizationData.frame_idx(actTraj_locIndices));
            complexTrajs.frame_N(idxTraj,1)=length(actTraj_locIndices);
            complexTrajs.sum_signal(idxTraj,1)=sum(localizationData.sum_signal(actTraj_locIndices));
            complexTrajs.signalPerFrame(idxTraj,1)=mean(localizationData.sum_signal(actTraj_locIndices(2:end-1)));
            complexTrajs.x_coord(idxTraj,1)=sum(localizationData.x_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            complexTrajs.y_coord(idxTraj,1)=sum(localizationData.y_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            complexTrajs.z_coord(idxTraj,1)=sum(localizationData.z_coord(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            complexTrajs.max_I(idxTraj,1)=max(localizationData.I(actTraj_locIndices));
            if isempty(actTraj_locIndices(2:end-1))
                complexTrajs.maxSignal(idxTraj,1)=NaN;
            else
                complexTrajs.maxSignal(idxTraj,1)=max(localizationData.sum_signal(actTraj_locIndices(2:end-1)));
            end
            complexTrajs.meanSigX(idxTraj,1)=sum(localizationData.sig_x(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            complexTrajs.meanSigY(idxTraj,1)=sum(localizationData.sig_y(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            complexTrajs.meanRes(idxTraj,1)=sum(localizationData.res(actTraj_locIndices).*localizationData.sum_signal(actTraj_locIndices))/complexTrajs.sum_signal(idxTraj,1);
            
        end
        end
        
        function [ trajQuantitesAccepted, trajQuantitesRejected ] = filter( dataSimpleTrajs, dataComplexTrajs, catComplexTrajsBool, calcParamsStruct, quantityType, mapBool, frameIdxBool, varargin)
        % 	Summary:	
        %	Version:    3.2	
        %	Note:		
        %	Package:	rainSTORM v3.1+
        %	Developer:	novakt
        %	Name:		Novák Tibor
        %	Company:	University of Szeged
        %	Date of the merger: 2017.11.08.	

        % This function concatenates the "simple" and "comoplex" trajectories,
        % filters the trajectories based on the given filter setting on the
        % trajectory "Vis. settings" panel and in case of using the histogram
        % tracker, cuts out the rectangular region.

        % parents: reconTrayGraphsGUIElementsFunc
        % children:

        %% catenate simple and complex trajectories if needed
        if catComplexTrajsBool && ~isempty(dataComplexTrajs.frame_idx)
            fieldNames=fieldnames(dataSimpleTrajs);
            for idxField=1:length(fieldNames)
                trajectoryData.(fieldNames{idxField})=[
                    dataSimpleTrajs.(fieldNames{idxField});...
                    dataComplexTrajs.(fieldNames{idxField})];
            end
        else
            trajectoryData=dataSimpleTrajs;
        end

        %% preparation
        % vectors for the quantites that are calculated from the raw trajectory
        % data, make them empty by default, calculate them only if needed
        vect_NPhoton=[];
        vect_emPower=[];
        vect_lifeTime=[];


        %% Examining which trajectories should be filtered out
        % boolean vector contining witch tharejtory should be kept or filtered out
        filterBoolVect=true(size(trajectoryData.frame_N));

        % filtering based on trajectory position, cutting outt a rectangular
        % region, for the histogram tracker
        if ~isempty(varargin)
            xBounds=varargin{1};
            yBounds=varargin{2};

            filterBoolVect_pos=(...
                xBounds(1)<trajectoryData.x_coord &...
                xBounds(2)>trajectoryData.x_coord &...
                yBounds(1)<trajectoryData.y_coord &...
                yBounds(2)>trajectoryData.y_coord );
        end

        % if filtering should be applied by the localized photon number
        if calcParamsStruct.NPhotonBounds(1)>-Inf || calcParamsStruct.NPhotonBounds(2)<Inf
            % number of emitted photons belonging to the blinking events, and a
            % little approximation never hurts
            vect_NPhoton=trajectoryData.sum_signal*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);
            % update the filter vector based on the emitted photon number
            filterBoolVect(calcParamsStruct.NPhotonBounds(1)>vect_NPhoton | calcParamsStruct.NPhotonBounds(2)<vect_NPhoton)=false;


            %!!!! EXPERIMENTAL
            %vect_NPhoton=dataTrajs.signalMax*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);
            % update the filter vector based on the emitted photon number
            %filterBoolVect(calcParamsStruct.NPhotonBounds(1)>vect_NPhoton | calcParamsStruct.NPhotonBounds(2)<vect_NPhoton)=false;
        end
        % if filtering should be applied by the frame duration of the blinking
        % events
        if calcParamsStruct.frameDurationBounds(1)>-Inf || calcParamsStruct.frameDurationBounds(2)<Inf
            % update the filter vector based on the frame duration
            filterBoolVect(calcParamsStruct.frameDurationBounds(1)>trajectoryData.frame_N | calcParamsStruct.frameDurationBounds(2)<trajectoryData.frame_N)=false;
        end
        % if filtering should be applied by the emission power
        if calcParamsStruct.emPowerBounds(1)>-Inf || calcParamsStruct.emPowerBounds(2)<Inf
            % emission power (photon/sec) of the blinking events
            vect_emPower=trajectoryData.signalPerFrame*(calcParamsStruct.photonPerCount/calcParamsStruct.exposureTime);
            % update the filter vector based on the emission power
            filterBoolVect(calcParamsStruct.emPowerBounds(1)>vect_emPower | calcParamsStruct.emPowerBounds(2)<vect_emPower)=false;
        end
        % if filtering should be applied by the lifetime of the ON state
        if calcParamsStruct.lifeTimeBounds(1)>-Inf || calcParamsStruct.lifeTimeBounds(2)<Inf
            if isempty(vect_NPhoton)
                % number of emitted photons belonging to the blinking events, and a
                % little approximation never hurts
                vect_NPhoton=trajectoryData.sum_signal*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);

                %!!!! EXPERIMENTAL
                %vect_NPhoton=dataTrajs.signalMax*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);
            end
            if isempty(vect_emPower)
                % emission power (photon/sec) of the blinking events
                vect_emPower=trajectoryData.signalPerFrame*(calcParamsStruct.photonPerCount/calcParamsStruct.exposureTime);
            end
            % ON state lifetime of the blinking events
            vect_lifeTime=vect_NPhoton./vect_emPower;
            % update the filter vector based on the ON state lifetime
            filterBoolVect(calcParamsStruct.lifeTimeBounds(1)>vect_lifeTime | calcParamsStruct.lifeTimeBounds(2)<vect_lifeTime)=false;
        end

        if ~isempty(varargin)
            filterBoolVect_accepted=filterBoolVect & filterBoolVect_pos;
            filterBoolVect_rejected=~filterBoolVect & filterBoolVect_pos;
        else
            filterBoolVect_accepted=filterBoolVect;
            filterBoolVect_rejected=~filterBoolVect;
        end

        %% Calculating the filtered, desired quantites (both accepted and rejected by the filtering)
        if frameIdxBool
            trajQuantitesAccepted.frame_idx=trajectoryData.frame_idx(filterBoolVect_accepted,:);
            trajQuantitesRejected.frame_idx=trajectoryData.frame_idx(filterBoolVect_rejected,:);
        end

        if mapBool
            trajQuantitesAccepted.x_coord=trajectoryData.x_coord(filterBoolVect_accepted,:);
            trajQuantitesRejected.x_coord=trajectoryData.x_coord(filterBoolVect_rejected,:);
            trajQuantitesAccepted.y_coord=trajectoryData.y_coord(filterBoolVect_accepted,:);
            trajQuantitesRejected.y_coord=trajectoryData.y_coord(filterBoolVect_rejected,:);
        end

        % accepted and rejected blinking event's photon numbers
        if any(strcmp(quantityType, 'photon number')) || sum(strcmp(quantityType, 'lifetime'))
            if isempty(vect_NPhoton)
                % number of emitted photons belonging to the blinking events, and a
                % little approximation never hurts
                vect_NPhoton=trajectoryData.sum_signal*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);

                %!!!! EXPERIMENTAL
                %vect_NPhoton=dataTrajs.signalMax*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);
            end
            trajQuantitesAccepted.NPhoton=vect_NPhoton(filterBoolVect_accepted);
            trajQuantitesRejected.NPhoton=vect_NPhoton(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's frame duration
        if any(strcmp(quantityType, 'frame duration'))
            trajQuantitesAccepted.frame_N=trajectoryData.frame_N(filterBoolVect_accepted);
            trajQuantitesRejected.frame_N=trajectoryData.frame_N(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's emission powers
        if any(strcmp(quantityType, 'emission power')) || sum(strcmp(quantityType, 'lifetime'))
            if isempty(vect_emPower)
                % emission power (photon/sec) of the blinking events
                vect_emPower=trajectoryData.signalPerFrame*(calcParamsStruct.photonPerCount/calcParamsStruct.exposureTime);
            end
            trajQuantitesAccepted.emPower=vect_emPower(filterBoolVect_accepted);
            trajQuantitesRejected.emPower=vect_emPower(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's ON state lifetimes
        if any(strcmp(quantityType, 'lifetime'))
            if isempty(vect_lifeTime)
                if isempty(vect_NPhoton)
                    % number of emitted photons belonging to the blinking events, and a
                    % little approximation never hurts
                    vect_NPhoton=trajectoryData.sum_signal*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);

                    %!!!! EXPERIMENTAL
                    %vect_NPhoton=dataTrajs.signalMax*(calcParamsStruct.photonPerCount*calcParamsStruct.frameTime/calcParamsStruct.exposureTime);
                end
                if isempty(vect_emPower)
                    % emission power (photon/sec) of the blinking events
                    vect_emPower=trajectoryData.signalPerFrame*(calcParamsStruct.photonPerCount/calcParamsStruct.exposureTime);
                end
                % ON state lifetime of the blinking events
                vect_lifeTime=vect_NPhoton./vect_emPower;
            end
            trajQuantitesAccepted.lifeTime=vect_lifeTime(filterBoolVect_accepted);
            trajQuantitesRejected.lifeTime=vect_lifeTime(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's residuum of the fitted Gaussians
        if any(strcmp(quantityType, 'PSF residuum'))
            trajQuantitesAccepted.meanRes=trajectoryData.meanRes(filterBoolVect_accepted);
            trajQuantitesRejected.meanRes=trajectoryData.meanRes(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's PSF width in the "x" direction
        if any(strcmp(quantityType, 'PSF width X'))
            trajQuantitesAccepted.meanSigX=trajectoryData.meanSigX(filterBoolVect_accepted);
            trajQuantitesRejected.meanSigX=trajectoryData.meanSigX(filterBoolVect_rejected);
        end
        % accepted and rejected blinking event's PSF width in the "y" direction
        if any(strcmp(quantityType, 'PSF width Y'))
            trajQuantitesAccepted.meanSigY=trajectoryData.meanSigY(filterBoolVect_accepted);
            trajQuantitesRejected.meanSigY=trajectoryData.meanSigY(filterBoolVect_rejected);
        end


        end
        
        function trajsInfo=concatenateLocalizations(locPos, locFrames, settings, varargin)
            % This function find the trajectories by simply concatenating
            % the localization on the subsequent frames
            
            tic
            
            % create a vector for the indices of the localizations
            locIndices=uint32(zeros(numel(locFrames), 1));
            locIndices(:)=1:numel(locIndices);
            
            % passing the settings for the trajectory fitting
            ZPrecConst=settings.ZPrecConst;
            
            % varible containing the number of frames for how long a trajectory can be
            % active without any new localization
            % it is intended to circumvent the error caused by fluorophores
            % with altering brightness as small brightness might interrupt
            % the trajectory fitting and slice the blinking event into
            % several trajectories
            maxNumberOfNoNewLocsInARow=settings.maxNumberOfNoNewLocsInARow;
            
            thresholdingMethod = settings.thresholdingMethod;
            distThreshold_nm = settings.distThreshold_nm;
            
            switch thresholdingMethod
                case 'fixed distance'
                    % concatenate the trajectories based purely on their
                    % Eucledian distances
                    distThresholdSquare=(distThreshold_nm)^2;
                case 'distance based on precision'
                    % for the concatenation distances, consider the
                    % localization's precision values
                    % constants for determining whether the localization can be
                    % concatenated to the trajectory ( this small section might be needed to be reworked)
                    normDistSquare_precConst=settings.normDistSquare_precConst;
                    
                    loc_X_STD=varargin{1};
                    loc_Y_STD=varargin{2};
                otherwise
                    error('Unknown thresholding method for the trajectory fitting.')
            end
            
            % template for the "trajsInfo" structure for starting new
            % trajectories that should not be catenated to another one
            trajsInfoBrandNewTemplate.locIndices=[];
            trajsInfoBrandNewTemplate.severalLocsBool=false;
            trajsInfoBrandNewTemplate.mergedTrajIndices=[];
            trajsInfoBrandNewTemplate.severalTrajBool=false;
            trajsInfoBrandNewTemplate.simpleTrajBool=true;
            %trajsInfoBrandNewTemplate.pos=[];
            
            % template for the "trajsInfo" structure for starting new
            % trajectories that should be catenated to another one
            trajsInfoMergedTemplate.locIndices=[];
            trajsInfoMergedTemplate.severalLocsBool=false;
            trajsInfoMergedTemplate.mergedTrajIndices=[];
            trajsInfoMergedTemplate.severalTrajBool=false;
            trajsInfoMergedTemplate.simpleTrajBool=false;
            %trajsInfoMergedTemplate.pos=[];
            
            
            firstMeaningfulFrameBool=true;
            firstMeaningfulFrameIdx=locFrames(1);
            while firstMeaningfulFrameBool
                % trajectories on the not empty first frame:
                actTrajLastLocIndices(:,1)=locIndices(locFrames==firstMeaningfulFrameIdx);
                % lets define the trajectory position this way:
                actTrajPos=permute(locPos(actTrajLastLocIndices,1,:),[2,1,3]); % initial trajectory positions, 1x"number of localizations"x2 dimensional double array, start it with the localization positions on the first frame
                % initialize the "trajInfo" structure with the localisations on the
                % not empty first frame
                actTrajIndices=(1:size(actTrajPos,2))';
                
                % variable containing information about for how many frames
                % have not the active trajectory got any new localization
                actTrajNoNewLocState=uint8(zeros(size(actTrajIndices)));
                
                if numel(actTrajIndices)>0
                    trajsInfo=repmat(trajsInfoBrandNewTemplate, size(actTrajIndices));
                    for idxTraj=1:numel(actTrajIndices)
                        trajsInfo(idxTraj).locIndices=actTrajIndices(idxTraj);  % stands because of the first frame
                        %trajsInfo(idxTraj).pos=actTrajPos(1,idxTraj,:);         % stands because of the first frame
                    end
                    
                    firstMeaningfulFrameBool=false;
                else
                    firstMeaningfulFrameIdx=firstMeaningfulFrameIdx+1;
                end
            end
            
            
            % trajectories on the subsequent frames:
            for idxFrame=firstMeaningfulFrameIdx+1:locFrames(end)
                %% The blinking events on the actual frame
                % positions of localized events on the actual frame
                currentLocPos=[];
                %currentLocPos(:,1,:)=locPos(locFrames==idxFrame,1,:);
                currentLocPos=locPos(locFrames==idxFrame,1,1:3);
                % indices of localizations on the current frame
                currentLocIndices=[];
                currentLocIndices(:,1)=locIndices(locFrames==idxFrame);
                
                % current number of localizations and active trajectories
                currentLocN=size(currentLocPos,1);
                actTrajN=size(actTrajPos,2);
                
                %% Determining which localization might belong to which trajectory
                switch thresholdingMethod
                    case 'fixed distance'
                        % matrix contaning the squares of distances between the trajectory and localized positions
                        distSquareTrajLoc=(repmat(actTrajPos,currentLocN,1,1)-repmat(currentLocPos,1, actTrajN,1)).^2;
                        % matrix contaning the squares of normalized distances between the trajectory and localized positions
                        normDistSquareTrajLoc=sum(distSquareTrajLoc(:,:,1:2),3)/distThresholdSquare+distSquareTrajLoc(:,:,3)/(distThresholdSquare*ZPrecConst);
                        
                        % boolean matrix contaning information about whether the blinking event were close enough to one of the trajectories
                        pairedTrajLocAccDistBoolMat=normDistSquareTrajLoc<=1;
                    case 'distance based on precision'
                        normDistSquareTrajLoc=rainSTORM_trajectoryFitting.trajLocDistPrecFunc(actTrajPos, currentLocPos, loc_Y_STD, loc_X_STD, ZPrecConst, currentLocIndices, actTrajLastLocIndices, normDistSquare_precConst);
                        
                        % boolean matrix contaning information about whether the blinking event were close enough to one of the trajectories
                        pairedTrajLocAccDistBoolMat=normDistSquareTrajLoc<=1;
                end
                
                % localization and trajectory indices in the accepted distance boolean matrix
                [locAccDistBoolMat_subscripts,trajAccDistBoolMat_subscripts]=ind2sub(size(pairedTrajLocAccDistBoolMat), find(pairedTrajLocAccDistBoolMat));
                if currentLocN==1 % MATLAB REALLY SHOULD RAPE ITSELF IN THE ASS
                    locAccDistBoolMat_subscripts=locAccDistBoolMat_subscripts';
                    trajAccDistBoolMat_subscripts=trajAccDistBoolMat_subscripts';
                end
                
                % localization and trajectory indices in the minimal distance (non-calculated) boolean matrix
                trajMinDistBoolMat_subscripts=unique(trajAccDistBoolMat_subscripts);        % to these trajcetories belongs at least one localization within accepted distance
                [~, locMinDistBoolMat_subscripts]=min(normDistSquareTrajLoc(:,trajMinDistBoolMat_subscripts), [], 1);        % indices of localizations of the accepted and minimal traj-loc distances
                % it does not matter if one localization belongs to several
                % frame
                
                if ~isempty(trajAccDistBoolMat_subscripts)
                    %% Loop through the trajectories
                    for trajLocalIdx=trajMinDistBoolMat_subscripts'
                        trajsInfo_structArrayIdx=actTrajIndices(trajLocalIdx);
                        
                        % append the nearest localizations within the acceptance distance to the trajectories
                        trajsInfo(trajsInfo_structArrayIdx).locIndices=[...
                            trajsInfo(trajsInfo_structArrayIdx).locIndices;...
                            currentLocIndices(locMinDistBoolMat_subscripts(trajMinDistBoolMat_subscripts==trajLocalIdx))];
                        
                        % finding out whether more localized events belongs to this
                        % trajectory on a single frame
                        if length(locAccDistBoolMat_subscripts(trajAccDistBoolMat_subscripts==trajLocalIdx))>1
                            trajsInfo(trajsInfo_structArrayIdx).severalLocsBool=true;
                        end
                        
                        
                        % finding out whether there is a localized event in this trajectory that
                        % also belong to another trajectory, e.g. whether the trajectory is
                        % connected to another one
                        % cut out the row of localization belonging to the actual
                        % trajectory (leave out the column of the actual trajectory)
                        % from the boolean matrix containing which
                        % localization is within which trajectory's acceptance
                        % radius, so the indices of trajectories that should be
                        % merged with the actual trajectory can be obtained
                        cutBoolMat_forMergeTraj=pairedTrajLocAccDistBoolMat(locAccDistBoolMat_subscripts(trajAccDistBoolMat_subscripts==trajLocalIdx),:);
                        cutBoolMat_forMergeTraj(:, trajLocalIdx)=false;
                        
                        % get the trajectory indices to be merged with the actual
                        % trajectory
                        [~,trajCutBoolMat_subscripts]=ind2sub(size(cutBoolMat_forMergeTraj), find(cutBoolMat_forMergeTraj));
                        trajCutBoolMat_uniqueSubscripts=unique(trajCutBoolMat_subscripts);
                        % append the trajectory indices to be merged
                        trajsInfo(trajsInfo_structArrayIdx).mergedTrajIndices=[...
                            trajsInfo(trajsInfo_structArrayIdx).mergedTrajIndices;...
                            actTrajIndices(trajCutBoolMat_uniqueSubscripts)];
                        
                        % if there are any trajectories that this trajectory should
                        % be merged with
                        if ~isempty(actTrajIndices(trajCutBoolMat_subscripts))
                            trajsInfo(trajsInfo_structArrayIdx).severalTrajBool=true;
                        end
                        
                        % if the trajectory is simple, e.g. does not connected to other
                        % trajectories or contain several localizations on a single frame
                        if trajsInfo(trajsInfo_structArrayIdx).severalTrajBool==true || trajsInfo(trajsInfo_structArrayIdx).severalLocsBool==true
                            trajsInfo(trajsInfo_structArrayIdx).simpleTrajBool=false;
                        end
                        
                        % update the trajectory position
                        %trajsInfo(trajsInfo_structArrayIdx).pos=actTrajPos(1,trajLocalIdx,:);
                        
                        
                    end
                end
                
                %% Finding the refused or the left out localizations
                % boolean indices of localizations that were not catenated to
                % any of the trajectories
                locRefusedToAnyTraj_boolIndices=~any(pairedTrajLocAccDistBoolMat, 2);
                
                % subscript indices of localizations that were concatenated to
                % any of the trajectories
                locAccToAnyTraj_subscripts=locMinDistBoolMat_subscripts;
                
                % boolean indices of localizations that were concatenated to any
                % of the trajectories
                locAccToAnyTraj_boolIndices=false(size(locRefusedToAnyTraj_boolIndices));
                locAccToAnyTraj_boolIndices(locAccToAnyTraj_subscripts)=true;
                
                % indices of localizations that were in the acceptance radius of
                % one trajectory but was not concatenated to it
                locAccButNotMinAnyTraj_boolIndices=~(locRefusedToAnyTraj_boolIndices | locAccToAnyTraj_boolIndices); % NOR logic
                
                % number of localizations that ere not catenated to any trajectory
                trajsN_toAppend_accButNotMin=nnz(locAccButNotMinAnyTraj_boolIndices);  % within accectance distance, but left out
                trajsN_toAppend_refused=nnz(locRefusedToAnyTraj_boolIndices);            % refused
                trajsN_toAppend=trajsN_toAppend_accButNotMin+trajsN_toAppend_refused;   % overall
                
                % first ( and  last) member (localization) indices of the new trajectories from the localizations that were
                % not catenated to any trajectory
                firstLocIndicesNewTraj_toAppend_accButNotMin=currentLocIndices(locAccButNotMinAnyTraj_boolIndices);
                firstLocIndicesNewTraj_toAppend_refused=currentLocIndices(locRefusedToAnyTraj_boolIndices);
                
                %% Preparing the preparation for the new trajectories for the next frame
                
                trajAccLoc_subscripts=unique(trajMinDistBoolMat_subscripts);
                
                % subscript indices of trajectories which have new
                % localizations
                trajAccLoc_boolIndices=false(size(actTrajIndices));
                trajAccLoc_boolIndices(trajAccLoc_subscripts)=true;
                
                % creating indices for trajectories to be created for the left
                % out or rejected localizations
                newTrajLocalIndices_accButNotMin=(1:trajsN_toAppend_accButNotMin)';
                newTrajLocalIndices_refused=(trajsN_toAppend_accButNotMin+1:trajsN_toAppend)';
                newTrajLocalIndices=[...
                    newTrajLocalIndices_accButNotMin;...
                    newTrajLocalIndices_refused];
                % indices of these trajectories
                newTrajIndices=newTrajLocalIndices+length(trajsInfo);
                
                % new indices of the last members (localizations) of the active trajectories
                newLastLocIndicesActTraj=currentLocIndices(locAccToAnyTraj_subscripts');
                
                % indices of trajectories that have no new localizations on the
                % current frame
                actTrajLocalIndices=1:numel(actTrajIndices);
                actTrajNoNewLastLocLocalIndices=actTrajLocalIndices(~trajAccLoc_boolIndices);
                
                % new indices of the first (which are the last, too) member (localizations) of the new trajectories
                lastLocIndicesNewTraj=[...
                    firstLocIndicesNewTraj_toAppend_accButNotMin;...
                    firstLocIndicesNewTraj_toAppend_refused]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                
                % increase the this value of trajectories that have no new
                % localizations on the current frame
                actTrajNoNewLocState(actTrajNoNewLastLocLocalIndices)=actTrajNoNewLocState(actTrajNoNewLastLocLocalIndices)+1;
                
                % positions, last member (localization) indices and trajectory
                % indices of trajectories that have no new localizations on the
                % current frame but are not yet terminated (still active)
                actTrajBoolIndices_noNewLocButNotTerminated=actTrajNoNewLocState<=maxNumberOfNoNewLocsInARow & actTrajNoNewLocState>0 & ~trajAccLoc_boolIndices;
                %actTrajN_noNewLocButNotTerminated=nnz(actTrajBoolIndices_noNewLocButNotTerminated);
                %actTrajPos_noNewLocButNotTerminated=[];
                %actTrajPos_noNewLocButNotTerminated(1,1:actTrajN_noNewLocButNotTerminated,:)=actTrajPos(1,actTrajBoolIndices_noNewLocButNotTerminated,:);
                actTrajPos_noNewLocButNotTerminated=actTrajPos(1,actTrajBoolIndices_noNewLocButNotTerminated,:);
                actTrajLastLocIndices_noNewLocButNotTerminated=actTrajLastLocIndices(actTrajBoolIndices_noNewLocButNotTerminated);
                actTrajIndices_noNewLocButNotTerminated=actTrajIndices(actTrajBoolIndices_noNewLocButNotTerminated);
                
                
                %% Appending the newly created trajectories for the "trajsInfo" structure
                % extending the "trajsInfo" structure
                if trajsN_toAppend_accButNotMin>0 || trajsN_toAppend_refused>0
                    if trajsN_toAppend_accButNotMin>0 && trajsN_toAppend_refused>0
                        trajsInfo=[...
                            trajsInfo;...
                            repmat(trajsInfoMergedTemplate, [trajsN_toAppend_accButNotMin,1]);...
                            repmat(trajsInfoBrandNewTemplate, [trajsN_toAppend_refused,1])];
                    elseif trajsN_toAppend_accButNotMin==0
                        trajsInfo=[...
                            trajsInfo;...
                            repmat(trajsInfoBrandNewTemplate, [trajsN_toAppend_refused,1])];
                    elseif trajsN_toAppend_refused==0
                        trajsInfo=[...
                            trajsInfo;...
                            repmat(trajsInfoMergedTemplate, [trajsN_toAppend_accButNotMin,1])];
                    end
                    
                    % initialize "trajsInfo" structure for new trajectories, needed
                    % to do it before overwriting "actTrajIndices"
                    currentLocSubscripts=(1:currentLocN)';          % indices (not boolean) of the localizations on the current frame
                    currentLocSubscripts_accButNotMin=currentLocSubscripts(locAccButNotMinAnyTraj_boolIndices);    % indices of localization that were within one of the acceptance radius but was not appended to any trajectory
                    % left out localizations within the acceptance radii should be
                    % merged with the corresponding trajectories
                    if trajsN_toAppend_accButNotMin>0
                        for idxTraj=newTrajLocalIndices_accButNotMin'
                            trajsInfo(newTrajIndices(idxTraj)).locIndices=lastLocIndicesNewTraj(idxTraj);
                            %trajsInfo(newTrajIndices(idxTraj)).pos=locPos(lastLocIndicesNewTraj(idxTraj),1,:);
                            trajsInfo(newTrajIndices(idxTraj)).mergedTrajIndices=actTrajIndices(any(pairedTrajLocAccDistBoolMat(currentLocSubscripts_accButNotMin,:),1));
                        end
                    end
                    if trajsN_toAppend_refused>0
                        % localizations that were rejected
                        for idxTraj=newTrajLocalIndices_refused'
                            trajsInfo(newTrajIndices(idxTraj)).locIndices=lastLocIndicesNewTraj(idxTraj);
                            %trajsInfo(newTrajIndices(idxTraj)).pos=locPos(lastLocIndicesNewTraj(idxTraj),1,:);
                        end
                    end
                end
                
                %% Overwriting the variables containing the information of the actual trajectories
                % indices of last localizations of trajectories on the next
                % frame
                if ~isempty(newLastLocIndicesActTraj)
                    actTrajLastLocIndices=[...
                        newLastLocIndicesActTraj;...
                        lastLocIndicesNewTraj]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                else
                    actTrajLastLocIndices=[...
                        lastLocIndicesNewTraj]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                end
                
                % trajectory indices on the next frame
                if ~isempty(newTrajIndices)
                    actTrajIndices=[...
                        actTrajIndices(trajAccLoc_boolIndices);...
                        newTrajIndices;...
                        actTrajIndices_noNewLocButNotTerminated];
                else
                    actTrajIndices=[...
                        actTrajIndices(trajAccLoc_boolIndices);...
                        actTrajIndices_noNewLocButNotTerminated];
                end
                
                % update the trajectory positions
                actTrajPos=[...
                    permute(locPos(actTrajLastLocIndices,1,:),[2,1,3]),...
                    actTrajPos_noNewLocButNotTerminated];
                
                % indices of last localizations of trajectories on the next
                % frame
                if ~isempty(actTrajLastLocIndices)
                    actTrajLastLocIndices=[...
                        actTrajLastLocIndices;...
                        actTrajLastLocIndices_noNewLocButNotTerminated]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                else
                    actTrajLastLocIndices=[...
                        actTrajLastLocIndices_noNewLocButNotTerminated]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                end
                
                actTrajNoNewLocState=[...
                    actTrajNoNewLocState(trajAccLoc_boolIndices);...
                    zeros(size(newTrajIndices));...
                    actTrajNoNewLocState(actTrajBoolIndices_noNewLocButNotTerminated)];
                
            end
            toc
            disp('First part of the trajectory fitting is fininshed.')
        end
        
        
        function trajsInfo_seperated = seperateTrajectories(trajsInfo, locPos, locFrames, locProperites, settings)
            tic
            
            trajN=numel(trajsInfo);
            trajIndices=(1:trajN)';
            
            % indices of trajectrories that are simple, e.g. probably consist of only
            % one blinking event
            if exist('trajsInfo', 'var')
                boolIsSimpleTraj=[trajsInfo(:).simpleTrajBool];
            else
                trajsInfo=[];
                boolIsSimpleTraj=[];
            end
            
            % cut out complex trajectories from the structure containing info about all
            % trajectories
            remainedComplexTrajBool=~boolIsSimpleTraj;
            
            % structure containing onformation of trajectories separated from
            % clusters
            cluster_trajsInfo=[];
            
            % going through the clusters of complex trajectories, e.g. use the complex
            % trajectory indices and delete the ones that have been catenated to a
            % cluster (that's why "while" is used)
            remainedComplexTrajIndices=trajIndices(remainedComplexTrajBool);
            while 1<=numel(remainedComplexTrajIndices)
                % the remained clusters are continously deltes, always
                % start with the actual first one
                idxClusterTraj=remainedComplexTrajIndices(1);
                
                % boolean for whether another trajectory should be appended to the
                % actual cluster
                anotherTrajs_toAppendBool=false;
                % localized events' indices of the cluster
                cluster_locIndices=trajsInfo(idxClusterTraj).locIndices;
                % trajectory indices that should be appended to the cluster
                cluster_trajIndices_toMerge=unique(trajsInfo(idxClusterTraj).mergedTrajIndices);
                
                remainedComplexTrajBool(idxClusterTraj)=false;
                
                % while there are trajectories to be appended to the cluster
                while ~isempty(cluster_trajIndices_toMerge)
                    
                    % for stoting the trajectories to be appended to the cluster in the next "while" loop
                    cluster_nextTrajIndices_toMerge=[];
                    % go through the trajectories to be merged to the cluster
                    for idxTraj_toMerge=cluster_trajIndices_toMerge'
                        % append the localization indices to the trajectory, merging
                        % trajectories to the cluster
                        
                        % needed because "horzcat" complains otherwise:
                        trajsInfoComplex_locIndices=trajsInfo(idxTraj_toMerge).locIndices;
                        trajsInfoComplex_mergedTrajIndices=trajsInfo(idxTraj_toMerge).mergedTrajIndices;
                        
                        % concatenae the localizations of the trajectroies
                        % to be merged
                        cluster_locIndices=[...
                            cluster_locIndices;...
                            trajsInfoComplex_locIndices];
                        
                        % find new trajectories to be merged
                        cluster_nextTrajIndices_toMerge=[...
                            cluster_nextTrajIndices_toMerge;...
                            trajsInfoComplex_mergedTrajIndices];
                        
                        % "0" marks the the complex trajectories that already have been
                        % catenated to any of the clusters so they can not start new
                        % clusters
                        remainedComplexTrajBool(idxTraj_toMerge)=false;
                        
                    end
                    
                    % pass the trajectories to be appended to the cluster in the next "while" loop
                    % as the merging indices are "two-way", both
                    % trajectories cointain the indices of merging, the
                    % already merged trajecttories has to be excluded
                    cluster_trajIndices_toMerge=cluster_nextTrajIndices_toMerge(...
                        remainedComplexTrajBool(unique(cluster_nextTrajIndices_toMerge)));
                end
                
                % keeping only the unique localizations, needed because the same
                % localization can belong to several comlex trajectories
                cluster_locIndices=unique(cluster_locIndices);
                
                
                % create array "number of localizations"x1x3 dimensional double array for the localization positions
                cluster_locPos=locPos(cluster_locIndices,1,1:3);
                
                % create a vector for the frame indices of the localizations
                cluster_locFrames=locFrames(cluster_locIndices);
                
                fieldNames=fieldnames(locProperites);
                for idxField=1:numel(fieldNames)
                    cluster_locProperites.(fieldNames{idxField})=locProperites.(fieldNames{idxField})(cluster_locIndices);
                end
                
                
                % trajectory fitting for seperating the complex trajectories in the cluster
                actCluster_trajsInfo=rainSTORM_trajectoryFitting.seperateTrajsFromClustersFunc(cluster_locPos, settings.ZPrecConst, cluster_locFrames, cluster_locIndices, cluster_locProperites);
                
                
                cluster_trajsInfo=[cluster_trajsInfo;...
                    actCluster_trajsInfo];
                
                % remained cluster complex trajectory indices that can start a new
                % cluster
                remainedComplexTrajIndices=trajIndices(remainedComplexTrajBool);
            end
            
            toc
            disp('Findind the clusters of complex trajectories is finished.')
            
            %%
            % indices of trajectrories that are cut out of clusters of complex
            % trajectories, e.g. probably consist of only one blinking event
            if ~isempty(cluster_trajsInfo)
                boolIsTrajRefused=[cluster_trajsInfo(:).trajRefusedBool];
                boolIsTrajClosedButGoOnBool=[cluster_trajsInfo(:).trajClosedButGoOnBool];
                boolIsAcceptedComplexTraj=~(boolIsTrajRefused | boolIsTrajClosedButGoOnBool);
            else
                boolIsAcceptedComplexTraj=[];
            end
            
            % cut out trajectories from the structure containing info about all
            % trajectories belonging to clusters of complex trajectories
            cluster_trajsInfo_accepted=cluster_trajsInfo(boolIsAcceptedComplexTraj);
            complexTrajN=numel(cluster_trajsInfo_accepted);
            
            % keep only the single trajectories from the original info
            % struct
            trajsInfo=trajsInfo(boolIsSimpleTraj);
            simpleTrajN=numel(trajsInfo);
            
            % concatenate localization indices of the simple and of the
            % accepted seperated complex rajectories
            for idxTraj=simpleTrajN+complexTrajN:-1:simpleTrajN+1
                trajsInfo_seperated(idxTraj).locIndices=cluster_trajsInfo_accepted(idxTraj-simpleTrajN).locIndices;
                trajsInfo_seperated(idxTraj).isSimple=false;
            end
            for idxTraj=simpleTrajN:-1:1
                trajsInfo_seperated(idxTraj).locIndices=trajsInfo(idxTraj).locIndices;
                trajsInfo_seperated(idxTraj).isSimple=true;
            end
            
        end
        
        
        function trajsInfo=seperateTrajsFromClustersFunc(locPos, ZPrecConst, locFrames, locIndices, cluster_locProperites)
            % This function separates distinc trajectories from the given
            % cluster.
            % Currently goes frame by frame (only frames that span the cluster)
            % and assay which localization (on the current frame) fits to
            % which trajectory (actual state) the best by calculating all
            % localization-trajectory combination and choosing the combination
            % with the highest number of accepted localization-trajectory pairs
            % with the minimal mean square distance. This methodshould be
            % revised in the future.
            
            
            % constants for determining whether the localization can be
            % concatenated to the trajectory
            normDistSquare_precConst=2;
            relDevSquare_sigXThreshold=0.04;
            relDevSquare_sigYThreshold=0.04;
            relDevSquare_GaussianPeakThreshold=0.01;
            relDevSquare_sumSignalThreshold=0.01;
            
            % passing the properties of the localizations of the cluster
            loc_X_STD=cluster_locProperites.X_STD;
            loc_Y_STD=cluster_locProperites.Y_STD;
            locSigX=cluster_locProperites.SigX;
            locSigY=cluster_locProperites.SigY;
            locGaussianPeak=cluster_locProperites.GaussianPeak;
            locSumSignal=cluster_locProperites.SumSignal;
            
            % template for the "trajsInfo" structure for starting new
            % trajectories that should not be catenated to another one
            trajsInfoBrandNewTemplate.locIndices=[];
            trajsInfoBrandNewTemplate.trajRefusedBool=false;
            trajsInfoBrandNewTemplate.trajClosedBool=false;
            trajsInfoBrandNewTemplate.trajClosedButGoOnBool=false;
            %trajsInfoBrandNewTemplate.pos=[];
            
            locLocalIndices=(1:length(locIndices))';
            
            firstMeaningfulFrameBool=true;
            firstMeaningfulFrameIdx=locFrames(1);
            while firstMeaningfulFrameBool
                % trajectories on the first not empty frame:
                actTrajLastLocIndices(:,1)=locLocalIndices(locFrames==firstMeaningfulFrameIdx);
                % lets define the trajectory position this way:
                actTrajPos=permute(locPos(actTrajLastLocIndices,1,:),[2,1,3]); % initial trajectory positions, 1x"number of localizations"x2 dimensional double array, start it with the localization positions on the first frame
                % initialize the "trajInfo" structure with the localisations on the
                % not empty first frame
                actTrajIndices=(1:size(actTrajPos,2))';
                
                % boolean vector containing information about which trajectry
                % is new (active and contain only one localization)
                trajSeveralElementsBoolVect=false(size(actTrajIndices));
                
                if numel(actTrajIndices)>0
                    trajsInfo=repmat(trajsInfoBrandNewTemplate, size(actTrajIndices));
                    for idxTraj=1:numel(actTrajIndices)
                        trajsInfo(idxTraj).locIndices=actTrajIndices(idxTraj);  % stands because of the first frame
                        %trajsInfo(idxTraj).pos=actTrajPos(1,idxTraj,:);         % stands because of the first frame
                    end
                    
                    firstMeaningfulFrameBool=false;
                else
                    firstMeaningfulFrameIdx=firstMeaningfulFrameIdx+1;
                end
            end
            
            
            % trajectories on the subsequent frames:
            for idxFrame=firstMeaningfulFrameIdx+1:locFrames(end)
                %% The blinking events on the actual frame
                % positions of localized events on the actual frame
                currentLocPos=[];
                %currentLocPos(:,1,:)=locPos(locFrames==idxFrame,1,:);
                currentLocPos=locPos(locFrames==idxFrame,1,1:3);
                % indices of localizations on the current frame
                
                currentLocLocalIndices=[];
                currentLocLocalIndices(:,1)=locLocalIndices(locFrames==idxFrame);
                
                % current number of localizations and active trajectories
                currentLocN=size(currentLocPos,1);
                
                if ~isempty(actTrajIndices)
                    %% Determining which localization might belong to which trajectory
                    
                    % boolean matrix containing information about which trajectry
                    % is new (active and contain only one localization) in the
                    % localization-trajectory matrix
                    trajSeveralElementsBoolMat=repmat(trajSeveralElementsBoolVect', [currentLocN,1]);
                    
                    % distance
                    normDistSquareTrajLoc=rainSTORM_trajectoryFitting.trajLocDistPrecFunc(actTrajPos, currentLocPos, loc_Y_STD, loc_X_STD, ZPrecConst, currentLocLocalIndices, actTrajLastLocIndices, normDistSquare_precConst);
                    % boolean matrix contaning information about whether the blinking event were close enough to one of the trajectories
                    pairedTrajLocAccDistBoolMat=normDistSquareTrajLoc<=1;
                    
                    % width of PSF (sigma) in the "x" direction
                    [~, relDevSquareSigXMat]=rainSTORM_trajectoryFitting.relDevLinSquareQuantFunc(locSigX, actTrajLastLocIndices, currentLocLocalIndices);
                    relDevSquareSigXBoolMat=relDevSquareSigXMat<=relDevSquare_sigXThreshold;
                    
                    % width of PSF (sigma) in the "y" direction
                    [~, relDevSquareSigYMat]=rainSTORM_trajectoryFitting.relDevLinSquareQuantFunc(locSigY, actTrajLastLocIndices, currentLocLocalIndices);
                    relDevSquareSigYBoolMat=relDevSquareSigYMat<=relDevSquare_sigYThreshold;
                    
                    % peak value of PSF
                    [~, relDevSquareGaussianPeakMat]=rainSTORM_trajectoryFitting.relDevLinSquareQuantFunc(locGaussianPeak, actTrajLastLocIndices, currentLocLocalIndices);
                    relDevSquareGaussianPeakBoolMat=relDevSquareGaussianPeakMat<=relDevSquare_GaussianPeakThreshold;
                    
                    % sum signal under the PSF
                    [relDevLinSumSignalMat, relDevSquareSumSignalMat]=rainSTORM_trajectoryFitting.relDevLinSquareQuantFunc(locSumSignal, actTrajLastLocIndices, currentLocLocalIndices);
                    relDevSquareSumSignalBoolMat=relDevSquareSumSignalMat<=relDevSquare_sumSignalThreshold;
                    
                    relDevLinSumSignalBoolMat=relDevLinSumSignalMat>0;
                    % boolean matrix, true if the localization is not accepted
                    % based on its sum signal and has a lower sum signal than the
                    % last element in the trajectory
                    isSumSignalLowerBoolMat=relDevSquareSumSignalBoolMat | relDevLinSumSignalBoolMat;
                    % boolean matrix, true if the localization is not accepted
                    % based on its sum signal and has a higher sum signal than the
                    % last element in the trajectory and the trajectory already ha
                    % more than one element
                    isSumSignalHigherBoolMat_severalElements=(~relDevSquareSumSignalBoolMat & relDevLinSumSignalBoolMat) & trajSeveralElementsBoolMat;
                    
                    % boolean matrix for the accepted localization-trajectory pairs
                    allAccBoolMat=pairedTrajLocAccDistBoolMat & relDevSquareSigXBoolMat & relDevSquareSigYBoolMat & relDevSquareGaussianPeakBoolMat & relDevSquareSumSignalBoolMat;
                    
                    % number of non-zero element in the accepted
                    % localization-trajectory boolean matrix along the rows and
                    % columns
                    allAccBoolMat_nonZeroNCol=sum(allAccBoolMat,1);
                    allAccBoolMat_nonZeroNRow=sum(allAccBoolMat,2);
                    
                    % boolean matrix containing only those localization-trajectory
                    % pairs in which one localization can belong to one trajectory
                    % and vica-versa
                    allAccBoolMatOneComb=allAccBoolMat;
                    allAccBoolMatOneComb(allAccBoolMat_nonZeroNRow~=1, allAccBoolMat_nonZeroNCol~=1)=false;
                    
                    % boolean matrix containing only those localization-trajectory
                    % pairs in which one localization can belong to several
                    % trajectory and vica-versa
                    allAccBoolMatSeveralComb=allAccBoolMat;
                    allAccBoolMatSeveralComb(allAccBoolMatOneComb)=false;
                    
                    % localization and trajectory indices in the parameters
                    % accepted boolean matrix, one localization per trajectory and
                    % vica-versa
                    [locAllAccBoolMatOneComb_subscripts,trajAllAccBoolMatOneComb_subscripts]=ind2sub(size(allAccBoolMatOneComb), find(allAccBoolMatOneComb));
                    if isempty(locAllAccBoolMatOneComb_subscripts)
                        locAllAccBoolMatOneComb_subscripts=[];
                        trajAllAccBoolMatOneComb_subscripts=[];
                    end
                    
                    % localization and trajectory indices in the parameters
                    % accepted boolean matrix, several localizations per trajectory
                    % or the other way around
                    [locAllAccBoolMatSeveralComb_subscripts,trajAllAccBoolMatSeveralComb_subscripts]=ind2sub(size(allAccBoolMatSeveralComb), find(allAccBoolMatSeveralComb));
                    locAllAccBoolMatSeveralComb_uniqueSubscripts=unique(locAllAccBoolMatSeveralComb_subscripts);
                    trajAllAccBoolMatSeveralComb_uniqueSubscripts=unique(trajAllAccBoolMatSeveralComb_subscripts);
                    
                    if currentLocN==1 % MATLAB REALLY SHOULD RAPE ITSELF IN THE ASS
                        locAllAccBoolMatOneComb_subscripts=locAllAccBoolMatOneComb_subscripts';
                        trajAllAccBoolMatOneComb_subscripts=trajAllAccBoolMatOneComb_subscripts';
                        
                        locAllAccBoolMatSeveralComb_uniqueSubscripts=locAllAccBoolMatSeveralComb_uniqueSubscripts';
                        trajAllAccBoolMatSeveralComb_uniqueSubscripts=trajAllAccBoolMatSeveralComb_uniqueSubscripts';
                    end
                    
                    % seperating the dubious localizations into distinct
                    % trajectories in the actual cluster
                    if ~isempty(locAllAccBoolMatSeveralComb_uniqueSubscripts) && ~isempty(trajAllAccBoolMatSeveralComb_uniqueSubscripts)
                        [locAllAccBoolMatSevCombSep_subscripts,trajAllAccBoolMatSevCombSep_subscripts]=rainSTORM_trajectoryFitting.seperateClusterTrajsFunc(locAllAccBoolMatSeveralComb_uniqueSubscripts,trajAllAccBoolMatSeveralComb_uniqueSubscripts, normDistSquareTrajLoc, allAccBoolMatSeveralComb);
                    else
                        locAllAccBoolMatSevCombSep_subscripts=[];
                        trajAllAccBoolMatSevCombSep_subscripts=[];
                    end
                    
                    % vector containing the subscript indices of the seperated
                    % localizations and the trajectories belonging to them
                    locSep_subscripts=[...
                        locAllAccBoolMatOneComb_subscripts;
                        locAllAccBoolMatSevCombSep_subscripts];
                    trajSep_subscripts=[...
                        trajAllAccBoolMatOneComb_subscripts;
                        trajAllAccBoolMatSevCombSep_subscripts];
                    
                    if ~isempty(trajSep_subscripts)
                        %% Loop through the trajectories
                        for trajTempIdx=1:numel(trajSep_subscripts)
                            trajLocalIdx=trajSep_subscripts(trajTempIdx);
                            locLocalIdx=locSep_subscripts(trajTempIdx);
                            
                            trajsInfo_structArrayIdx=actTrajIndices(trajLocalIdx);
                            
                            % append the nearest localizations within the acceptance distance to the trajectories
                            trajsInfo(trajsInfo_structArrayIdx).locIndices=[...
                                trajsInfo(trajsInfo_structArrayIdx).locIndices;...
                                locIndices(currentLocLocalIndices(locSep_subscripts(trajSep_subscripts==trajLocalIdx)))];
                            
                            % update the trajectory position
                            %trajsInfo(trajsInfo_structArrayIdx).pos=actTrajPos(1,trajLocalIdx,:);
                            
                            % if the trajectory has been marked as closed but
                            % continues
                            if trajsInfo(trajsInfo_structArrayIdx).trajClosedBool
                                trajsInfo(trajsInfo_structArrayIdx).trajClosedButGoOnBool=true;
                            end
                            % if the sum signal rises in the trajectory even after
                            % the first element
                            if isSumSignalHigherBoolMat_severalElements(locLocalIdx, trajLocalIdx)
                                trajsInfo(trajsInfo_structArrayIdx).trajRefusedBool=true;
                                % if the sum signal decreases in the trajectory
                            elseif isSumSignalLowerBoolMat(locLocalIdx, trajLocalIdx)
                                % the trajectory should be marked as closed (at least should not continue)
                                trajsInfo(trajsInfo_structArrayIdx).trajClosedBool=true;
                            end
                            
                        end
                    end
                else
                    pairedTrajLocAccDistBoolMat=false(currentLocN,1);
                end
                
                %% Finding the refused or the left out localizations
                
                % boolean indices of localizations that was not catenated to any
                % trajectory
                locRefusedTraj_boolIndices=~any(pairedTrajLocAccDistBoolMat, 2);
                
                % number of localizations that was not catenated to any trajectory
                trajsN_toAppend_refused=nnz(locRefusedTraj_boolIndices);            % refused
                trajsN_toAppend=trajsN_toAppend_refused;   % overall
                
                % first ( and  last) member (localization) indices of the new trajectories from the localizations that were
                % not catenated to any trajectory
                newTrajFirstLocIndices_toAppend_refused=currentLocLocalIndices(locRefusedTraj_boolIndices);
                
                %% Preparing the preparation for the new trajectories for the next frame
                % creating indices for trajectories to be created for the left
                % out or rejected localizations
                newTrajLocalIndices_refused=(1:trajsN_toAppend)';
                newTrajLocalIndices=...
                    newTrajLocalIndices_refused;
                % indices of these trajectories
                newTrajIndices=newTrajLocalIndices+length(trajsInfo);
                
                % last member (localization) indices of the old trajectories from the localizations that were
                % catenated to any of the trajectories
                actTrajNewLastLocIndices=currentLocLocalIndices(locSep_subscripts);
                % first ( and  last) member (localization) indices of the new trajectories from the localizations that were
                % not catenated to any trajectory
                newTrajLastLocIndices=...
                    newTrajFirstLocIndices_toAppend_refused; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                % indices of last localizations of trajectories on the next
                % frame
                actTrajLastLocIndices=[...
                    actTrajNewLastLocIndices;...
                    newTrajLastLocIndices]; % trajectory positions on the next frame, 1x"number of localizations"x2 dimensional double array
                
                %% Appending the newly created trajectories for the "trajsInfo" structure
                % extending the "trajsInfo" structure
                if trajsN_toAppend_refused>0
                    trajsInfo=[...
                        trajsInfo;...
                        repmat(trajsInfoBrandNewTemplate, [trajsN_toAppend_refused,1])];
                    
                    % localizations that were rejected
                    for idxTraj=newTrajLocalIndices_refused'
                        trajsInfo(newTrajIndices(idxTraj)).locIndices=newTrajLastLocIndices(idxTraj);
                        %trajsInfo(newTrajIndices(idxTraj)).pos=locPos(newTrajLastLocIndices(idxTraj),1,:);
                    end
                end
                
                %% Overwriting the variables containing the information of the actual trajectories
                % trajectory indices on the next frame
                if ~isempty(newTrajIndices)
                    actTrajIndices=[...
                        actTrajIndices(trajSep_subscripts);...
                        newTrajIndices];
                else
                    actTrajIndices=...
                        actTrajIndices(trajSep_subscripts);
                end
                
                % boolean vector containing information about which trajectry
                % is new (active and contain only one localization)
                newTrajsN=numel(newTrajIndices);
                trajSeveralElementsBoolVect=true(size(actTrajIndices));
                trajSeveralElementsBoolVect(end-newTrajsN+1:end)=false;
                
                % update the trajectory positions
                actTrajPos=[];
                actTrajPos=permute(locPos(actTrajLastLocIndices,1,:),[2,1,3]);
            end
            
        end
        
        
        function normDistSquareTrajLoc=trajLocDistPrecFunc(actTrajPos, currentLocPos, loc_Y_STD, loc_X_STD, ZPrecConst, currentLocIndices, actTrajLastLocIndices, probConst)
            % Function for calculating the normalized distances between the
            % localizations taking into account their localization
            % precisions.
            
            currentLocPrecision(:,1,3)=ZPrecConst*(loc_X_STD(currentLocIndices)+loc_Y_STD(currentLocIndices))*0.5;
            currentLocPrecision(:,1,2)=loc_Y_STD(currentLocIndices);
            currentLocPrecision(:,1,1)=loc_X_STD(currentLocIndices);
            actTrajPrecision(1,:,3)=ZPrecConst*(loc_X_STD(actTrajLastLocIndices)+loc_Y_STD(actTrajLastLocIndices))*0.5;
            
            actTrajPrecision(1,:,2)=loc_Y_STD(actTrajLastLocIndices);
            actTrajPrecision(1,:,1)=loc_X_STD(actTrajLastLocIndices);
            
            distMat=repmat(actTrajPos,size(currentLocPos,1),1,1)-repmat(currentLocPos,1,size(actTrajPos,2),1);
            precisionMat=repmat(actTrajPrecision,size(currentLocPrecision,1),1,1)+repmat(currentLocPrecision,1,size(actTrajPrecision,2),1);
            ellipAxesMat=probConst*precisionMat;
            
            % matrix containing the squares of distances between the
            % trajectories' last element's positions and localized
            % positions
            normDistSquareTrajLoc=sum(((distMat)./ellipAxesMat).^2,3);
            
        end
        
        
        function [relDevLinQuantMat, relDevSquareQuantMat]=relDevLinSquareQuantFunc(quantity, actTrajLastLocIndices, currentLocIndices)
            % function for calculation the rel�tive deviation of one quantity
            % between that of the trajectories' last elements and the
            % localizations
            
            actQuantLocVect(:,1)=quantity(currentLocIndices);
            actQuantTrajVect(1,:)=quantity(actTrajLastLocIndices);
            
            actQuantLocMat=repmat(actQuantLocVect,1,size(actQuantTrajVect,2),1);
            actQuantTrajMat=repmat(actQuantTrajVect,size(actQuantLocVect,1),1,1);
            
            relDevLinQuantMat=(actQuantLocMat-actQuantTrajMat)./actQuantTrajMat;
            relDevSquareQuantMat=relDevLinQuantMat.^2;
            
        end
        
        
        function [linAllCombVect, reshapeDims]=trajPosAllCombFunc(locIdxVect, trajIdxVect, locIdxN)
            %This function calculates all the possible pairs of localization
            %and trajectory indices and creates linearized indices from them
            
            locIdxN_acc=numel(locIdxVect);      % number of localization indices
            trajIdxN_acc=numel(trajIdxVect);      % number of trajectory indices
            
            
            minN=min(trajIdxN_acc,locIdxN_acc);     % number of localization-trajectory pairs in one combination
            maxN=max(trajIdxN_acc,locIdxN_acc);
            combN=factorial(maxN)/factorial(maxN-minN);    % number of all combinations of localization-trajetory pairs
            
            % if there are less trajectory than localizations
            if trajIdxN_acc<=locIdxN_acc
                % vector for trajectory indixes, for all combination
                trajIdxRepeated=repmat(reshape(trajIdxVect, [1, trajIdxN_acc]), [1,combN]);
                
                % all possible combinations of localization indices
                locIdxAllComb=nchoosek(locIdxVect, minN);
                
                % number of permutations in one combination of localization
                % of localization indices
                permN=uint16(minN*factorial(minN));
                
                % go through the combinations
                for locCombIdx=1:size(locIdxAllComb,1)
                    % all permutations of the actual combination
                    locIdxAllCombPerm=perms(locIdxAllComb(locCombIdx,:));
                    locIdxRepeated((locCombIdx-1)*permN+1:locCombIdx*permN)=reshape(locIdxAllCombPerm', [1,permN]);
                end
            else
                % vector for trajectory indices, for all combination, row
                % vector
                locIdxRepeated=repmat(reshape(locIdxVect, [1, locIdxN_acc]), [1,combN]);
                
                % all possible combinations of localization indices
                trajIdxAllComb=nchoosek(trajIdxVect, minN);
                
                % number of permutations in one combination of localization
                % of localization indices
                permN=uint16(minN*factorial(minN));
                
                % go through the combinations
                for trajCombIdx=1:size(trajIdxAllComb,1)
                    % all permutations of the actual combination
                    trajIdxAllCombPerm=perms(trajIdxAllComb(trajCombIdx,:));
                    trajIdxRepeated((trajCombIdx-1)*permN+1:trajCombIdx*permN)=reshape(trajIdxAllCombPerm', [1,permN]);
                end
            end
            
            % create linearized indices from the localization-trajectory index
            % pairs
            linAllCombVect=(trajIdxRepeated-1)*locIdxN+locIdxRepeated;
            
            % dimenison for reshaping the vector of localization-trajectory
            % linearized indices to seperate the different combinations
            reshapeDims=[minN, combN];
            
        end
        
        
        function [locIdxVectSep,trajIdxVectSep]=seperateClusterTrajsFunc(locIdxVect, trajIdxVect, normDistSquareTrajLoc, allAccBoolMatSeveralComb)
            % This function seperates the remnant trajectories in the cluster
            
            locIdxN=size(normDistSquareTrajLoc,1);      % number of localization indices
            
            % get the linearized indices (in the boolean matrix) for all
            % localization-trajectory combination
            [linAllCombVect, reshapeDims]=rainSTORM_trajectoryFitting.trajPosAllCombFunc(locIdxVect, trajIdxVect, locIdxN);
            
            % boolean containing which localization-trajectory were accepted
            % previously
            trajLocCombPairBool=reshape(allAccBoolMatSeveralComb(linAllCombVect),reshapeDims);
            % number of accepted localization-trajectory pairs in each
            % combination
            trajLocCombPairN=sum(trajLocCombPairBool,1);
            % distances of (in accepted pairs) localizations and the
            % trajectories
            trajLocCombPairDist=reshape(normDistSquareTrajLoc(linAllCombVect),reshapeDims).*trajLocCombPairBool;
            % mean value of these distances
            trajLocCombPairDistQuant=sum(trajLocCombPairDist,1)./trajLocCombPairN;
            % ignore those combinations where the number of accepted
            % localization-trajectory pairs is not maximal
            trajLocCombPairDistQuant(trajLocCombPairN~=max(trajLocCombPairN))=NaN;
            % choose the combination where the mean distance of
            % localization-trajectory pairs is minimal
            [~,choosenComb]=min(trajLocCombPairDistQuant);
            % reshaping the linear indices of the (boolean matrix), so that
            % they are sorted by index of the combinations along the second
            % dimension
            comb_linIndices=reshape(linAllCombVect,reshapeDims);
            % linear indices (in the boolean matrix) of the choosen
            % localization-trajectory pairs
            choosenTrajLocPair_linIndices=comb_linIndices(:,choosenComb);
            
            % geting back the subscripts of trajectories and localizations from
            % the linear indices
            locIdxVectSep=mod(choosenTrajLocPair_linIndices, locIdxN);
            locIdxVectSep(locIdxVectSep==0)=locIdxN;
            trajIdxVectSep=floor((choosenTrajLocPair_linIndices-1)/locIdxN)+1;
            
        end
        
    end
end

