function SupResParams = rainSTORM_fitLocGF_2D_linearBg(frameIdx, myFrame,myPixels,settings)
    % 2D Gausian Fit with Local Background Subtraction. Background is assumed to be
    % a linear function (slope)
    % Fit [x0,C,sig,Dx,Dy,E] to make
    % f(x,y)=C*exp( -((x-x0)^2+(y-y0)^2)/(2*sig^2) ) + Dx*x + Dy*y + E
    % Reject fits with far-out x0, sigX, or residual.
    
    persistent initX0 initSig tol maxIts allowSig allowX rad
    
    initX0 = 0;
    initSig = settings.initSig;
    tol = settings.tol;
    maxIts = settings.maxIts;
    allowSig = settings.allowSig;
    allowX = settings.allowX;
    rad = settings.rad; % Radius of PSF ROI [pix]
    index = 0;
    
    N = 2*rad+1;
    xx = repmat( (-rad:rad)', [1 N] );      % x-positions (rows) (pixel widths) as column vector
    yy = repmat( (-rad:rad),  [N 1] );      % y-positions (cols) (pixel widths) as column vector
    
    xx = xx(:);             % Using 1D representation during optimization
    yy = yy(:);
    ONES = ones(size(xx));
    
    SupResParams = [];
    for lpPx = 1:size(myPixels,1) % For each candidate
        myRow = myPixels(lpPx,1);
        myCol = myPixels(lpPx,2);
        myROI = double(myFrame(myRow-rad:myRow+rad,myCol-rad:myCol+rad));
        
        myROI_min = min(myROI(:));
%       myROI_bg = mean( [myROI(:,1); myROI(:,end); myROI(1,2:end-1)'; myROI(end,2:end-1)' ] );
        % Estimate ROI background on the side-pixels -> slow version
        myROI_bg = (sum(myROI(:,1))+ sum(myROI(:,end))+ sum(myROI(1,2:end-1))+sum(myROI(end,2:end-1)))/...
            (4*N-4); % Estimate ROI background on the side-pixels -> fast version
                     % Notice the two 2:end-1, thats why the -4
        
        myROI = myROI - myROI_bg;  % Square region to fit. Subtract background.
        flagFit = false;   % Begin by noting the centre-position is not fitted
        
        % Initial parameters
        x0 = initX0;
        y0 = initX0;
        Dx = 0;
        Dy = 0;
        E = 0;
        sig = initSig;
        C  = myROI(rad+1, rad+1); % Guess height of f(x). Centre value is a good guess.
        
        myROI = myROI(:);
        
        fofX = C*exp(-((xx-x0).^2+(yy-y0).^2)/(2*sig^2)); % Initial guess of f(x,y)
        Beta = myROI - fofX - Dx*xx - Dy*yy - E; % Change needed in f(x,y)
        
        for lpLSF = 1:maxIts
            D1 = fofX/C;                                     % dC
            D2 = fofX.*(xx-x0)/sig^2;                       % dx0
            D3 = fofX.*(yy-y0)/sig^2;                       % dy0
            D4 = fofX.*((xx-x0).^2+(yy-y0).^2)/sig^3;       % dsig
            
            %     dC,   dx0,  dy0, dsig, dDx   dDy    dE
            A = [D1 D2 D3 D4 xx yy ONES];  % Jacobian
            
            b = A'*Beta;
            a = A'*A;
            RC = rcond(a);
            if isnan( RC ) || RC < 1e-12; break; end
            dL= a\b;
            
            C = C + dL(1);
            x0 = x0 + dL(2);
            y0 = y0 + dL(3);
            sig = sig + dL(4);
            Dx = Dx + dL(5);
            Dy = Dy + dL(6);
            E = E + dL(7);
            
            fofX = C*exp(-((xx-x0).^2+(yy-y0).^2)/(2*sig^2));
            Beta = myROI - fofX - Dx*xx - Dy*yy - E;
            
            if(abs(x0)>allowX || abs(y0)>allowX || (sig < allowSig(1)) || (sig > allowSig(2)) )
                break; % Stop iterating if solution drifts too far
            end
        end
        
        % Judge the fit. Accept if residue is a small
        myROI = myROI - (Dx*xx + Dy*yy + E);            % Removing bg, we need only the Gaussian
        
        myROI = reshape( myROI, [N N] );
        fofX = reshape( fofX, [N N] );
        
        BetaRows = sum(myROI,1)-sum(fofX,1);
        BetaCols = sum(myROI,2)-sum(fofX,2);
        residueRows = sum(BetaRows.^2)/sum( sum(myROI,1).^2 );
        residueCols = sum(BetaCols.^2)/sum( sum(myROI,2).^2 );
        residue = sum(Beta.^2)/sum(myROI(:).^2);
        
        if (residueRows<tol && residueCols<tol && abs(x0)<allowX  && abs(y0)<allowX && sig>allowSig(1) && sig<allowSig(2))
            fitRowPos = double(myRow)+x0-0.5; % Note (-0.5) for image registration
            fitColPos = double(myCol)+y0-0.5;
            flagFit = true; % Flag the row-direction fit as acceptable
        end
        
        if(flagFit )     % Accept iff Row and Col fits are OK
            index = index + 1;
            SupResParams(index).frame_idx = uint32(frameIdx);
            SupResParams(index).x_coord = single(fitRowPos);
            SupResParams(index).y_coord = single(fitColPos);
            SupResParams(index).sig = single(sig);  % X-width (sigma, rows, fitted) of this Gaussian
            SupResParams(index).background = single(E+myROI_bg); % Background for each ROI
            SupResParams(index).backgroundSTD = single.empty; % standard deviation of the background signal (noise level)
            SupResParams(index).res = single(residue); % Mean critical tol for fit
            SupResParams(index).sum_signal = single(2*pi*C*sig*sig); %volume under the gaussian in counts (sum of signal
            
%             SupResParams(index).z_coord = single(0.);
%             SupResParams(index).I = single(C); % fitted maximum %myPixels(lpPx,3); % Averaged magnitude of this signal
%             SupResParams(index).res_Row = single(residueRows);
%             SupResParams(index).res_Col = single(residueCols);
%             SupResParams(index).roi_min = uint16(myROI_min);
        end
        % No halting condition for multiple fitting failures are applied in this
    end  % Loop to the next local maximum
end
