% rainSTORM_recon_JH
%
% 22 May 2013 Eric Rees
%
% Under Development
%
% FUNCTION
%   Plot a Jittered Histogram visualisation of Localisation Microscopy data
%   Make use of principles from [Silverman] and [Krizek] refs
%
% MOTIVATION
%   Need an optimised method for plotting fluorophore density
%   Want to minimise user-supplied parameters (e.g. pixel size)
%   Consider using Simple Histogram as a Pilot Estimate
%   Epanechnikov kernel might be better (Silverman)
%
% NOTES
%   A Jittered Histogram ~ a digitised Adaptive Kernel Density Estimate
%   The tweakable parameters (jhArbitFactor, jhAlpha) are partly able to
%   compensate for the imperfect evaluation of the Thompson Precision
%   which is used as the baseline width for jittering
%
% FURTHER READING
%   D. Baddeley, (2010) doi:10.1017/S143192760999122X
%   P. Krizek, (2011) Vol. 19,  No. 4, OPTICS EXPRESS 3226
%   B. Silverman, Density Estimation Theory (CRC Press) pp. 100-110
%   E. Rees, (2012) http://www.optnano.com/content/1/1/12
%

function [imageMatrix, pixelCenters, jhPixelWidth] = rainSTORM_recon_JH(pointillisticData, settings)

reviewedPosits = double([pointillisticData.x_coord, pointillisticData.y_coord]);

if pointillisticData_fieldNameManagement.isMemberField(pointillisticData, 'sig')
    reviewedDeltaX = double([pointillisticData.std, pointillisticData.std]);
elseif pointillisticData_fieldNameManagement.isMemberField(pointillisticData, {'sig_x', 'sig_y'})
    reviewedDeltaX = double([pointillisticData.std_x, pointillisticData.std_x]);
else
    error('The localization data given for jittered histogram creation does not contain the neccessary "Gaussian sigma" fields.')
end

% 1. INPUTS

jhAlpha       = settings.jhAlpha;
jhArbitFactor = settings.jhArbitFactor;
jhFinesse     = settings.jhFinesse;
jhLinMag_max = settings.jhLinMag_max;
jhNJitters    = settings.jhNJitters;
jhPilotCoarseness = settings.jhPilotCoarseness;
flagJHBlur = settings.flagJHBlur;
cameraPixelSize_nm     = settings.cameraPixelSize_nm;
stackSize = settings.stackSize;
frameSize = [stackSize(1), stackSize(2)];

% % % jhAlpha       = 0.5;   % See [Silverman, page 101 onwards]
% % % jhArbitFactor = 0.85;  % For tweaking the jittering (kernel) width. Try 1
% % % jhFinesse     = 0.5;   % Tweaks the pixel width of reconstruction. Try 1
% % % jhNJitters    = 60;    % Number of Jitters per localisation
% % % jhPilotCoarseness = 1; % To coarsen pilot estimate
% % % flagJHBlur = 1;        % To blur pilot estimate by convolution w/ ones(3)
% % % pixelWidth     = params.rawdata_mgr.myImInfo.pixelWidth;
% % % sizeOfCCDFrame = params.rawdata_mgr.myImInfo.frame_size;


% 2. PROCESS
%    Determine reconstruction pixel size and smoothing,
%    Calculate jittered histogram (JH) reconstruction
% DETERMINE RECONSTRUCTION PIXEL WIDTH:
% Try the nearest power of 2 nm below mean Precision, * jhFinesse to tweak
% (Issue: this will often be 16 or 32 nm in practice... Any problems?)
jhPixelWidthPow = floor(log2(jhFinesse*mean(reviewedDeltaX(:))));
jhPixelWidth    = 2^jhPixelWidthPow;

jhLinMag = cameraPixelSize_nm/jhPixelWidth; % CCD pixel width / required width
jhLinMag = min(jhLinMag,jhLinMag_max); % This limits the maximum image size.
jhLinMagPilot = jhLinMag / jhPilotCoarseness;

% Make a Simple Histogram image to get a pilot estimate of density
settings_recon = [];
settings_recon.linMag = jhLinMagPilot;
settings_recon.stackSize = stackSize;
settings_recon.cameraPixelSize_nm = cameraPixelSize_nm;
jhPilotIm = double(rainSTORM_recon_counts(pointillisticData,settings_recon));
if (flagJHBlur)
    jhPilotIm = conv2(jhPilotIm, ones(3), 'same'); % Blur the pilot estim
end

% Pre-allocate memory for adaptive kernel width estimation
% jhParams means "jittered histogram Parameters"
jhParams  = zeros(size(reviewedPosits,1),4); % N,f_pilot, Lambda, jhSig

% Find number of nearby localisations for each reviewedPosit
% Column 1 of jhParams is for "N"
for lpLoc = 1: size(reviewedPosits,1)
    myRow = ceil( reviewedPosits(lpLoc,1) * jhLinMagPilot );
    myCol = ceil( reviewedPosits(lpLoc,2) * jhLinMagPilot );
    jhParams(lpLoc,1) = jhPilotIm(myRow,myCol);
end

% Find pilot probability distribution
numberAcceptedLocalisations = size(reviewedPosits,1);  %
jhPilotF = jhParams(:,1)./numberAcceptedLocalisations; % Pilot prob density
jhParams(:,2) = jhPilotF;
% jhG = geomean(jhPilotF);% this was removed to decrease toolbox dependency
jhG = exp(sum(log(jhPilotF))./length(jhPilotF));

% Calculate Local Bandwidth Factors:
jhLambda = (jhPilotF/jhG).^-jhAlpha; % varies on 0.4 to 1.4 in test run
% jhLambda(jhLambda > 2) = 2; % Doesn't seem helpful
jhParams(:,3) = jhLambda;

% Calculate Adaptive Kernel Widths
% These can act as std dev's for a (digitised) Gaussian Kernel
% An arbitrary scaling factor may be useful
jhKWidths = jhLambda .* mean(reviewedDeltaX,2) * jhArbitFactor;
jhParams(:,4) = jhKWidths;

% Create a jittered histogram image

settings_recon = [];
settings_recon.linMag = jhLinMag;
settings_recon.stackSize = stackSize;
settings_recon.cameraPixelSize_nm = cameraPixelSize_nm;

imageMatrix = zeros(frameSize * jhLinMag);     % Pre-allocate memory
for lpJts = 1:jhNJitters
    jtOffset = jhKWidths*ones(1,2);             % Envelope of offset
    jtOffset = jtOffset.*randn(size(jtOffset)); % In nm, randomised offset
    jtOffset = jtOffset./cameraPixelSize_nm;            % In CCD pixel Widths
    jtPosits = reviewedPosits + jtOffset;
    reviewed_SupResParams.x_coord = jtPosits(:,1);
    reviewed_SupResParams.y_coord = jtPosits(:,2);
    imageMatrix = imageMatrix + double(rainSTORM_recon_counts(reviewed_SupResParams, settings_recon));
end

nRows = size(imageMatrix, 1);
nCols = size(imageMatrix, 2);
pixelCenters.x = (repmat(transpose(1:nRows), [1,nCols])-0.5) / jhLinMag;
pixelCenters.y = (repmat(1:nCols, [nRows,1])-0.5) / jhLinMag;
pixelCenters.unit = 'camera pixel length';

end