classdef rainSTORM_clusterAnalysis
    %UNTITLED Summary of this class goes here
    %   Detailed explanation goes here
    
    methods (Static)
        
        
        function [pointillisticData, properties, metadata, sharedParameters] = clusterize(pointillisticData, method, settings)
            switch method
                case 'DBSCAN'
                    % performs the DBSCAN on a 2D pointillistic data

                    epsilon_nm=unitConversion.convertScalarQuantity(settings.epsilon, 'nm', settings.cameraSignalConversion);
                    epsilon_nm = epsilon_nm.value;
                    minPts = settings.minPts;
                    threeDimFlag = settings.threeDimFlag;
                    zScaleFactor = settings.zScaleFactor;
                    
                    if ~threeDimFlag
                        X(:,2)=pointillisticData.y_coord;
                        X(:,1)=pointillisticData.x_coord;
                    else
                        X(:,1)=pointillisticData.x_coord;
                        X(:,2)=pointillisticData.y_coord;
                        X(:,3)=pointillisticData.z_coord*zScaleFactor;  % scale for the "z" coordiante to taje into acount the worse (roughl 2X) localization precision
                    end

                    X_nm = X*settings.cameraSignalConversion.pixelSize_nm;
                    IDX=uint32(DBSCAN_boundaryIndexPrecalculation(X_nm, epsilon_nm, minPts));
                    
                    properties = rainSTORM_clusterAnalysis.clusterProperties(X, IDX);
                    
                otherwise
                    error('Invalid clusterization algorithm: %s.', method)
            end
            
            pointillisticData = pointillisticData_fieldManagement.add(pointillisticData, 'cluster_idx', IDX);
            
            metadata = struct();
            sharedParameters = struct();
        end
        
        
        function properties = clusterProperties(X, IDX)
            % calculate other properties of the clusters
            
            dimension=size(X, 2);
            
            %% center of clusters
            XC=zeros([max(IDX), size(X,2)]);
            for indcl=1:max(IDX)
                XC(indcl,:)=mean(X(IDX==indcl,:), 1);
            end 
            
            %% num of elements of each cl
            nOfEl=zeros([max(IDX), 1]);
            for indcl=1:max(IDX)
                nOfEl(indcl,:)=sum(IDX==indcl);
            end
        
            %% area/volume of clusters
            volumeOfCl=zeros([max(IDX), 1]);
            if dimension==2
                minPoints=3;
            elseif dimension==3
                minPoints=4;
            else
                error('Invalid cluster data')
            end
            for indcl=1:max(IDX)
                clusterPoints=double(X(IDX==indcl,:));
                if size(clusterPoints, 1)>=minPoints
                    [~,volumeOfCl(indcl,:)]=convhull(clusterPoints);
                else
                    % give zero volume of there are too few localization
                    % points in the cluster to form a volume
                    volumeOfCl(indcl,:)=0;
                end
            end
            
            %% distance of closest cluster
            minDistM=pdist2(XC,XC);
            minDistM(~minDistM)=nan;
            distOfCl=transpose(min(minDistM));
            
            properties.centers = XC;
            properties.elementNumber = nOfEl;
            properties.volumes = volumeOfCl;
            properties.nearestNeighbourDistance = distOfCl;
            
        end
        
        
        function clusterFig=plotClusters(pointillisticDataData, XC, settings)
            % create a figure containing the clusters
            
            minPts = settings.minPts;
            epsilon = settings.epsilon;
            threeDimFlag = settings.threeDimFlag;

            if ~threeDimFlag
                X(:,2)=pointillisticDataData.y_coord;
                X(:,1)=pointillisticDataData.x_coord;
            else
                X(:,1)=pointillisticDataData.x_coord;
                X(:,2)=pointillisticDataData.y_coord;
                X(:,3)=pointillisticDataData.z_coord;
            end
            
            if ~pointillisticData_fieldNameManagement.isMemberField(pointillisticDataData, 'cluster_idx')
                warning('Cannot show the the clusters, the clusterization data is missing. Perform the clusterization first!')
                clusterFig = matlab.ui.Figure.empty();
                return
            end
            IDX = pointillisticDataData.cluster_idx;
            
            figureTitle = ['DBSCAN Clustering (\epsilon = ' num2str(epsilon.value) ' ', epsilon.unit, ', minPts = ' num2str(minPts) ', NofCl = ' num2str(max(IDX)) ')'];
            clusterFig=figure('Name',figureTitle , 'visible', 'off');
            title(figureTitle);
            if ~threeDimFlag
                PlotClusterinResult(X, IDX);
                % plot centers
                hold on
                plot(XC(:,2), -XC(:,1),'kx','MarkerSize',8)
            else
                PlotClusterinResult3D(X, IDX);
                % plot centers
                hold on
                scatter3(XC(:,2), -XC(:,1), XC(:,3),'x','k')
            end
            axis equal
            grid off
        end
        
        
        function clusterPropFig=plotClusterProperties(properties, settings)
            % create a figure about the properties of the clusters
            
            pixelSize_nm = settings.cameraSignalConversion.pixelSize_nm;
            
            minPts = settings.minPts;
            epsilon = settings.epsilon;
            threeDimFlag = settings.threeDimFlag;
            
            XC = properties.centers;
            nOfEl = properties.elementNumber;
            if ~threeDimFlag
                volumeOfCl = properties.volumes * pixelSize_nm^2;
            else
                volumeOfCl = properties.volumes * pixelSize_nm^3;
            end
            distOfCl = properties.nearestNeighbourDistance * pixelSize_nm;
            
            nOfEl_limits = [0 200];
            volumeOfCl_limits = [0 40000];
            distOfCl_limits = [0 500];
            
            % get the histograms of cluster properties
            [nOfEl_hist, volumeOfCl_hist, distOfCl_hist] = getClusterHistograms(nOfEl, nOfEl_limits, volumeOfCl, volumeOfCl_limits, distOfCl, distOfCl_limits);
            
            figureTitle = ['DBSCAN Clustering (\epsilon = ' num2str(epsilon.value) ' ', epsilon.unit, ', minPts = ' num2str(minPts) ', NofCl = ' num2str(size(XC,1)) ')'];
            clusterPropFig=figure('Name', figureTitle , 'OuterPosition', [1,100,1620,840], 'Visible', 'off');
            title(figureTitle);
            
            %% plot num of el
            subplot(2,3,1);
            if ~threeDimFlag
                scatter(XC(:,2), -XC(:,1),125,nOfEl,'.' );
            else
                scatter3(XC(:,2), -XC(:,1), XC(:,3), 125,nOfEl,'.' );
            end
            colorbar;
            caxis([1 50])
            %ylabel(c,'Number of cluster elements')
            %% plot hist of nofel
            subplot(2,3,4);
            histogram('BinEdges', nOfEl_hist.BinEdges, 'BinCounts', nOfEl_hist.BinCounts);
            xlim([0 200])
            xlabel('Number of cluster elements')
            %% plot area/volume of cl
            subplot(2,3,2);
            if ~threeDimFlag
                scatter(XC(:,2), -XC(:,1),125,volumeOfCl,'.');
            else
                scatter3(XC(:,2), -XC(:,1), XC(:,3),125,volumeOfCl,'.');
            end
            colorbar;
            caxis([1 10000])
            
            % make a title here (upper center subplot) for all the subplots
            % not the most elegant solution, but there is no other solution
            % for Matlab versions older than 2018b....
            title(figureTitle, 'FontSize',16);
            %ylabel(c,'Area of cluster (nm^2)')
            %% plot hist of area
            subplot(2,3,5);
            histogram('BinEdges', volumeOfCl_hist.BinEdges, 'BinCounts', volumeOfCl_hist.BinCounts);
            xlim([0 40000])
            if ~threeDimFlag
                xlabel('Area of cluster (nm^2)')
            else
                xlabel('Volume of cluster (nm^3)')
            end
            %% plot dist
            subplot(2,3,3);
            if ~threeDimFlag
                scatter(XC(:,2), -XC(:,1),125,distOfCl,'.');
            else
                scatter3(XC(:,2), -XC(:,1), XC(:,3),125,distOfCl,'.');
            end
            caxis([1 200])
            colorbar;
            %ylabel(c,'Distance of the closest cluster (nm)')
            %% hist of dist
            subplot(2,3,6);
            histogram('BinEdges', distOfCl_hist.BinEdges, 'BinCounts', distOfCl_hist.BinCounts);
            xlim([0 500])
            xlabel('Distance of the closest cluster (nm)')
        end
        
        
        function xlsWrite(properties, settings, exportPath, exportName)
            
            nOfEl = properties.elementNumber;
            volumeOfCl = properties.volumes;
            distOfCl = properties.nearestNeighbourDistance;
            
            minPts = settings.minPts;
            epsilon = settings.epsilon.value;
            threeDimFlag = settings.threeDimFlag;
            ROI_nm = unitConversion.convert2DVectorQuantity(settings.ROI, 'nm', settings.cameraSignalConversion);
            
            nOfEl_limits = [0 200];
            volumeOfCl_limits = [0 40000];
            distOfCl_limits = [0 500];
            
            [nofel_hist, volume_hist, dist_hist] = getClusterHistograms(nOfEl, nOfEl_limits, volumeOfCl, volumeOfCl_limits, distOfCl, distOfCl_limits);
            
            % write out the cluster data
            if ~threeDimFlag
                xlsfilename = fullfile(exportPath, [exportName, '_cluster_datas']);
                header={'Examined area','','','','Number of cluster elements','pcs','Area of cluster [nm^2]','pcs','Distance of cluster','pcs'};
            else
                xlsfilename = fullfile(exportPath, [exportName, '_cluster_datas_3D']);
                header={'Examined area','','','','Number of cluster elements','pcs','Volume of cluster [nm^3]','pcs','Distance of cluster','pcs'};
            end
            
            clusterisation_params= [num2str(epsilon),'-',num2str(minPts),'-',num2str(numel(nOfEl))];
            sheet_name=num2str(clusterisation_params);
            examined_area_h=['x';'y'];
            area_nm2=abs(ROI_nm.x(1)-ROI_nm.x(2))*abs(ROI_nm.y(1)-ROI_nm.y(2));
            T=num2str(area_nm2);
            T_h={'Area [nm^2]',T};
            
            nofel_col1=nofel_hist.BinEdges';
            nofel_col2=nofel_hist.BinCounts';
            N={'Element numbers','pcs'};
            %NonZeroEl=find(nofel_col2);
            %Pcs=nofel_col2(NonZeroEl);
            %ElNum=nofel_col1(NonZeroEl);
            Pcs=nofel_col2;
            ElNum=nofel_col1;
            
            volume_col1=volume_hist.BinEdges';
            volume_col2=volume_hist.BinCounts';
            
            dist_col1=dist_hist.BinEdges';
            dist_col2=dist_hist.BinCounts';
            
            xlswrite(xlsfilename,header,sheet_name);
            
            xlswrite(xlsfilename,examined_area_h,sheet_name,'A2');
            xlswrite(xlsfilename,[ROI_nm.x(1), ROI_nm.x(2), ROI_nm.y(1), ROI_nm.y(2)] ,sheet_name,'B2');
            xlswrite(xlsfilename,T_h,sheet_name,'A4');
            xlswrite(xlsfilename,N,sheet_name,'A6');
            xlswrite(xlsfilename,ElNum,sheet_name,'A7');
            xlswrite(xlsfilename,Pcs,sheet_name,'B7');
            
            xlswrite(xlsfilename,nofel_col1,sheet_name,'E2');
            xlswrite(xlsfilename,nofel_col2,sheet_name,'F2');
            
            xlswrite(xlsfilename,volume_col1,sheet_name,'G2');
            xlswrite(xlsfilename,volume_col2,sheet_name,'H2');
            
            xlswrite(xlsfilename,dist_col1,sheet_name,'I2');
            xlswrite(xlsfilename,dist_col2,sheet_name,'J2');
        end
        
    end
end


function [nOfEl_hist, volumeOfCl_hist, distOfCl_hist] = getClusterHistograms(nOfEl, nOfEl_limits, volumeOfCl, volumeOfCl_limits, distOfCl, distOfCl_limits)

    [nOfEl_hist.BinCounts, nOfEl_hist.BinEdges]=histcounts(nOfEl, 'BinLimits', nOfEl_limits);
    [volumeOfCl_hist.BinCounts, volumeOfCl_hist.BinEdges]=histcounts(volumeOfCl, 'BinLimits', volumeOfCl_limits);
    [distOfCl_hist.BinCounts, distOfCl_hist.BinEdges]=histcounts(distOfCl, 'BinLimits', distOfCl_limits);
    
end


function PlotClusterinResult(X, IDX)

%
% Copyright (c) 2015, Yarpiz (www.yarpiz.com)
% All rights reserved. Please read the "license.txt" for license terms.
%
% Project Code: YPML110
% Project Title: Implementation of DBSCAN Clustering in MATLAB
% Publisher: Yarpiz (www.yarpiz.com)
% 
% Developer: S. Mostapha Kalami Heris (Member of Yarpiz Team)
% 
% Contact Info: sm.kalami@gmail.com, info@yarpiz.com
%

    k=double(max(IDX));

    Colors=hsv(k);

    Legends = {};
    for i=0:k
        Xi=X(IDX==i,:);
        if i~=0
            Style = '.';
            MarkerSize = 5;
            Color = Colors(i,:);
            Legends{end+1} = ['Cluster #' num2str(i)];
        else
            Style = '.';
            MarkerSize = 5;
            Color = [0 0 0];
            if ~isempty(Xi)
                Legends{end+1} = 'Noise';
            end
        end
        if ~isempty(Xi)
            plot(Xi(:,2),-Xi(:,1),Style,'MarkerSize',MarkerSize,'Color',Color);
        end
        hold on;
    end
    hold off;
    axis equal;
    grid on;
    %legend(Legends);
    %legend('Location', 'NorthEastOutside');

end


function PlotClusterinResult3D(X, IDX)

%
% Copyright (c) 2015, Yarpiz (www.yarpiz.com)
% All rights reserved. Please read the "license.txt" for license terms.
%
% Project Code: YPML110
% Project Title: Implementation of DBSCAN Clustering in MATLAB
% Publisher: Yarpiz (www.yarpiz.com)
% 
% Developer: S. Mostapha Kalami Heris (Member of Yarpiz Team)
% 
% Contact Info: sm.kalami@gmail.com, info@yarpiz.com
%

    k=max(IDX);

    Colors=hsv(k);

    Legends = {};
    for i=0:k
        Xi=X(IDX==i,:);
        if i~=0
            Style = '.';
            MarkerSize = 7;
            Color = Colors(i,:);
            Legends{end+1} = ['Cluster #' num2str(i)];
        else
            Style = '.';
            MarkerSize = 1;
            Color = [0 0 0];
            if ~isempty(Xi)
                Legends{end+1} = 'Noise';
            end
        end
        if ~isempty(Xi)
            scatter3(Xi(:,2),-Xi(:,1),Xi(:,3),MarkerSize,Color,'filled');
        end
        hold on;
    end
    hold off;
    axis equal;
    grid on;         
        
    %legend(Legends);
    %legend('Location', 'NorthEastOutside');

end
