function figHistsHandle = rainSTORM_histograms(localizationData_accepted, localizationData_rejected, titleText, numberOfFrames, exposureTime_ms, countsPerElectron, pixelSize_nm)
% Plot Quality Control Data for Localisations

text_figure_name = '';
if iscell(titleText)
    text_figure_name = ['Histograms of Dataset: ',titleText{1},...
    ', ',titleText{2}];

elseif ischar 
    text_figure_name = titleText;
else
    text_figure_name = 'Histograms of localizations';
    warning('Invalid histogram title was given for rainSTORm histogram visualization.')
end

figSize = [1,100,1120,840];
figHistsHandle = figure(...
    'Name',text_figure_name,...
    'NumberTitle','off',...
    'OuterPosition', figSize, ...
    'Visible', 'Off'...
    );


%% First Subplot:
% subplot(2,2,1) % Use this if screen shape spoils the method below
subplot('Position', [0.07,0.56,0.39,0.39])

% Histogram of counts associated with localisations, in 3x3 pixel centres
% Determine bar chart data:

sumSignal_accepted = double(localizationData_accepted.sum_signal);
sumSignal_rejected = double(localizationData_rejected.sum_signal);

% sumSignal_max = max(max([sumSignal_accepted; sumSignal_rejected]));
% sumSignal_min = min(min([sumSignal_accepted; sumSignal_rejected]));
% 
% if isempty(sumSignal_bounds)
%     sumSignal_bounds=[sumSignal_min, sumSignal_max];
% else
%    if sumSignal_bounds(1)==-Inf
%        sumSignal_bounds(1)=sumSignal_min;
%    end
%    if sumSignal_bounds(2)==Inf
%        sumSignal_bounds(2)=sumSignal_max;
%    end
% end

binEdges = linspace(0, 10000, 50);
binCenters = (binEdges(1:end-1) + binEdges(2:end))/2;
bar_factor = 0.6;


[myNGood,bin_centers] = hist(sumSignal_accepted / countsPerElectron, binCenters); % Get Hist data
myNBad = hist(sumSignal_rejected / countsPerElectron, binCenters); % Get Hist data

hold on
h_bar = bar(bin_centers,[myNGood(:) myNBad(:)],'r','stacked', 'BarWidth', bar_factor);
set(h_bar(2),'FaceColor',[0.4 0.4 0.4]);
hold off
set(gca, 'FontSize', 12, 'FontWeight', 'bold', 'FontName', 'Times');
title('Sum Signal');
xlabel('Photoelectrons')
ylabel('Number of Candidates');



%% Second subplot:
% subplot(2,2,2)
subplot('Position', [0.57,0.56,0.39,0.39])
smooth_width = double(min(ceil(numberOfFrames/100),100));

frameIdx_accepted = localizationData_accepted.frame_idx;
frameIdx_rejected = localizationData_rejected.frame_idx;

numberAcceptedPerFrame=zeros(numberOfFrames, 1);
for lpLoc = 1:numel(frameIdx_accepted)
  numberAcceptedPerFrame(frameIdx_accepted(lpLoc)) = ...
      numberAcceptedPerFrame(frameIdx_accepted(lpLoc))+1;
end

numberRejectedPerFrame=zeros(numberOfFrames, 1);
for lpLoc = 1:numel(frameIdx_rejected)
  numberRejectedPerFrame(frameIdx_rejected(lpLoc)) = ...
      numberRejectedPerFrame(frameIdx_rejected(lpLoc))+1;
end

to_plotRejected=zeros(1,numberOfFrames);
to_plotAccepted=zeros(1,numberOfFrames);
to_plotRejected(1:numel(numberRejectedPerFrame)) = numberRejectedPerFrame;
to_plotAccepted(1:numel(numberAcceptedPerFrame)) = numberAcceptedPerFrame;
to_plotRejectedSM = conv(to_plotRejected,ones(1,smooth_width),'same')./smooth_width;
to_plotAcceptedSM = conv(to_plotAccepted,ones(1,smooth_width),'same')./smooth_width;

time_s = ((1:numberOfFrames)-1)*exposureTime_ms/1000;
plot(time_s, to_plotRejectedSM, 'r')
hold on
plot(time_s, to_plotAcceptedSM,'b')
set(gca, 'FontSize', 12, 'FontWeight', 'bold', 'FontName', 'Times');
title('Localisations per Frame (smoothed)');
xlabel('Time [s]');
ylabel('Number per Frame');
legend('Rejected','Accepted','Location','Best');
hold off


%% Third Plot
% Localisation fitted widths (fitted sigma values)
% New 22/3/2013 - CHECK for some bars "hiding" others, or other bugs
% subplot(2,2,3)
subplot('Position', [0.07,0.07,0.39,0.39])

if pointillisticData_fieldNameManagement.isMemberField(localizationData_accepted, {'sig_x', 'sig_y'})
    sigX_accepted = double(localizationData_accepted.sig_x) * pixelSize_nm;
    sigY_accepted = double(localizationData_accepted.sig_y) * pixelSize_nm;
    sigX_rejected = double(localizationData_rejected.sig_x) * pixelSize_nm;
    sigY_rejected = double(localizationData_rejected.sig_y * pixelSize_nm);
elseif pointillisticData_fieldNameManagement.isMemberField(localizationData_accepted, {'sig'})
    sigX_accepted = double(localizationData_accepted.sig) * pixelSize_nm;
    sigY_accepted = double(localizationData_accepted.sig) * pixelSize_nm;
    sigX_rejected = double(localizationData_rejected.sig) * pixelSize_nm;
    sigY_rejected = double(localizationData_rejected.sig) * pixelSize_nm;
else
    error('The pointillistic data supplied to histogram creation does not contain "sig" or "sig_x/sig_y" fields.')
end

% TODO: the histogram bounds are bound to camera pixel size, consider connecting it to
% a physical quantity of just fix it
step = 0.1 * pixelSize_nm;
lowerBound = 0.5 * pixelSize_nm;
upperBound = 4.0 * pixelSize_nm;

bins = (lowerBound:step:upperBound);
[count_row_good,bin_centers] = hist(sigX_accepted,bins); % Get Hist data
count_row_bad = hist(sigX_rejected,bins); % Get Hist data
count_col_good = hist(sigY_accepted,bins); % Get Hist data
count_col_bad = hist(sigY_rejected,bins); % Get Hist data

bar_factor = 0.4;
hold on
h_row = bar(bin_centers-bar_factor*step/2,[count_row_good(:) count_row_bad(:)],'r','stacked', 'BarWidth', bar_factor);
h_col = bar(bin_centers+bar_factor*step/2,[count_col_good(:) count_col_bad(:)],'b','stacked', 'BarWidth', bar_factor);
set(h_row(2),'FaceColor',[0.4 0.4 0.4]);
set(h_col(2),'FaceColor',[0.4 0.4 0.4]);
hold off
set(gca, 'FontSize', 12, 'FontWeight', 'bold', 'FontName', 'Times');
xlim([lowerBound-bar_factor*step, upperBound+bar_factor*step])
title('Fitted Gaussian RMS widths');
ylabel('Number of Candidates');
xlabel('Width [nm]');
if ~sum(count_row_good)==0 && ~sum(count_col_good)==0     % If there are red and blue bars
  legend([h_row(1) h_col(1)],'Row-direction','Col-direction'); % Then give a legend for them
end


%% Fourth Figure
% Thompson localisation precisions
% subplot(2,2,4)
subplot('Position', [0.57,0.07,0.39,0.39])

if pointillisticData_fieldNameManagement.isMemberField(localizationData_accepted, {'std_x', 'std_y'})
    stdX_accepted = double(localizationData_accepted.std_x) * pixelSize_nm;
    stdY_accepted = double(localizationData_accepted.std_y) * pixelSize_nm;
    stdX_rejected = double(localizationData_rejected.std_x) * pixelSize_nm;
    stdY_rejected = double(localizationData_rejected.std_y) * pixelSize_nm;
elseif pointillisticData_fieldNameManagement.isMemberField(localizationData_accepted, {'std'})
    stdX_accepted = double(localizationData_accepted.std) * pixelSize_nm;
    stdY_accepted = double(localizationData_accepted.std) * pixelSize_nm;
    stdX_rejected = double(localizationData_rejected.std) * pixelSize_nm;
    stdY_rejected = double(localizationData_rejected.std) * pixelSize_nm;
else
    error('The pointillistic data supplied to histogram creation does not contain "std" or "std_x/std_y" fields.')
end

bins = 0:5:150;

[count_row_good,bin_centers] = hist(stdX_accepted,bins); % Inter-Row direction
count_row_bad = hist(stdX_rejected,bins); % Inter-Row direction
count_col_good = hist(stdY_accepted,bins); % Inter-Column direction
count_col_bad = hist(stdY_rejected,bins); % Inter-Column direction

hold on
h_row = bar(bin_centers-1,[count_row_good(:) count_row_bad(:)],'r','stacked', 'BarWidth', 0.4);
h_col = bar(bin_centers+1,[count_col_good(:) count_col_bad(:)],'b','stacked', 'BarWidth', 0.4);
set(h_row(2),'FaceColor',[0.4 0.4 0.4]);
set(h_col(2),'FaceColor',[0.4 0.4 0.4]);
hold off
set(gca, 'FontSize', 12, 'FontWeight', 'bold', 'FontName', 'Times');
xlim([0 150])    % Set x-range to the interesting 0-150 nm region
title('Thompson Localisation Precisions');
ylabel('Number of Candidates');
xlabel('Precision [nm]')
if ~sum(count_row_good)==0 && ~sum(count_col_good)==0     % If there are red and blue bars
  legend([h_row(1) h_col(1)],'Row-direction','Col-direction'); % Then give a legend for them
end

end