function [pointillisticData, analysisParameters] = rainSTORM_legacy_dataImport(dataFileFullName, definedCameraSignalConversion)

% Beware!!! it can only import the 'Recon 1' info, it is not handled correctly even in the original rainSTORM...
infoFileName = strrep(strrep(dataFileFullName, 'reviewed', ''), 'SupResParams.full.csv', 'Recon 1-info.txt');

infoText = pointillisticData_load.infoText(infoFileName);
[dataHeaders, dataMatrix] = pointillisticData_load.headerAndMatrixData(dataFileFullName);

% the frame number information is not saved, so guess it.... 
frameNumber = max(dataMatrix(:, strcmp(dataHeaders, 'frame_idx')));

analysisParameters = processInfoText(infoText, frameNumber);

[algorithm, analyses] = inferDataTypes(dataMatrix, dataHeaders, analysisParameters);
convention = 'rainSTORM legacy';

analysisParameters.sharedParameters.convention.name = convention;
analysisParameters.sharedParameters.convention.algorithm = algorithm;
analysisParameters.sharedParameters.convention.analyses = analyses;

% overwrite the parametrs that could not be read out from the
% metadata with the default ones:
loadedCameraSignalConversion = analysisParameters.sharedParameters.cameraSignalConversion;
missingOnlyBoolean = true;
overwrittenCameraSignalConversion = imageStack_getConversionFactors.overwriteParameters(loadedCameraSignalConversion, definedCameraSignalConversion, missingOnlyBoolean);
analysisParameters.sharedParameters.cameraSignalConversion = overwrittenCameraSignalConversion;

[dataPath, dataName, ~] = fileparts(dataFileFullName);
analysisParameters.sharedParameters.originalDataPath = dataPath;
analysisParameters.sharedParameters.originalDataName = strrep(dataName, '.full', '');

pointillisticData = convertImportedData(dataMatrix, dataHeaders, convention, algorithm, analyses, overwrittenCameraSignalConversion);


end

function [algorithm, analyses] = inferDataTypes(dataMatrix, dataHeaders, analysisParameters)

algorithm = analysisParameters.creationOperation{1}.settings.algorithmName;

analyses = {};

if sum(strcmp(dataHeaders, 'sig_x')) && sum(strcmp(dataHeaders, 'sig_y'))
    
    sig_x = dataMatrix(:, strcmp(dataHeaders, 'sig_x'));
    sig_y = dataMatrix(:, strcmp(dataHeaders, 'sig_y'));


    if sum(sig_x==sig_y) == numel(sig_x)
        analyses{end+1} = 'precision';
    else
        analyses{end+1} = 'astigmatic precision';
    end

end

if sum(strcmp(dataHeaders, 'ellipticity'))
    analyses{end+1} = 'ellipticity';
end



if sum(strcmp(dataHeaders, 'z_coord'))
    z_coord = dataMatrix(:, strcmp(dataHeaders, 'z_coord'));
    if sum(z_coord~=0)>0
        analyses{end+1} = 'astigmatic 3D';
    end
end


% TODO: trajectories, polarization

end

function analysisParameters = processInfoText(infotex, frameNumber, measured_quantity)

imageFileFullName = getValueText(infotex, 'Raw Image Data:');
imageStackReading_settings.dataFullFileName = imageFileFullName;
imageStackReading_settings.imageType='';
imageStackReading_settings.cameraPropertiesFile='';
imageStackReading_settings.additionalMagnification=NaN;


[imageFileNamePath, imageFileName] = fileparts(imageFileFullName);
sharedParameters.originalDataPath = imageFileNamePath;
sharedParameters.originalDataName = imageFileName;

sizeOfCCDFrame = str2num(getValueText(infotex, 'sizeOfCCDFrame:'));

stackSize = [sizeOfCCDFrame, frameNumber];
imageStackReading_metadata.stackSize = stackSize;
sharedParameters.stackSize = stackSize;

cameraSignalConversion.pixelSize_nm = str2double(getValueText(infotex, 'CCD pixel width on sample (nm):'));
if ~isempty(getValueText(infotex, '# ND2 data from BioFormats:'))
    cameraSignalConversion.exposureTime_ms = str2num(getValueText(infotex, 'Exposure:'));
    %cameraSignalConversion.exposureTime_ms = str2num(getValueText(infotex, 'dExposureTime:'));
else
    cameraSignalConversion.exposureTime_ms = NaN;
end
cameraSignalConversion.frameInterval_ms = NaN;
cameraSignalConversion.countsPerPhoton = str2double(getValueText(infotex, 'Counts per photon for calibration:'));
cameraSignalConversion.countsPerElectron = NaN;
cameraSignalConversion.baseline_count = NaN;
imageStackReading_metadata = cameraSignalConversion;
sharedParameters.cameraSignalConversion = cameraSignalConversion;

% localization settings
algoName = getValueText(infotex, 'Localization Algorithm:');
locSettings.ImageFilter = getValueText(infotex, 'Image filter algorithm:');
locSettings.Thresh = str2double(getValueText(infotex, 'ROI Candidate Threshold:'));
locSettings.Thresh2 = str2double(getValueText(infotex, 'ROI Candidate Relative Peak Value Threshold:'));
locSettings.allowSig = str2num(getValueText(infotex, 'Allowed Sigma [pix]:'));
locSettings.allowX = str2double(getValueText(infotex, 'Allowed  Distance [pix]:'));
locSettings.initSig = str2double(getValueText(infotex, 'Initial guess of PSF Sigma [pix]:'));
locSettings.maxIts = str2double(getValueText(infotex, 'Maximum Iterations:'));
locSettings.prevSF = str2double(getValueText(infotex, 'Preview Scale Factor [int]:'));
locSettings.rad = str2double(getValueText(infotex, 'Radius of ROI [pix]:'));
locSettings.tol = str2double(getValueText(infotex, 'Tolerance [-]:'));
localizationSettings.algorithmName = algoName;
localizationSettings.algorithmSettings = locSettings;


locMetadata.ROI.x = [0, sizeOfCCDFrame(1)];
locMetadata.ROI.y = [0, sizeOfCCDFrame(2)];
locMetadata.ROI.unit = 'camera pixel length';
locMetadata.frameRange = [1, frameNumber];
sharedParameters.ROI = locMetadata.ROI;
sharedParameters.frameRange = locMetadata.frameRange;
 
reconstruction_algorithm = getValueText(infotex, 'Reconstruction Algorithm:');
reconstruction_scale_factor = str2double(getValueText(infotex, 'Reconstruction Scale Factor:'));
contrast_windowing = str2num(getValueText(infotex, 'Contrast Windowing:'));

superResolvedImageSettings.imageType = reconstruction_algorithm;
superResolvedImageSettings.pixelization.pixelSize.x=cameraSignalConversion.pixelSize_nm/reconstruction_scale_factor;
superResolvedImageSettings.pixelization.pixelSize.y=cameraSignalConversion.pixelSize_nm/reconstruction_scale_factor;
superResolvedImageSettings.pixelization.pixelSize.unit='nm';
superResolvedImageSettings.visualization.colormapRange=contrast_windowing;


totalDrift_array = str2num(getValueText(infotex, 'Total Drift:'));
if ~isempty(totalDrift_array)
    totalDrift_struct.x = totalDrift_array(1);
    totalDrift_struct.y = totalDrift_array(2);
else
    totalDrift_struct.x = 0;
    totalDrift_struct.y = 0;
end
totalDrift_struct.unit = 'camera pixel length';
driftCorrection_metadata.totalDrift = totalDrift_struct;

% filter data
thresholds = struct();
thresholds.frame_idx = str2num(getValueText(infotex, 'Frame Index:'));
thresholds.x_coord = str2num(getValueText(infotex, 'X coordinate:'));
thresholds.y_coord = str2num(getValueText(infotex, 'Y coordinate:'));
thresholds.z_coord = str2num(getValueText(infotex, 'Z coordinate:'));
thresholds.x_std = str2num(getValueText(infotex, 'Thompson Std X:'));
thresholds.y_std = str2num(getValueText(infotex, 'Thompson Std Y:'));
thresholds.I = str2num(getValueText(infotex, 'I parameter:'));
thresholds.sig_x = str2num(getValueText(infotex, 'Sigma X:'));
thresholds.sig_y = str2num(getValueText(infotex, 'Sigma Y:'));
thresholds.ellip_xy = str2num(getValueText(infotex, 'Ellipticity Sig_XY:'));
thresholds.avg_brightness = str2num(getValueText(infotex, 'Average Brightness:'));
thresholds.res = str2num(getValueText(infotex, 'Residue:'));
thresholds.res_Row = str2num(getValueText(infotex, 'Residue Row:'));
thresholds.res_Col = str2num(getValueText(infotex, 'Residue Col:'));
thresholds.roi_min = str2num(getValueText(infotex, 'ROI Min:'));
thresholds.Sum_signal = str2num(getValueText(infotex, 'Sum Signal:'));
localizationFiltering.settings.thresholds = thresholds;


cameraName = split(getValueText(infotex, 'Camera Name:'));
if ~isempty(getValueText(infotex, 'Camera Name:'))
    cameraSettings.manufacturer = cameraName{1};
    cameraSettings.model = cameraName{2};
    cameraSettings.serialNumber = cameraName{3};
    cameraSettings.settings.conversionGain = str2num(getValueText(infotex, 'Conversion Gain #1:'));
    cameraSettings.settings.amplificationGain = str2num(getValueText(infotex, 'GainMultiplier:'));
    cameraSettings.settings.readOutRate = str2num(getValueText(infotex, 'Readout Speed:'));
else
    cameraSettings.manufacturer = '';
    cameraSettings.model = '';
    cameraSettings.serialNumber = '';
    cameraSettings.settings.conversionGain = [];
    cameraSettings.settings.amplificationGain = [];
    cameraSettings.settings.readOutRate = [];
end
imageStackReading_metadata.cameraSettings = cameraSettings;


analysisParameters_imageStack.dataType = 'imageStack';
analysisParameters_imageStack.note = 'legacy rainSTORM localization data import';
analysisParameters_imageStack.creationOperation = {...
    struct(...
    'operationType','image stack loading',...
    'note','legacy rainSTORM localization data import',...
    'function','',...
    'settings',imageStackReading_settings,...
    'metadata',imageStackReading_metadata...
    )...
};
analysisParameters_imageStack.processingOperations = [];
analysisParameters_imageStack.derivativeDatas = {};

% the localization data is the upper in the hierarchy within the
% "analysisParameters" in case of the legacy rainSTORM
analysisParameters.dataType = 'localizationData';
analysisParameters.note = 'legacy rainSTORM localization data import';
analysisParameters.creationOperation = {...
    struct(...
    'operationType','localization',...
    'note','legacy rainSTORM localization data import',...
    'function',algoName,...
    'settings',localizationSettings,...
    'metadata',locMetadata...
    )...
    };
% put the operaation into a cell array instead of a normal as, as the html
% export does not work with normal struct arrays....
analysisParameters.processingOperations = {...
    struct(...
    'operationType','filtering',...
    'note','legacy rainSTORM localization data import',...
    'function','',...
    'settings',localizationFiltering,...
    'metadata',struct([])...% this info is not saved in the rainSTORM info file
    ),...
    struct(...
    'operationType','drift correction',...
    'note','legacy rainSTORM localization data import',...
    'function','',...
    'settings',struct([]),...% this info is not saved in the rainSTORM info file
    'metadata',driftCorrection_metadata...
    ),...
    struct(...
    'operationType','visualization',...
    'note','legacy rainSTORM localization data import',...
    'function','',...
    'settings',superResolvedImageSettings,...
    'metadata',struct([])...
    ),...
};
analysisParameters.derivativeDatas = {analysisParameters_imageStack};
analysisParameters.sharedParameters = sharedParameters;

end

function valueText = getValueText(infoText, propertyText)

propertyText = strrep(propertyText, '(', '\(');
propertyText = strrep(propertyText, ')', '\)');
propertyText = strrep(propertyText, '[', '\[');
propertyText = strrep(propertyText, ']', '\]');

%valueText_cells = regexp(infoText, ['(?<=', propertyText, '\s).+?(?=\r?\n)'],'match');
% safer to match starting with a newline character:
valueText_cells = regexp(infoText, ['(?<=\r?\n', propertyText, '\s).+?(?=\r?\n)'],'match');

valueText = '';
if isempty(valueText_cells)
    warning(['The poperty \"', propertyText, '\" to be read from rainSORM info file is empty.'])
else
    valueText = valueText_cells{1};
end

valueText = strtrim(valueText);

end

